# Fhircraft modules
from enum import Enum

# Standard modules
from typing import Literal, Optional, Union

# Pydantic modules
from pydantic import BaseModel, Field, field_validator, model_validator
from pydantic.fields import FieldInfo

import fhircraft
import fhircraft.fhir.resources.validators as fhir_validators
from fhircraft.fhir.resources.base import FHIRBaseModel
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.utils import model_rebuild_all

NoneType = type(None)

# Dynamic modules

from typing import List, Literal, Optional

from fhircraft.fhir.resources.base import FHIRBaseModel
from fhircraft.fhir.resources.datatypes.primitives import (
    Code,
    Decimal,
    Instant,
    String,
    UnsignedInt,
    Uri,
)
from fhircraft.fhir.resources.datatypes.R4.complex import (
    BackboneElement,
    Element,
    Identifier,
    Meta,
    Resource,
    Signature,
)


class BundleLink(BackboneElement):
    """
    A series of links that provide context to this bundle.
    """

    relation: Optional[String] = Field(
        description="See http://www.iana.org/assignments/link-relations/link-relations.xhtml#link-relations-1",
        default=None,
    )
    relation_ext: Optional[Element] = Field(
        description="Placeholder element for relation extensions",
        default=None,
        alias="_relation",
    )
    url: Optional[Uri] = Field(
        description="Reference details for the link",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "relation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BundleEntryLink(BackboneElement):
    """
    A series of links that provide context to this entry.
    """

    relation: Optional[String] = Field(
        description="See http://www.iana.org/assignments/link-relations/link-relations.xhtml#link-relations-1",
        default=None,
    )
    relation_ext: Optional[Element] = Field(
        description="Placeholder element for relation extensions",
        default=None,
        alias="_relation",
    )
    url: Optional[Uri] = Field(
        description="Reference details for the link",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "relation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BundleEntrySearch(BackboneElement):
    """
    Information about the search process that lead to the creation of this entry.
    """

    mode: Optional[Code] = Field(
        description="match | include | outcome - why this is in the result set",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    score: Optional[Decimal] = Field(
        description="Search ranking (between 0 and 1)",
        default=None,
    )
    score_ext: Optional[Element] = Field(
        description="Placeholder element for score extensions",
        default=None,
        alias="_score",
    )

    @field_validator(
        *(
            "score",
            "mode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BundleEntryRequest(BackboneElement):
    """
    Additional information about how this entry should be processed as part of a transaction or batch.  For history, it shows how the entry was processed to create the version contained in the entry.
    """

    method: Optional[Code] = Field(
        description="GET | HEAD | POST | PUT | DELETE | PATCH",
        default=None,
    )
    method_ext: Optional[Element] = Field(
        description="Placeholder element for method extensions",
        default=None,
        alias="_method",
    )
    url: Optional[Uri] = Field(
        description="URL for HTTP equivalent of this entry",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    ifNoneMatch: Optional[String] = Field(
        description="For managing cache currency",
        default=None,
    )
    ifNoneMatch_ext: Optional[Element] = Field(
        description="Placeholder element for ifNoneMatch extensions",
        default=None,
        alias="_ifNoneMatch",
    )
    ifModifiedSince: Optional[Instant] = Field(
        description="For managing cache currency",
        default=None,
    )
    ifModifiedSince_ext: Optional[Element] = Field(
        description="Placeholder element for ifModifiedSince extensions",
        default=None,
        alias="_ifModifiedSince",
    )
    ifMatch: Optional[String] = Field(
        description="For managing update contention",
        default=None,
    )
    ifMatch_ext: Optional[Element] = Field(
        description="Placeholder element for ifMatch extensions",
        default=None,
        alias="_ifMatch",
    )
    ifNoneExist: Optional[String] = Field(
        description="For conditional creates",
        default=None,
    )
    ifNoneExist_ext: Optional[Element] = Field(
        description="Placeholder element for ifNoneExist extensions",
        default=None,
        alias="_ifNoneExist",
    )

    @field_validator(
        *(
            "ifNoneExist",
            "ifMatch",
            "ifModifiedSince",
            "ifNoneMatch",
            "url",
            "method",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BundleEntryResponse(BackboneElement):
    """
    Indicates the results of processing the corresponding 'request' entry in the batch or transaction being responded to or what the results of an operation where when returning history.
    """

    status: Optional[String] = Field(
        description="Status response code (text optional)",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    location: Optional[Uri] = Field(
        description="The location (if the operation returns a location)",
        default=None,
    )
    location_ext: Optional[Element] = Field(
        description="Placeholder element for location extensions",
        default=None,
        alias="_location",
    )
    etag: Optional[String] = Field(
        description="The Etag for the resource (if relevant)",
        default=None,
    )
    etag_ext: Optional[Element] = Field(
        description="Placeholder element for etag extensions",
        default=None,
        alias="_etag",
    )
    lastModified: Optional[Instant] = Field(
        description="Server\u0027s date time modified",
        default=None,
    )
    lastModified_ext: Optional[Element] = Field(
        description="Placeholder element for lastModified extensions",
        default=None,
        alias="_lastModified",
    )
    outcome: Optional[Resource] = Field(
        description="OperationOutcome with hints and warnings (for batch/transaction)",
        default=None,
    )

    @field_validator(
        *(
            "lastModified",
            "etag",
            "location",
            "status",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class BundleEntry(BackboneElement):
    """
    An entry in a bundle resource - will either contain a resource or information about a resource (transactions and history only).
    """

    link: Optional[List[BundleEntryLink]] = Field(
        description="Links related to this entry",
        default=None,
    )
    fullUrl: Optional[Uri] = Field(
        description="URI for resource (Absolute URL server address or URI for UUID/OID)",
        default=None,
    )
    fullUrl_ext: Optional[Element] = Field(
        description="Placeholder element for fullUrl extensions",
        default=None,
        alias="_fullUrl",
    )
    resource: Optional[Resource] = Field(
        description="A resource in the bundle",
        default=None,
    )
    search: Optional[BundleEntrySearch] = Field(
        description="Search related information",
        default=None,
    )
    request: Optional[BundleEntryRequest] = Field(
        description="Additional execution information (transaction/batch/history)",
        default=None,
    )
    response: Optional[BundleEntryResponse] = Field(
        description="Results of execution (transaction/batch/history)",
        default=None,
    )

    @field_validator(
        *(
            "response",
            "request",
            "search",
            "fullUrl",
            "link",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Bundle(Resource):
    """
    A container for a collection of resources.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Bundle"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    identifier: Optional[Identifier] = Field(
        description="Persistent identifier for the bundle",
        default=None,
    )
    type: Optional[Code] = Field(
        description="document | message | transaction | transaction-response | batch | batch-response | history | searchset | collection",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    timestamp: Optional[Instant] = Field(
        description="When the bundle was assembled",
        default=None,
    )
    timestamp_ext: Optional[Element] = Field(
        description="Placeholder element for timestamp extensions",
        default=None,
        alias="_timestamp",
    )
    total: Optional[UnsignedInt] = Field(
        description="If search, the total number of matches",
        default=None,
    )
    total_ext: Optional[Element] = Field(
        description="Placeholder element for total extensions",
        default=None,
        alias="_total",
    )
    link: Optional[List[BundleLink]] = Field(
        description="Links related to this Bundle",
        default=None,
    )
    entry: Optional[List[BundleEntry]] = Field(
        description="Entry in the bundle - will have a resource or information",
        default=None,
    )
    signature: Optional[Signature] = Field(
        description="Digital Signature",
        default=None,
    )
    resourceType: Literal["Bundle"] = Field(
        description=None,
        default="Bundle",
    )

    @field_validator(
        *(
            "signature",
            "entry",
            "link",
            "total",
            "timestamp",
            "type",
            "identifier",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("entry",), mode="after", check_fields=None)
    @classmethod
    def FHIR_bdl_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="resource.exists() or request.exists() or response.exists()",
            human="must be a resource unless there's a request or response",
            key="bdl-5",
            severity="error",
        )

    @field_validator(*("entry",), mode="after", check_fields=None)
    @classmethod
    def FHIR_bdl_8_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="fullUrl.contains('/_history/').not()",
            human="fullUrl cannot be a version specific reference",
            key="bdl-8",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="total.empty() or (type = 'searchset') or (type = 'history')",
            human="total only when a search or history",
            key="bdl-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="entry.search.empty() or (type = 'searchset')",
            human="entry.search only when a search",
            key="bdl-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="entry.all(request.exists() = (%resource.type = 'batch' or %resource.type = 'transaction' or %resource.type = 'history'))",
            human="entry.request mandatory for batch/transaction/history, otherwise prohibited",
            key="bdl-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="entry.all(response.exists() = (%resource.type = 'batch-response' or %resource.type = 'transaction-response' or %resource.type = 'history'))",
            human="entry.response mandatory for batch-response/transaction-response/history, otherwise prohibited",
            key="bdl-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_7_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(type = 'history') or entry.where(fullUrl.exists()).select(fullUrl&resource.meta.versionId).isDistinct()",
            human="FullUrl must be unique in a bundle, or else entries with the same fullUrl must have different meta.versionId (except in history bundles)",
            key="bdl-7",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_9_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="type = 'document' implies (identifier.system.exists() and identifier.value.exists())",
            human="A document must have an identifier with a system and a value",
            key="bdl-9",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_10_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="type = 'document' implies (timestamp.hasValue())",
            human="A document must have a date",
            key="bdl-10",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_11_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="type = 'document' implies entry.first().resource.is(Composition)",
            human="A document must have a Composition as the first resource",
            key="bdl-11",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_bdl_12_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="type = 'message' implies entry.first().resource.is(MessageHeader)",
            human="A message must have a MessageHeader as the first resource",
            key="bdl-12",
            severity="error",
        )
