# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Instant,
    Boolean,
    Base64Binary,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Coding,
    Period,
    CodeableConcept,
    BackboneElement,
    Reference,
    DomainResource,
)


class AuditEventAgentNetwork(BackboneElement):
    """
    Logical network location for application activity, if the activity has a network location.
    """

    address: Optional[String] = Field(
        description="Identifier for the network access point of the user device",
        default=None,
    )
    address_ext: Optional[Element] = Field(
        description="Placeholder element for address extensions",
        default=None,
        alias="_address",
    )
    type: Optional[Code] = Field(
        description="The type of network access point",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )

    @field_validator(
        *(
            "type",
            "address",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AuditEventAgent(BackboneElement):
    """
    An actor taking an active role in the event or activity that is logged.
    """

    type: Optional[CodeableConcept] = Field(
        description="How agent participated",
        default=None,
    )
    role: Optional[List[CodeableConcept]] = Field(
        description="Agent role in the event",
        default=None,
    )
    who: Optional[Reference] = Field(
        description="Identifier of who",
        default=None,
    )
    altId: Optional[String] = Field(
        description="Alternative User identity",
        default=None,
    )
    altId_ext: Optional[Element] = Field(
        description="Placeholder element for altId extensions",
        default=None,
        alias="_altId",
    )
    name: Optional[String] = Field(
        description="Human friendly name for the agent",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    requestor: Optional[Boolean] = Field(
        description="Whether user is initiator",
        default=None,
    )
    requestor_ext: Optional[Element] = Field(
        description="Placeholder element for requestor extensions",
        default=None,
        alias="_requestor",
    )
    location: Optional[Reference] = Field(
        description="Where",
        default=None,
    )
    policy: Optional[List[Uri]] = Field(
        description="Policy that authorized event",
        default=None,
    )
    policy_ext: Optional[Element] = Field(
        description="Placeholder element for policy extensions",
        default=None,
        alias="_policy",
    )
    media: Optional[Coding] = Field(
        description="Type of media",
        default=None,
    )
    network: Optional[AuditEventAgentNetwork] = Field(
        description="Logical network location for application activity",
        default=None,
    )
    purposeOfUse: Optional[List[CodeableConcept]] = Field(
        description="Reason given for this user",
        default=None,
    )

    @field_validator(
        *(
            "purposeOfUse",
            "network",
            "media",
            "policy",
            "location",
            "requestor",
            "name",
            "altId",
            "who",
            "role",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AuditEventSource(BackboneElement):
    """
    The system that is reporting the event.
    """

    site: Optional[String] = Field(
        description="Logical source location within the enterprise",
        default=None,
    )
    site_ext: Optional[Element] = Field(
        description="Placeholder element for site extensions",
        default=None,
        alias="_site",
    )
    observer: Optional[Reference] = Field(
        description="The identity of source detecting the event",
        default=None,
    )
    type: Optional[List[Coding]] = Field(
        description="The type of source where event originated",
        default=None,
    )

    @field_validator(
        *(
            "type",
            "observer",
            "site",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AuditEventEntityDetail(BackboneElement):
    """
    Tagged value pairs for conveying additional information about the entity.
    """

    type: Optional[String] = Field(
        description="Name of the property",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    valueString: Optional[String] = Field(
        description="Property value",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBase64Binary: Optional[Base64Binary] = Field(
        description="Property value",
        default=None,
    )
    valueBase64Binary_ext: Optional[Element] = Field(
        description="Placeholder element for valueBase64Binary extensions",
        default=None,
        alias="_valueBase64Binary",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Base64Binary],
            field_name_base="value",
            required=True,
        )


class AuditEventEntity(BackboneElement):
    """
    Specific instances of data or objects that have been accessed.
    """

    what: Optional[Reference] = Field(
        description="Specific instance of resource",
        default=None,
    )
    type: Optional[Coding] = Field(
        description="Type of entity involved",
        default=None,
    )
    role: Optional[Coding] = Field(
        description="What role the entity played",
        default=None,
    )
    lifecycle: Optional[Coding] = Field(
        description="Life-cycle stage for the entity",
        default=None,
    )
    securityLabel: Optional[List[Coding]] = Field(
        description="Security labels on the entity",
        default=None,
    )
    name: Optional[String] = Field(
        description="Descriptor for entity",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[String] = Field(
        description="Descriptive text",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    query: Optional[Base64Binary] = Field(
        description="Query parameters",
        default=None,
    )
    query_ext: Optional[Element] = Field(
        description="Placeholder element for query extensions",
        default=None,
        alias="_query",
    )
    detail: Optional[List[AuditEventEntityDetail]] = Field(
        description="Additional Information about the entity",
        default=None,
    )

    @field_validator(
        *(
            "detail",
            "query",
            "description",
            "name",
            "securityLabel",
            "lifecycle",
            "role",
            "type",
            "what",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AuditEvent(DomainResource):
    """
    A record of an event made for purposes of maintaining a security log. Typical uses include detection of intrusion attempts and monitoring for inappropriate usage.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/AuditEvent"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    type: Optional[Coding] = Field(
        description="Type/identifier of event",
        default=None,
    )
    subtype: Optional[List[Coding]] = Field(
        description="More specific type/id for the event",
        default=None,
    )
    action: Optional[Code] = Field(
        description="Type of action performed during the event",
        default=None,
    )
    action_ext: Optional[Element] = Field(
        description="Placeholder element for action extensions",
        default=None,
        alias="_action",
    )
    period: Optional[Period] = Field(
        description="When the activity occurred",
        default=None,
    )
    recorded: Optional[Instant] = Field(
        description="Time when the event was recorded",
        default=None,
    )
    recorded_ext: Optional[Element] = Field(
        description="Placeholder element for recorded extensions",
        default=None,
        alias="_recorded",
    )
    outcome: Optional[Code] = Field(
        description="Whether the event succeeded or failed",
        default=None,
    )
    outcome_ext: Optional[Element] = Field(
        description="Placeholder element for outcome extensions",
        default=None,
        alias="_outcome",
    )
    outcomeDesc: Optional[String] = Field(
        description="Description of the event outcome",
        default=None,
    )
    outcomeDesc_ext: Optional[Element] = Field(
        description="Placeholder element for outcomeDesc extensions",
        default=None,
        alias="_outcomeDesc",
    )
    purposeOfEvent: Optional[List[CodeableConcept]] = Field(
        description="The purposeOfUse of the event",
        default=None,
    )
    agent: Optional[List[AuditEventAgent]] = Field(
        description="Actor involved in the event",
        default=None,
    )
    source: Optional[AuditEventSource] = Field(
        description="Audit Event Reporter",
        default=None,
    )
    entity: Optional[List[AuditEventEntity]] = Field(
        description="Data or objects used",
        default=None,
    )
    resourceType: Literal["AuditEvent"] = Field(
        description=None,
        default="AuditEvent",
    )

    @field_validator(
        *(
            "entity",
            "source",
            "agent",
            "purposeOfEvent",
            "outcomeDesc",
            "outcome",
            "recorded",
            "period",
            "action",
            "subtype",
            "type",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("entity",), mode="after", check_fields=None)
    @classmethod
    def FHIR_sev_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="name.empty() or query.empty()",
            human="Either a name or a query (NOT both)",
            key="sev-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
