# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Age,
    Period,
    Range,
    Annotation,
    BackboneElement,
    DomainResource,
)


class AllergyIntoleranceReaction(BackboneElement):
    """
    Details about each adverse reaction event linked to exposure to the identified substance.
    """

    substance: Optional[CodeableConcept] = Field(
        description="Specific substance or pharmaceutical product considered to be responsible for event",
        default=None,
    )
    manifestation: Optional[List[CodeableConcept]] = Field(
        description="Clinical symptoms/signs associated with the Event",
        default=None,
    )
    description: Optional[String] = Field(
        description="Description of the event as a whole",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    onset: Optional[DateTime] = Field(
        description="Date(/time) when manifestations showed",
        default=None,
    )
    onset_ext: Optional[Element] = Field(
        description="Placeholder element for onset extensions",
        default=None,
        alias="_onset",
    )
    severity: Optional[Code] = Field(
        description="mild | moderate | severe (of event as a whole)",
        default=None,
    )
    severity_ext: Optional[Element] = Field(
        description="Placeholder element for severity extensions",
        default=None,
        alias="_severity",
    )
    exposureRoute: Optional[CodeableConcept] = Field(
        description="How the subject was exposed to the substance",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Text about event not captured in other fields",
        default=None,
    )

    @field_validator(
        *(
            "note",
            "exposureRoute",
            "severity",
            "onset",
            "description",
            "manifestation",
            "substance",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AllergyIntolerance(DomainResource):
    """
    Risk of harmful or undesirable, physiological response which is unique to an individual and associated with exposure to a substance.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/AllergyIntolerance"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External ids for this item",
        default=None,
    )
    clinicalStatus: Optional[CodeableConcept] = Field(
        description="active | inactive | resolved",
        default=None,
    )
    verificationStatus: Optional[CodeableConcept] = Field(
        description="unconfirmed | confirmed | refuted | entered-in-error",
        default=None,
    )
    type: Optional[Code] = Field(
        description="allergy | intolerance - Underlying mechanism (if known)",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    category: Optional[List[Code]] = Field(
        description="food | medication | environment | biologic",
        default=None,
    )
    category_ext: Optional[Element] = Field(
        description="Placeholder element for category extensions",
        default=None,
        alias="_category",
    )
    criticality: Optional[Code] = Field(
        description="low | high | unable-to-assess",
        default=None,
    )
    criticality_ext: Optional[Element] = Field(
        description="Placeholder element for criticality extensions",
        default=None,
        alias="_criticality",
    )
    code: Optional[CodeableConcept] = Field(
        description="Code that identifies the allergy or intolerance",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="Who the sensitivity is for",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter when the allergy or intolerance was asserted",
        default=None,
    )
    onsetDateTime: Optional[DateTime] = Field(
        description="When allergy or intolerance was identified",
        default=None,
    )
    onsetDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for onsetDateTime extensions",
        default=None,
        alias="_onsetDateTime",
    )
    onsetAge: Optional[Age] = Field(
        description="When allergy or intolerance was identified",
        default=None,
    )
    onsetPeriod: Optional[Period] = Field(
        description="When allergy or intolerance was identified",
        default=None,
    )
    onsetRange: Optional[Range] = Field(
        description="When allergy or intolerance was identified",
        default=None,
    )
    onsetString: Optional[String] = Field(
        description="When allergy or intolerance was identified",
        default=None,
    )
    onsetString_ext: Optional[Element] = Field(
        description="Placeholder element for onsetString extensions",
        default=None,
        alias="_onsetString",
    )
    recordedDate: Optional[DateTime] = Field(
        description="Date first version of the resource instance was recorded",
        default=None,
    )
    recordedDate_ext: Optional[Element] = Field(
        description="Placeholder element for recordedDate extensions",
        default=None,
        alias="_recordedDate",
    )
    recorder: Optional[Reference] = Field(
        description="Who recorded the sensitivity",
        default=None,
    )
    asserter: Optional[Reference] = Field(
        description="Source of the information about the allergy",
        default=None,
    )
    lastOccurrence: Optional[DateTime] = Field(
        description="Date(/time) of last known occurrence of a reaction",
        default=None,
    )
    lastOccurrence_ext: Optional[Element] = Field(
        description="Placeholder element for lastOccurrence extensions",
        default=None,
        alias="_lastOccurrence",
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional text not captured in other fields",
        default=None,
    )
    reaction: Optional[List[AllergyIntoleranceReaction]] = Field(
        description="Adverse Reaction Events linked to exposure to substance",
        default=None,
    )
    resourceType: Literal["AllergyIntolerance"] = Field(
        description=None,
        default="AllergyIntolerance",
    )

    @property
    def onset(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="onset",
        )

    @field_validator(
        *(
            "reaction",
            "note",
            "lastOccurrence",
            "asserter",
            "recorder",
            "recordedDate",
            "encounter",
            "patient",
            "code",
            "criticality",
            "category",
            "type",
            "verificationStatus",
            "clinicalStatus",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def onset_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Age, Period, Range, String],
            field_name_base="onset",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_ait_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="verificationStatus.coding.where(system = 'http://terminology.hl7.org/CodeSystem/allergyintolerance-verification' and code = 'entered-in-error').exists() or clinicalStatus.exists()",
            human="AllergyIntolerance.clinicalStatus SHALL be present if verificationStatus is not entered-in-error.",
            key="ait-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_ait_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="verificationStatus.coding.where(system = 'http://terminology.hl7.org/CodeSystem/allergyintolerance-verification' and code = 'entered-in-error').empty() or clinicalStatus.empty()",
            human="AllergyIntolerance.clinicalStatus SHALL NOT be present if verification Status is entered-in-error",
            key="ait-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
