# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    DomainResource,
)


class AdverseEventSuspectEntityCausality(BackboneElement):
    """
    Information on the possible cause of the event.
    """

    assessment: Optional[CodeableConcept] = Field(
        description="Assessment of if the entity caused the event",
        default=None,
    )
    productRelatedness: Optional[String] = Field(
        description="AdverseEvent.suspectEntity.causalityProductRelatedness",
        default=None,
    )
    productRelatedness_ext: Optional[Element] = Field(
        description="Placeholder element for productRelatedness extensions",
        default=None,
        alias="_productRelatedness",
    )
    author: Optional[Reference] = Field(
        description="AdverseEvent.suspectEntity.causalityAuthor",
        default=None,
    )
    method: Optional[CodeableConcept] = Field(
        description="ProbabilityScale | Bayesian | Checklist",
        default=None,
    )

    @field_validator(
        *(
            "method",
            "author",
            "productRelatedness",
            "assessment",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AdverseEventSuspectEntity(BackboneElement):
    """
    Describes the entity that is suspected to have caused the adverse event.
    """

    instance: Optional[Reference] = Field(
        description="Refers to the specific entity that caused the adverse event",
        default=None,
    )
    causality: Optional[List[AdverseEventSuspectEntityCausality]] = Field(
        description="Information on the possible cause of the event",
        default=None,
    )

    @field_validator(
        *(
            "causality",
            "instance",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class AdverseEvent(DomainResource):
    """
    Actual or  potential/avoided event causing unintended physical injury resulting from or contributed to by medical care, a research study or other healthcare setting factors that requires additional monitoring, treatment, or hospitalization, or that results in death.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/AdverseEvent"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="Business identifier for the event",
        default=None,
    )
    actuality: Optional[Code] = Field(
        description="actual | potential",
        default=None,
    )
    actuality_ext: Optional[Element] = Field(
        description="Placeholder element for actuality extensions",
        default=None,
        alias="_actuality",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="product-problem | product-quality | product-use-error | wrong-dose | incorrect-prescribing-information | wrong-technique | wrong-route-of-administration | wrong-rate | wrong-duration | wrong-time | expired-drug | medical-device-use-error | problem-different-manufacturer | unsafe-physical-environment",
        default=None,
    )
    event: Optional[CodeableConcept] = Field(
        description="Type of the event itself in relation to the subject",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Subject impacted by event",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter created as part of",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="When the event occurred",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    detected: Optional[DateTime] = Field(
        description="When the event was detected",
        default=None,
    )
    detected_ext: Optional[Element] = Field(
        description="Placeholder element for detected extensions",
        default=None,
        alias="_detected",
    )
    recordedDate: Optional[DateTime] = Field(
        description="When the event was recorded",
        default=None,
    )
    recordedDate_ext: Optional[Element] = Field(
        description="Placeholder element for recordedDate extensions",
        default=None,
        alias="_recordedDate",
    )
    resultingCondition: Optional[List[Reference]] = Field(
        description="Effect on the subject due to this event",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Location where adverse event occurred",
        default=None,
    )
    seriousness: Optional[CodeableConcept] = Field(
        description="Seriousness of the event",
        default=None,
    )
    severity: Optional[CodeableConcept] = Field(
        description="mild | moderate | severe",
        default=None,
    )
    outcome: Optional[CodeableConcept] = Field(
        description="resolved | recovering | ongoing | resolvedWithSequelae | fatal | unknown",
        default=None,
    )
    recorder: Optional[Reference] = Field(
        description="Who recorded the adverse event",
        default=None,
    )
    contributor: Optional[List[Reference]] = Field(
        description="Who  was involved in the adverse event or the potential adverse event",
        default=None,
    )
    suspectEntity: Optional[List[AdverseEventSuspectEntity]] = Field(
        description="The suspected agent causing the adverse event",
        default=None,
    )
    subjectMedicalHistory: Optional[List[Reference]] = Field(
        description="AdverseEvent.subjectMedicalHistory",
        default=None,
    )
    referenceDocument: Optional[List[Reference]] = Field(
        description="AdverseEvent.referenceDocument",
        default=None,
    )
    study: Optional[List[Reference]] = Field(
        description="AdverseEvent.study",
        default=None,
    )
    resourceType: Literal["AdverseEvent"] = Field(
        description=None,
        default="AdverseEvent",
    )

    @field_validator(
        *(
            "study",
            "referenceDocument",
            "subjectMedicalHistory",
            "suspectEntity",
            "contributor",
            "recorder",
            "outcome",
            "severity",
            "seriousness",
            "location",
            "resultingCondition",
            "recordedDate",
            "detected",
            "date",
            "encounter",
            "subject",
            "event",
            "category",
            "actuality",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
