"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeAspects = exports.ModelSynthesizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs = require("fs-extra");
const annotations_1 = require("./annotations");
const aspect_1 = require("./aspect");
const yaml_1 = require("./yaml");
class ModelSynthesizer {
    constructor(model, continueOnErrorAnnotations = false) {
        this.model = model;
        this.continueOnErrorAnnotations = continueOnErrorAnnotations;
    }
    synthesize(session) {
        invokeAspects(this.model);
        if (!session.skipValidation) {
            const errors = this.model.node
                .findAll()
                .map((node) => node.node
                .validate()
                .map((error) => ({ message: error, source: node })))
                .reduce((prev, curr) => [...prev, ...curr], []);
            if (errors.length > 0) {
                const errorList = errors
                    .map((e) => `[${e.source.node.path}] ${e.message}`)
                    .join("\n  ");
                throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
            }
            const manifest = session.manifest;
            const modelManifest = manifest.forModel(this.model);
            const workingDirectory = path.join(session.outdir, modelManifest.workingDirectory);
            if (!fs.existsSync(workingDirectory))
                fs.mkdirSync(workingDirectory);
            // collect Annotations into Manifest
            const annotations = this.model.node
                .findAll()
                .map((node) => ({
                node,
                metadatas: node.node.metadata.filter(isAnnotationMetadata),
            }))
                .map(({ node, metadatas }) => metadatas.map((metadata) => ({
                constructPath: node.node.path,
                level: metadata.type,
                message: metadata.data,
                stacktrace: metadata.trace,
            })))
                .reduce((list, metadatas) => [...list, ...metadatas], []); // Array.flat()
            // it is readonly but this is the place where we are allowed to write to it
            modelManifest.annotations = annotations;
            // abort if one or more error annotations have been encountered
            if (!this.continueOnErrorAnnotations &&
                annotations.some(isErrorAnnotation)) {
                throw new Error(`Encountered Annotations with level "ERROR":\n${annotations
                    .filter(isErrorAnnotation)
                    .map((a) => `[${a.constructPath}] ${a.message}`)
                    .join("\n")}`);
            }
            const threagile = this.model._toThreagile();
            yaml_1.Yaml.save(path.join(session.outdir, modelManifest.synthesizedModelPath), [
                threagile,
            ]);
        }
    }
}
exports.ModelSynthesizer = ModelSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
ModelSynthesizer[_a] = { fqn: "cdktg.ModelSynthesizer", version: "0.0.11" };
const invokedByPath = {};
/**
 * Invoke aspects on the given construct tree.
 *
 * originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/@aws-cdk/core/lib/private/synthesis.ts#L99-L137
 */
function invokeAspects(root) {
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...(inheritedAspects ?? []), ...aspects.all];
        const nodeAspectsCount = aspects.all.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.all.length) {
                annotations_1.Annotations.of(construct).addWarning("We detected an Aspect was added via another Aspect, and will not be applied");
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            recurse(child, allAspectsHere);
        }
    }
}
exports.invokeAspects = invokeAspects;
const annotationMetadataEntryTypes = [
    annotations_1.AnnotationMetadataEntryType.INFO,
    annotations_1.AnnotationMetadataEntryType.WARN,
    annotations_1.AnnotationMetadataEntryType.ERROR,
];
function isAnnotationMetadata(metadata) {
    return annotationMetadataEntryTypes.includes(metadata.type);
}
function isErrorAnnotation(annotation) {
    return annotation.level === annotations_1.AnnotationMetadataEntryType.ERROR;
}
//# sourceMappingURL=data:application/json;base64,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