"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UrlShortener = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const redirect_function_1 = require("./redirect-function");
const shortener_function_1 = require("./shortener-function");
/**
 * URL shortener
 */
class UrlShortener extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        // Table to save a counter
        const table = new dynamodb.Table(this, 'Table', {
            partitionKey: {
                name: 'key',
                type: dynamodb.AttributeType.STRING,
            },
        });
        // Bucket to save redirects
        const bucket = new s3.Bucket(this, 'Bucket', {
            lifecycleRules: [{
                    expiration: (_b = props.expiration) !== null && _b !== void 0 ? _b : aws_cdk_lib_1.Duration.days(365),
                }],
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        // CloudFront distribution
        const domainName = props.recordName ? `${props.recordName}.${props.hostedZone.zoneName}` : props.hostedZone.zoneName;
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        const distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(bucket),
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        functionVersion: new redirect_function_1.RedirectFunction(this, 'Redirect'),
                    },
                ],
            },
            certificate,
            domainNames: [domainName],
        });
        // Route53 records
        new route53.ARecord(this, 'ARecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        // Lambda function to increment counter and write redirect in bucket
        const handler = new shortener_function_1.ShortenerFunction(this, 'handler', {
            logRetention: logs.RetentionDays.ONE_MONTH,
            environment: {
                DOMAIN_NAME: domainName,
                BUCKET_NAME: bucket.bucketName,
                TABLE_NAME: table.tableName,
            },
        });
        if (props.corsAllowOrigins) {
            handler.addEnvironment('CORS_ALLOW_ORIGINS', props.corsAllowOrigins.join(' '));
        }
        bucket.grantPut(handler);
        bucket.grantPutAcl(handler);
        table.grant(handler, 'dynamodb:UpdateItem');
        // API
        this.api = new apigateway.RestApi(this, `UrlShortener${props.hostedZone.zoneName}`, {
            endpointTypes: props.apiGatewayEndpoint ? [apigateway.EndpointType.PRIVATE] : undefined,
            policy: props.apiGatewayEndpoint
                ? new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: ['execute-api:Invoke'],
                            principals: [new iam.AnyPrincipal()],
                            resources: [aws_cdk_lib_1.Fn.join('', ['execute-api:/', '*'])],
                            conditions: {
                                StringEquals: { 'aws:SourceVpce': props.apiGatewayEndpoint.vpcEndpointId },
                            },
                        }),
                    ],
                })
                : undefined,
            defaultCorsPreflightOptions: props.corsAllowOrigins
                ? { allowOrigins: props.corsAllowOrigins }
                : undefined,
        });
        this.api.root.addMethod('ANY', new apigateway.LambdaIntegration(handler), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.api.root
            .addResource('{proxy+}')
            .addMethod('ANY', new apigateway.LambdaIntegration(handler), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.apiEndpoint = this.api.url;
    }
}
exports.UrlShortener = UrlShortener;
_a = JSII_RTTI_SYMBOL_1;
UrlShortener[_a] = { fqn: "cloudstructs.UrlShortener", version: "0.4.28" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvdXJsLXNob3J0ZW5lci9pbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQUEyQztBQUMzQyx5REFBeUQ7QUFDekQsMERBQTBEO0FBQzFELHlEQUF5RDtBQUN6RCw4REFBOEQ7QUFDOUQscURBQXFEO0FBRXJELDJDQUEyQztBQUMzQyw2Q0FBNkM7QUFDN0MsbURBQW1EO0FBQ25ELDJEQUEyRDtBQUMzRCx5Q0FBeUM7QUFDekMsMkNBQXVDO0FBQ3ZDLDJEQUF1RDtBQUN2RCw2REFBeUQ7QUFrRHpEOztHQUVHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsc0JBQVM7SUFXekMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF3Qjs7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQiwwQkFBMEI7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDOUMsWUFBWSxFQUFFO2dCQUNaLElBQUksRUFBRSxLQUFLO2dCQUNYLElBQUksRUFBRSxRQUFRLENBQUMsYUFBYSxDQUFDLE1BQU07YUFDcEM7U0FDRixDQUFDLENBQUM7UUFFSCwyQkFBMkI7UUFDM0IsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUU7WUFDM0MsY0FBYyxFQUFFLENBQUM7b0JBQ2YsVUFBVSxRQUFFLEtBQUssQ0FBQyxVQUFVLG1DQUFJLHNCQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztpQkFDbkQsQ0FBQztZQUNGLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO1NBQ2xELENBQUMsQ0FBQztRQUVILDBCQUEwQjtRQUMxQixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUM7UUFDckgsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsdUJBQXVCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUN2RSxVQUFVO1lBQ1YsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1lBQzVCLE1BQU0sRUFBRSxXQUFXO1NBQ3BCLENBQUMsQ0FBQztRQUNILE1BQU0sWUFBWSxHQUFHLElBQUksVUFBVSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3JFLGVBQWUsRUFBRTtnQkFDZixNQUFNLEVBQUUsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQztnQkFDcEMsV0FBVyxFQUFFO29CQUNYO3dCQUNFLFNBQVMsRUFBRSxVQUFVLENBQUMsbUJBQW1CLENBQUMsZUFBZTt3QkFDekQsZUFBZSxFQUFFLElBQUksb0NBQWdCLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQztxQkFDeEQ7aUJBQ0Y7YUFDRjtZQUNELFdBQVc7WUFDWCxXQUFXLEVBQUUsQ0FBQyxVQUFVLENBQUM7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsa0JBQWtCO1FBQ2xCLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ25DLElBQUksRUFBRSxLQUFLLENBQUMsVUFBVTtZQUN0QixNQUFNLEVBQUUsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDbEYsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1NBQzdCLENBQUMsQ0FBQztRQUNILElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3pDLElBQUksRUFBRSxLQUFLLENBQUMsVUFBVTtZQUN0QixNQUFNLEVBQUUsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDbEYsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1NBQzdCLENBQUMsQ0FBQztRQUVILG9FQUFvRTtRQUNwRSxNQUFNLE9BQU8sR0FBRyxJQUFJLHNDQUFpQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDckQsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztZQUMxQyxXQUFXLEVBQUU7Z0JBQ1gsV0FBVyxFQUFFLFVBQVU7Z0JBQ3ZCLFdBQVcsRUFBRSxNQUFNLENBQUMsVUFBVTtnQkFDOUIsVUFBVSxFQUFFLEtBQUssQ0FBQyxTQUFTO2FBQzVCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsT0FBTyxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDaEY7UUFDRCxNQUFNLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3pCLE1BQU0sQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDNUIsS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUscUJBQXFCLENBQUMsQ0FBQztRQUU1QyxNQUFNO1FBQ04sSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLFVBQVUsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLGVBQWUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsRUFBRTtZQUNsRixhQUFhLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDdkYsTUFBTSxFQUFFLEtBQUssQ0FBQyxrQkFBa0I7Z0JBQzlCLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7b0JBQ3ZCLFVBQVUsRUFBRTt3QkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7NEJBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7NEJBQ3hCLE9BQU8sRUFBRSxDQUFDLG9CQUFvQixDQUFDOzRCQUMvQixVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxZQUFZLEVBQUUsQ0FBQzs0QkFDcEMsU0FBUyxFQUFFLENBQUMsZ0JBQUUsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsZUFBZSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7NEJBQ2hELFVBQVUsRUFBRTtnQ0FDVixZQUFZLEVBQUUsRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsYUFBYSxFQUFFOzZCQUMzRTt5QkFDRixDQUFDO3FCQUNIO2lCQUNGLENBQUM7Z0JBQ0YsQ0FBQyxDQUFDLFNBQVM7WUFDYiwyQkFBMkIsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO2dCQUNqRCxDQUFDLENBQUMsRUFBRSxZQUFZLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixFQUFFO2dCQUMxQyxDQUFDLENBQUMsU0FBUztTQUNkLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDeEUsVUFBVSxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7U0FDdkMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJO2FBQ1YsV0FBVyxDQUFDLFVBQVUsQ0FBQzthQUN2QixTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzNELFVBQVUsRUFBRSxLQUFLLENBQUMsb0JBQW9CO1NBQ3ZDLENBQUMsQ0FBQztRQUVMLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUM7SUFDbEMsQ0FBQzs7QUFoSEgsb0NBaUhDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24sIEZuIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgYXBpZ2F0ZXdheSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXBpZ2F0ZXdheSc7XG5pbXBvcnQgKiBhcyBhY20gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNlcnRpZmljYXRlbWFuYWdlcic7XG5pbXBvcnQgKiBhcyBjbG91ZGZyb250IGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZGZyb250JztcbmltcG9ydCAqIGFzIG9yaWdpbnMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnQtb3JpZ2lucyc7XG5pbXBvcnQgKiBhcyBkeW5hbW9kYiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZHluYW1vZGInO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyByb3V0ZTUzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXJvdXRlNTMtdGFyZ2V0cyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBSZWRpcmVjdEZ1bmN0aW9uIH0gZnJvbSAnLi9yZWRpcmVjdC1mdW5jdGlvbic7XG5pbXBvcnQgeyBTaG9ydGVuZXJGdW5jdGlvbiB9IGZyb20gJy4vc2hvcnRlbmVyLWZ1bmN0aW9uJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIFVybFNob3J0ZW5lclxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVybFNob3J0ZW5lclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBob3N0ZWQgem9uZSBmb3IgdGhlIHNob3J0IFVSTHMgZG9tYWluXG4gICAqL1xuICByZWFkb25seSBob3N0ZWRab25lOiByb3V0ZTUzLklIb3N0ZWRab25lO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVjb3JkIG5hbWUgdG8gdXNlIGluIHRoZSBob3N0ZWQgem9uZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHpvbmUgcm9vdFxuICAgKi9cbiAgcmVhZG9ubHkgcmVjb3JkTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogRXhwaXJhdGlvbiBmb3Igc2hvcnQgdXJsc1xuICAgKlxuICAgKiBAZGVmYXVsdCBjZGsuRHVyYXRpb24uZGF5cygzNjUpXG4gICAqL1xuICByZWFkb25seSBleHBpcmF0aW9uPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIEFuIGludGVyZmFjZSBWUEMgZW5kcG9pbnQgZm9yIEFQSSBnYXRld2F5LiBTcGVjaWZ5aW5nIHRoaXMgcHJvcGVydHkgd2lsbFxuICAgKiBtYWtlIHRoZSBBUEkgcHJpdmF0ZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYXBpZ2F0ZXdheS1wcml2YXRlLWFwaXMuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFQSSBpcyBwdWJsaWNcbiAgICovXG4gIHJlYWRvbmx5IGFwaUdhdGV3YXlFbmRwb2ludD86IGVjMi5JSW50ZXJmYWNlVnBjRW5kcG9pbnQ7XG5cbiAgLyoqXG4gICAqIEF1dGhvcml6ZXIgZm9yIEFQSSBnYXRld2F5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRvIG5vdCB1c2UgYW4gYXV0aG9yaXplciBmb3IgdGhlIEFQSVxuICAgKi9cbiAgcmVhZG9ubHkgYXBpR2F0ZXdheUF1dGhvcml6ZXI/OiBhcGlnYXRld2F5LklBdXRob3JpemVyO1xuXG4gIC8qKlxuICAgKiBBbGxvd2VkIG9yaWdpbnMgZm9yIENPUlNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBDT1JTIGlzIG5vdCBlbmFibGVkXG4gICAqL1xuICByZWFkb25seSBjb3JzQWxsb3dPcmlnaW5zPzogc3RyaW5nW107XG59XG5cbi8qKlxuICogVVJMIHNob3J0ZW5lclxuICovXG5leHBvcnQgY2xhc3MgVXJsU2hvcnRlbmVyIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIFRoZSBlbmRwb2ludCBvZiB0aGUgVVJMIHNob3J0ZW5lciBBUElcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcGlFbmRwb2ludDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgdW5kZXJseWluZyBBUEkgR2F0ZXdheSBSRVNUIEFQSVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFwaTogYXBpZ2F0ZXdheS5MYW1iZGFSZXN0QXBpO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBVcmxTaG9ydGVuZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAvLyBUYWJsZSB0byBzYXZlIGEgY291bnRlclxuICAgIGNvbnN0IHRhYmxlID0gbmV3IGR5bmFtb2RiLlRhYmxlKHRoaXMsICdUYWJsZScsIHtcbiAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICBuYW1lOiAna2V5JyxcbiAgICAgICAgdHlwZTogZHluYW1vZGIuQXR0cmlidXRlVHlwZS5TVFJJTkcsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gQnVja2V0IHRvIHNhdmUgcmVkaXJlY3RzXG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnQnVja2V0Jywge1xuICAgICAgbGlmZWN5Y2xlUnVsZXM6IFt7XG4gICAgICAgIGV4cGlyYXRpb246IHByb3BzLmV4cGlyYXRpb24gPz8gRHVyYXRpb24uZGF5cygzNjUpLFxuICAgICAgfV0sXG4gICAgICBibG9ja1B1YmxpY0FjY2VzczogczMuQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgIH0pO1xuXG4gICAgLy8gQ2xvdWRGcm9udCBkaXN0cmlidXRpb25cbiAgICBjb25zdCBkb21haW5OYW1lID0gcHJvcHMucmVjb3JkTmFtZSA/IGAke3Byb3BzLnJlY29yZE5hbWV9LiR7cHJvcHMuaG9zdGVkWm9uZS56b25lTmFtZX1gIDogcHJvcHMuaG9zdGVkWm9uZS56b25lTmFtZTtcbiAgICBjb25zdCBjZXJ0aWZpY2F0ZSA9IG5ldyBhY20uRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUodGhpcywgJ0NlcnRpZmljYXRlJywge1xuICAgICAgZG9tYWluTmFtZSxcbiAgICAgIGhvc3RlZFpvbmU6IHByb3BzLmhvc3RlZFpvbmUsXG4gICAgICByZWdpb246ICd1cy1lYXN0LTEnLFxuICAgIH0pO1xuICAgIGNvbnN0IGRpc3RyaWJ1dGlvbiA9IG5ldyBjbG91ZGZyb250LkRpc3RyaWJ1dGlvbih0aGlzLCAnRGlzdHJpYnV0aW9uJywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIG9yaWdpbjogbmV3IG9yaWdpbnMuUzNPcmlnaW4oYnVja2V0KSxcbiAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBldmVudFR5cGU6IGNsb3VkZnJvbnQuTGFtYmRhRWRnZUV2ZW50VHlwZS5PUklHSU5fUkVTUE9OU0UsXG4gICAgICAgICAgICBmdW5jdGlvblZlcnNpb246IG5ldyBSZWRpcmVjdEZ1bmN0aW9uKHRoaXMsICdSZWRpcmVjdCcpLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICAgY2VydGlmaWNhdGUsXG4gICAgICBkb21haW5OYW1lczogW2RvbWFpbk5hbWVdLFxuICAgIH0pO1xuXG4gICAgLy8gUm91dGU1MyByZWNvcmRzXG4gICAgbmV3IHJvdXRlNTMuQVJlY29yZCh0aGlzLCAnQVJlY29yZCcsIHtcbiAgICAgIHpvbmU6IHByb3BzLmhvc3RlZFpvbmUsXG4gICAgICB0YXJnZXQ6IHJvdXRlNTMuUmVjb3JkVGFyZ2V0LmZyb21BbGlhcyhuZXcgdGFyZ2V0cy5DbG91ZEZyb250VGFyZ2V0KGRpc3RyaWJ1dGlvbikpLFxuICAgICAgcmVjb3JkTmFtZTogcHJvcHMucmVjb3JkTmFtZSxcbiAgICB9KTtcbiAgICBuZXcgcm91dGU1My5BYWFhUmVjb3JkKHRoaXMsICdBYWFhUmVjb3JkJywge1xuICAgICAgem9uZTogcHJvcHMuaG9zdGVkWm9uZSxcbiAgICAgIHRhcmdldDogcm91dGU1My5SZWNvcmRUYXJnZXQuZnJvbUFsaWFzKG5ldyB0YXJnZXRzLkNsb3VkRnJvbnRUYXJnZXQoZGlzdHJpYnV0aW9uKSksXG4gICAgICByZWNvcmROYW1lOiBwcm9wcy5yZWNvcmROYW1lLFxuICAgIH0pO1xuXG4gICAgLy8gTGFtYmRhIGZ1bmN0aW9uIHRvIGluY3JlbWVudCBjb3VudGVyIGFuZCB3cml0ZSByZWRpcmVjdCBpbiBidWNrZXRcbiAgICBjb25zdCBoYW5kbGVyID0gbmV3IFNob3J0ZW5lckZ1bmN0aW9uKHRoaXMsICdoYW5kbGVyJywge1xuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgRE9NQUlOX05BTUU6IGRvbWFpbk5hbWUsXG4gICAgICAgIEJVQ0tFVF9OQU1FOiBidWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgVEFCTEVfTkFNRTogdGFibGUudGFibGVOYW1lLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBpZiAocHJvcHMuY29yc0FsbG93T3JpZ2lucykge1xuICAgICAgaGFuZGxlci5hZGRFbnZpcm9ubWVudCgnQ09SU19BTExPV19PUklHSU5TJywgcHJvcHMuY29yc0FsbG93T3JpZ2lucy5qb2luKCcgJykpO1xuICAgIH1cbiAgICBidWNrZXQuZ3JhbnRQdXQoaGFuZGxlcik7XG4gICAgYnVja2V0LmdyYW50UHV0QWNsKGhhbmRsZXIpO1xuICAgIHRhYmxlLmdyYW50KGhhbmRsZXIsICdkeW5hbW9kYjpVcGRhdGVJdGVtJyk7XG5cbiAgICAvLyBBUElcbiAgICB0aGlzLmFwaSA9IG5ldyBhcGlnYXRld2F5LlJlc3RBcGkodGhpcywgYFVybFNob3J0ZW5lciR7cHJvcHMuaG9zdGVkWm9uZS56b25lTmFtZX1gLCB7XG4gICAgICBlbmRwb2ludFR5cGVzOiBwcm9wcy5hcGlHYXRld2F5RW5kcG9pbnQgPyBbYXBpZ2F0ZXdheS5FbmRwb2ludFR5cGUuUFJJVkFURV0gOiB1bmRlZmluZWQsXG4gICAgICBwb2xpY3k6IHByb3BzLmFwaUdhdGV3YXlFbmRwb2ludFxuICAgICAgICA/IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2V4ZWN1dGUtYXBpOkludm9rZSddLFxuICAgICAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IGlhbS5BbnlQcmluY2lwYWwoKV0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW0ZuLmpvaW4oJycsIFsnZXhlY3V0ZS1hcGk6LycsICcqJ10pXSxcbiAgICAgICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgICAgIFN0cmluZ0VxdWFsczogeyAnYXdzOlNvdXJjZVZwY2UnOiBwcm9wcy5hcGlHYXRld2F5RW5kcG9pbnQudnBjRW5kcG9pbnRJZCB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSlcbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBkZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM6IHByb3BzLmNvcnNBbGxvd09yaWdpbnNcbiAgICAgICAgPyB7IGFsbG93T3JpZ2luczogcHJvcHMuY29yc0FsbG93T3JpZ2lucyB9XG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcGkucm9vdC5hZGRNZXRob2QoJ0FOWScsIG5ldyBhcGlnYXRld2F5LkxhbWJkYUludGVncmF0aW9uKGhhbmRsZXIpLCB7XG4gICAgICBhdXRob3JpemVyOiBwcm9wcy5hcGlHYXRld2F5QXV0aG9yaXplcixcbiAgICB9KTtcbiAgICB0aGlzLmFwaS5yb290XG4gICAgICAuYWRkUmVzb3VyY2UoJ3twcm94eSt9JylcbiAgICAgIC5hZGRNZXRob2QoJ0FOWScsIG5ldyBhcGlnYXRld2F5LkxhbWJkYUludGVncmF0aW9uKGhhbmRsZXIpLCB7XG4gICAgICAgIGF1dGhvcml6ZXI6IHByb3BzLmFwaUdhdGV3YXlBdXRob3JpemVyLFxuICAgICAgfSk7XG5cbiAgICB0aGlzLmFwaUVuZHBvaW50ID0gdGhpcy5hcGkudXJsO1xuICB9XG59XG4iXX0=