"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const eks = require("../lib");
const util_1 = require("./util");
/* eslint-disable max-len */
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'a default cluster spans all subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.LegacyCluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Cluster', {
            ResourcesVpcConfig: {
                SubnetIds: [
                    { Ref: 'VPCPublicSubnet1SubnetB4246D30' },
                    { Ref: 'VPCPublicSubnet2Subnet74179F39' },
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        }));
        test.done();
    },
    'if "vpc" is not specified, vpc with default configuration will be created'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        new eks.LegacyCluster(stack, 'cluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        test.done();
    },
    'default capacity': {
        'x2 m5.large by default'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.LegacyCluster(stack, 'cluster', { version: CLUSTER_VERSION });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                InstanceTypes: [
                    'm5.large',
                ],
                ScalingConfig: {
                    DesiredSize: 2,
                    MaxSize: 2,
                    MinSize: 2,
                },
            }));
            test.done();
        },
        'quantity and type can be customized'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.LegacyCluster(stack, 'cluster', {
                defaultCapacity: 10,
                defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
                version: CLUSTER_VERSION,
            });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                ScalingConfig: {
                    DesiredSize: 10,
                    MaxSize: 10,
                    MinSize: 10,
                },
            }));
            // expect(stack).to(haveResource('AWS::AutoScaling::LaunchConfiguration', { InstanceType: 'm2.xlarge' }));
            test.done();
        },
        'defaultCapacity=0 will not allocate at all'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.LegacyCluster(stack, 'cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.ok(!cluster.defaultCapacity);
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration'));
            test.done();
        },
    },
    'creating a cluster tags the private VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.LegacyCluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Private' },
                { Key: 'aws-cdk:subnet-type', Value: 'Private' },
                { Key: 'kubernetes.io/role/internal-elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PrivateSubnet1' },
            ],
        }));
        test.done();
    },
    'creating a cluster tags the public VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.LegacyCluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            MapPublicIpOnLaunch: true,
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Public' },
                { Key: 'aws-cdk:subnet-type', Value: 'Public' },
                { Key: 'kubernetes.io/role/elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PublicSubnet1' },
            ],
        }));
        test.done();
    },
    'adding capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'ClusterEB0386A7' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Default',
                },
            ],
        }));
        test.done();
    },
    'create nodegroup with existing role'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        const cluster = new eks.LegacyCluster(stack, 'cluster', {
            defaultCapacity: 10,
            defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
            version: CLUSTER_VERSION,
        });
        const existingRole = new iam.Role(stack, 'ExistingRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            nodeRole: existingRole,
        });
        // THEN
        test.ok(cluster.defaultNodegroup);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
            ScalingConfig: {
                DesiredSize: 10,
                MaxSize: 10,
                MinSize: 10,
            },
        }));
        test.done();
    },
    'adding bottlerocket capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'ClusterEB0386A7' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Bottlerocket',
                },
            ],
        }));
        test.done();
    },
    'adding bottlerocket capacity with bootstrapOptions throws error'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        test.throws(() => cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
            bootstrapOptions: {},
        }), /bootstrapOptions is not supported for Bottlerocket/);
        test.done();
    },
    'exercise export/import'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.LegacyCluster(stack1, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        const imported = eks.LegacyCluster.fromClusterAttributes(stack2, 'Imported', {
            clusterArn: cluster.clusterArn,
            vpc: cluster.vpc,
            clusterEndpoint: cluster.clusterEndpoint,
            clusterName: cluster.clusterName,
            securityGroups: cluster.connections.securityGroups,
            clusterCertificateAuthorityData: cluster.clusterCertificateAuthorityData,
            clusterSecurityGroupId: cluster.clusterSecurityGroupId,
            clusterEncryptionConfigKeyArn: cluster.clusterEncryptionConfigKeyArn,
        });
        // this should cause an export/import
        new cdk.CfnOutput(stack2, 'ClusterARN', { value: imported.clusterArn });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                ClusterARN: {
                    Value: {
                        'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttClusterEB0386A7Arn2F2E3C3F',
                    },
                },
            },
        });
        test.done();
    },
    'disabled features when kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        test.throws(() => cluster.addCapacity('boo', { instanceType: new ec2.InstanceType('r5d.24xlarge'), mapRole: true }), /Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster/);
        test.done();
    },
    'addCapacity will *not* map the IAM role if mapRole is false'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
            mapRole: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE)));
        test.done();
    },
    'addCapacity will *not* map the IAM role if kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE)));
        test.done();
    },
    'outputs': {
        'aws eks update-kubeconfig is the only output synthesized by default'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.LegacyCluster(stack, 'Cluster', { version: CLUSTER_VERSION });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'ClusterEB0386A7' }, ' --region us-east-1']] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'ClusterEB0386A7' }, ' --region us-east-1']] } },
            });
            test.done();
        },
        'if `outputConfigCommand=false` will disabled the output'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.LegacyCluster(stack, 'Cluster', {
                outputConfigCommand: false,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.ok(!template.Outputs); // no outputs
            test.done();
        },
        '`outputClusterName` can be used to synthesize an output with the cluster name'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.LegacyCluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputClusterName: true,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterClusterNameEB26049E: { Value: { Ref: 'ClusterEB0386A7' } },
            });
            test.done();
        },
        'boostrap user-data': {
            'rendered by default for ASGs'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.LegacyCluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', { instanceType: new ec2.InstanceType('m3.xlargs') });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'ClusterEB0386A7' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'not rendered if bootstrap is disabled'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.LegacyCluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapEnabled: false,
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': '#!/bin/bash' });
                test.done();
            },
            // cursory test for options: see test.user-data.ts for full suite
            'bootstrap options'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.LegacyCluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapOptions: {
                        kubeletExtraArgs: '--node-labels FOO=42',
                    },
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'ClusterEB0386A7' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand  --node-labels FOO=42" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'spot instances': {
                'nodes labeled an tainted accordingly'(test) {
                    // GIVEN
                    const { app, stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.LegacyCluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    const template = app.synth().getStackByName(stack.stackName).template;
                    const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                    test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'ClusterEB0386A7' }, ' --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                    test.done();
                },
                'if kubectl is disabled, interrupt handler is not added'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.LegacyCluster(stack, 'Cluster', {
                        defaultCapacity: 0,
                        version: CLUSTER_VERSION,
                    });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).notTo(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE));
                    test.done();
                },
            },
        },
        'if bootstrap is disabled cannot specify options'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.LegacyCluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.throws(() => cluster.addCapacity('MyCapcity', {
                instanceType: new ec2.InstanceType('m3.xlargs'),
                bootstrapEnabled: false,
                bootstrapOptions: { awsApiRetryAttempts: 10 },
            }), /Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false/);
            test.done();
        },
        'EksOptimizedImage() with no nodeType always uses STANDARD with LATEST_KUBERNETES_VERSION'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const LATEST_KUBERNETES_VERSION = '1.14';
            // WHEN
            new eks.EksOptimizedImage().getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes(LATEST_KUBERNETES_VERSION)), 'LATEST_KUBERNETES_VERSION should be in ssm parameters');
            test.done();
        },
        'EksOptimizedImage() with specific kubernetesVersion return correct AMI'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.EksOptimizedImage({ kubernetesVersion: '1.15' }).getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/1.15/')), 'kubernetesVersion should be in ssm parameters');
            test.done();
        },
        'EKS-Optimized AMI with GPU support when addCapacity'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.LegacyCluster(stack, 'cluster', {
                defaultCapacity: 0,
                version: CLUSTER_VERSION,
            }).addCapacity('GPUCapacity', {
                instanceType: new ec2.InstanceType('g4dn.xlarge'),
            });
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') && v.Default.includes('amazon-linux-2-gpu')), 'EKS AMI with GPU should be in ssm parameters');
            test.done();
        },
    },
    'create a cluster with secrets encryption using KMS CMK'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            version: CLUSTER_VERSION,
            secretsEncryptionKey: new kms.Key(stack, 'Key'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Cluster', {
            EncryptionConfig: [{
                    Provider: {
                        KeyArn: {
                            'Fn::GetAtt': [
                                'Key961B73FD',
                                'Arn',
                            ],
                        },
                    },
                    Resources: ['secrets'],
                }],
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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