"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAccount = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
/**
 * Service Account
 */
class ServiceAccount extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const { cluster } = props;
        this.serviceAccountName = (_a = props.name) !== null && _a !== void 0 ? _a : this.node.uniqueId.toLowerCase();
        this.serviceAccountNamespace = (_b = props.namespace) !== null && _b !== void 0 ? _b : 'default';
        /* Add conditions to the role to improve security. This prevents other pods in the same namespace to assume the role.
        * See documentation: https://docs.aws.amazon.com/eks/latest/userguide/create-service-account-iam-policy-and-role.html
        */
        const conditions = new core_1.CfnJson(this, 'ConditionJson', {
            value: {
                [`${cluster.clusterOpenIdConnectIssuer}:aud`]: 'sts.amazonaws.com',
                [`${cluster.clusterOpenIdConnectIssuer}:sub`]: `system:serviceaccount:${this.serviceAccountNamespace}:${this.serviceAccountName}`,
            },
        });
        const principal = new aws_iam_1.OpenIdConnectPrincipal(cluster.openIdConnectProvider).withConditions({
            StringEquals: conditions,
        });
        this.role = new aws_iam_1.Role(this, 'Role', { assumedBy: principal });
        this.assumeRoleAction = this.role.assumeRoleAction;
        this.grantPrincipal = this.role.grantPrincipal;
        this.policyFragment = this.role.policyFragment;
        cluster.addManifest(`${id}ServiceAccountResource`, {
            apiVersion: 'v1',
            kind: 'ServiceAccount',
            metadata: {
                name: this.serviceAccountName,
                namespace: this.serviceAccountNamespace,
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                },
                annotations: {
                    'eks.amazonaws.com/role-arn': this.role.roleArn,
                },
            },
        });
    }
    addToPolicy(statement) {
        return this.role.addToPolicy(statement);
    }
    addToPrincipalPolicy(statement) {
        return this.role.addToPrincipalPolicy(statement);
    }
}
exports.ServiceAccount = ServiceAccount;
//# sourceMappingURL=data:application/json;base64,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