"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = exports.KubernetesVersion = exports.EndpointAccess = void 0;
const fs = require("fs");
const path = require("path");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const YAML = require("yaml");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const k8s_manifest_1 = require("./k8s-manifest");
const k8s_object_value_1 = require("./k8s-object-value");
const k8s_patch_1 = require("./k8s-patch");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * Endpoint access characteristics.
 */
class EndpointAccess {
    constructor(
    /**
     * Configuration properties.
     *
     * @internal
     */
    _config) {
        this._config = _config;
        if (!_config.publicAccess && _config.publicCidrs && _config.publicCidrs.length > 0) {
            throw new Error('CIDR blocks can only be configured when public access is enabled');
        }
    }
    /**
     * Restrict public access to specific CIDR blocks.
     * If public access is disabled, this method will result in an error.
     *
     * @param cidr CIDR blocks.
     */
    onlyFrom(...cidr) {
        return new EndpointAccess({
            ...this._config,
            // override CIDR
            publicCidrs: cidr,
        });
    }
}
exports.EndpointAccess = EndpointAccess;
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic will leave your VPC to connect to the endpoint.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC = new EndpointAccess({ privateAccess: false, publicAccess: true });
/**
 * The cluster endpoint is only accessible through your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 */
EndpointAccess.PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: false });
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC_AND_PRIVATE.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC_AND_PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: true });
/**
 * Kubernetes cluster version
 */
class KubernetesVersion {
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom cluster version
     * @param version custom version number
     */
    static of(version) { return new KubernetesVersion(version); }
}
exports.KubernetesVersion = KubernetesVersion;
/**
 * Kubernetes version 1.14
 */
KubernetesVersion.V1_14 = KubernetesVersion.of('1.14');
/**
 * Kubernetes version 1.15
 */
KubernetesVersion.V1_15 = KubernetesVersion.of('1.15');
/**
 * Kubernetes version 1.16
 */
KubernetesVersion.V1_16 = KubernetesVersion.of('1.16');
/**
 * Kubernetes version 1.17
 */
KubernetesVersion.V1_17 = KubernetesVersion.of('1.17');
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        if (props.kubectlEnabled === false) {
            throw new Error('The "eks.Cluster" class no longer allows disabling kubectl support. ' +
                'As a temporary workaround, you can use the drop-in replacement class `eks.LegacyCluster`, ' +
                'but bear in mind that this class will soon be removed and will no longer receive additional ' +
                'features or bugfixes. See https://github.com/aws/aws-cdk/issues/9332 for more details');
        }
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        this.vpcSubnets = (_a = props.vpcSubnets) !== null && _a !== void 0 ? _a : [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        // Get subnetIds for all selected subnets
        const subnetIds = [...new Set(Array().concat(...this.vpcSubnets.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
            ...(props.secretsEncryptionKey ? {
                encryptionConfig: [{
                        provider: {
                            keyArn: props.secretsEncryptionKey.keyArn,
                        },
                        resources: ['secrets'],
                    }],
            } : {}),
        };
        this.endpointAccess = (_b = props.endpointAccess) !== null && _b !== void 0 ? _b : EndpointAccess.PUBLIC_AND_PRIVATE;
        this.kubectlProviderEnv = props.kubectlEnvironment;
        if (this.endpointAccess._config.privateAccess && this.vpc instanceof ec2.Vpc) {
            // validate VPC properties according to: https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html
            if (!this.vpc.dnsHostnamesEnabled || !this.vpc.dnsSupportEnabled) {
                throw new Error('Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled. Use `enableDnsHostnames: true` and `enableDnsSupport: true` when creating the VPC.');
            }
        }
        this.kubctlProviderSecurityGroup = new ec2.SecurityGroup(this, 'KubectlProviderSecurityGroup', {
            vpc: this.vpc,
            description: 'Comminication between KubectlProvider and EKS Control Plane',
        });
        // grant the kubectl provider access to the cluster control plane.
        this.connections.allowFrom(this.kubctlProviderSecurityGroup, this.connections.defaultPort);
        const resource = this._clusterResource = new cluster_resource_1.ClusterResource(this, 'Resource', {
            ...clusterProps,
            endpointPrivateAccess: this.endpointAccess._config.privateAccess,
            endpointPublicAccess: this.endpointAccess._config.publicAccess,
            publicAccessCidrs: this.endpointAccess._config.publicCidrs,
        });
        // the security group and vpc must exist in order to properly delete the cluster (since we run `kubectl delete`).
        // this ensures that.
        this._clusterResource.node.addDependency(this.kubctlProviderSecurityGroup, this.vpc);
        // see https://github.com/aws/aws-cdk/issues/9027
        this._clusterResource.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['ec2:DescribeVpcs'],
            resources: [stack.formatArn({
                    service: 'ec2',
                    resource: 'vpc',
                    resourceName: this.vpc.vpcId,
                })],
        }));
        // grant cluster creation role sufficient permission to access the specified key
        // see https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html
        if (props.secretsEncryptionKey) {
            this._clusterResource.creationRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:DescribeKey',
                    'kms:CreateGrant',
                ],
                resources: [props.secretsEncryptionKey.keyArn],
            }));
        }
        // we use an SSM parameter as a barrier because it's free and fast.
        this._kubectlReadyBarrier = new core_1.CfnResource(this, 'KubectlReadyBarrier', {
            type: 'AWS::SSM::Parameter',
            properties: {
                Type: 'String',
                Value: 'aws:cdk:eks:kubectl-ready',
            },
        });
        // add the cluster resource itself as a dependency of the barrier
        this._kubectlReadyBarrier.node.addDependency(this._clusterResource);
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // if an explicit role is not configured, define a masters role that can
        // be assumed by anyone in the account (with sts:AssumeRole permissions of
        // course)
        const mastersRole = (_c = props.mastersRole) !== null && _c !== void 0 ? _c : new iam.Role(this, 'MastersRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // map the IAM role to the `system:masters` group.
        this.awsAuth.addMastersRole(mastersRole);
        if (props.outputMastersRoleArn) {
            new core_1.CfnOutput(this, 'MastersRoleArn', { value: mastersRole.roleArn });
        }
        commonCommandOptions.push(`--role-arn ${mastersRole.roleArn}`);
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroup('DefaultCapacity', { instanceType, minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        this.defineCoreDnsComputeType((_d = props.coreDnsComputeType) !== null && _d !== void 0 ? _d : CoreDnsComputeType.EC2);
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Fetch the load balancer address of a service of type 'LoadBalancer'.
     *
     * @param serviceName The name of the service.
     * @param options Additional operation options.
     */
    getServiceLoadBalancerAddress(serviceName, options = {}) {
        const loadBalancerAddress = new k8s_object_value_1.KubernetesObjectValue(this, `${serviceName}LoadBalancerAddress`, {
            cluster: this,
            objectType: 'service',
            objectName: serviceName,
            objectNamespace: options.namespace,
            jsonPath: '.status.loadBalancer.ingress[0].hostname',
            timeout: options.timeout,
        });
        return loadBalancerAddress.value;
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Note that if you specify `updateType: RollingUpdate` or `updateType: ReplacingUpdate`, your nodes might be replaced at deploy
     * time without notice in case the recommended AMI for your machine image type has been updated by AWS.
     * The default behavior for `updateType` is `None`, which means only new instances will be launched using the new AMI.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * In addition, the [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id, options) {
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new BottleRocketImage() :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
            updateType: options.updateType,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id, options) {
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tags.of(autoScalingGroup).add(`kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
        });
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice) {
            this.addSpotInterruptHandler();
        }
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuerUrl() {
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuer() {
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used
     * to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     */
    get openIdConnectProvider() {
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new iam.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
                clientIds: ['sts.amazonaws.com'],
                /**
                 * For some reason EKS isn't validating the root certificate but a intermediat certificate
                 * which is one level up in the tree. Because of the a constant thumbprint value has to be
                 * stated with this OpenID Connect provider. The certificate thumbprint is the same for all the regions.
                 */
                thumbprints: ['9e99a48a9960b14926bb7f3b02e22da2b0ab7280'],
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * Defines a Kubernetes manifest in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesManifest` object.
     */
    addManifest(id, ...manifest) {
        return new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     */
    addChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Adds a service account to this cluster.
     *
     * @param id the id of this service account
     * @param options service account options
     */
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Returns the role ARN for the cluster creation role for kubectl-enabled
     * clusters.
     * @param assumedBy The IAM that will assume this role. If omitted, the
     * creation role will be returned witout modification of its trust policy.
     *
     * @internal
     */
    get _kubectlCreationRole() {
        return this._clusterResource.creationRole;
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        // add all profiles as a dependency of the "kubectl-ready" barrier because all kubectl-
        // resources can only be deployed after all fargate profiles are created.
        if (this._kubectlReadyBarrier) {
            this._kubectlReadyBarrier.node.addDependency(fargateProfile);
        }
        return this._fargateProfiles;
    }
    /**
     * Adds a resource scope that requires `kubectl` to this cluster and returns
     * the `KubectlProvider` which is the custom resource provider that should be
     * used as the resource provider.
     *
     * Called from `HelmResource` and `KubernetesResource`
     *
     * @property resourceScope the construct scope in which kubectl resources are defined.
     *
     * @internal
     */
    _attachKubectlResourceScope(resourceScope) {
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        // singleton
        let provider = this.stack.node.tryFindChild(uid);
        if (!provider) {
            // create the provider.
            let providerProps = {
                env: this.kubectlProviderEnv,
            };
            if (!this.endpointAccess._config.publicAccess) {
                const privateSubents = this.selectPrivateSubnets().slice(0, 16);
                if (privateSubents.length === 0) {
                    throw new Error('Vpc must contain private subnets to configure private endpoint access');
                }
                // endpoint access is private only, we need to attach the
                // provider to the VPC so that it can access the cluster.
                providerProps = {
                    ...providerProps,
                    vpc: this.vpc,
                    // lambda can only be accociated with max 16 subnets and they all need to be private.
                    vpcSubnets: { subnets: privateSubents },
                    securityGroups: [this.kubctlProviderSecurityGroup],
                };
            }
            provider = new kubectl_provider_1.KubectlProvider(this.stack, uid, providerProps);
        }
        // allow the kubectl provider to assume the cluster creation role.
        this._clusterResource.addTrustedRole(provider.role);
        if (!this._kubectlReadyBarrier) {
            throw new Error('unexpected: kubectl enabled clusters should have a kubectl-ready barrier resource');
        }
        resourceScope.node.addDependency(this._kubectlReadyBarrier);
        return provider;
    }
    selectPrivateSubnets() {
        const privateSubnets = [];
        for (const placement of this.vpcSubnets) {
            for (const subnet of this.vpc.selectSubnets(placement).subnets) {
                if (this.vpc.privateSubnets.includes(subnet)) {
                    // definitely private, take it.
                    privateSubnets.push(subnet);
                    continue;
                }
                if (this.vpc.publicSubnets.includes(subnet)) {
                    // definitely public, skip it.
                    continue;
                }
                // neither public and nor private - what is it then? this means its a subnet instance that was explicitly passed
                // in the subnet selection. since ISubnet doesn't contain information on type, we have to assume its private and let it
                // fail at deploy time :\ (its better than filtering it out and preventing a possibly successful deployment)
                privateSubnets.push(subnet);
            }
        }
        return privateSubnets;
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.7.3',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    'nodeSelector.lifecycle': user_data_1.LifecycleLabel.SPOT,
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addManifest('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    this.node.addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                core_1.Tags.of(subnet).add(tag, '1');
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.connections = new ec2.Connections();
        this.vpc = ec2.Vpc.fromVpcAttributes(this, 'VPC', props.vpc);
        this.clusterName = props.clusterName;
        this.clusterEndpoint = props.clusterEndpoint;
        this.clusterArn = props.clusterArn;
        this.clusterCertificateAuthorityData = props.clusterCertificateAuthorityData;
        this.clusterSecurityGroupId = props.clusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = props.clusterEncryptionConfigKeyArn;
        let i = 1;
        for (const sgProps of props.securityGroups) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgProps.securityGroupId));
            i++;
        }
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props = {}) {
        var _a, _b;
        this.nodeType = (_a = props.nodeType) !== null && _a !== void 0 ? _a : NodeType.STANDARD;
        this.kubernetesVersion = (_b = props.kubernetesVersion) !== null && _b !== void 0 ? _b : LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? 'amazon-linux-2/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
/**
 * Construct an Bottlerocket image from the latest AMI published in SSM
 */
class BottleRocketImage {
    /**
     * Constructs a new instance of the BottleRocketImage class.
     */
    constructor() {
        // only 1.15 is currently available
        this.kubernetesVersion = '1.15';
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/bottlerocket/aws-k8s-${this.kubernetesVersion}/x86_64/latest/image_id`;
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.custom(''),
        };
    }
}
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
    /**
     * Inferentia instances
     */
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    /**
     * managed node group
     */
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    /**
     * EC2 autoscaling group
     */
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon EKS-optimized Linux AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
const INFERENTIA_INSTANCETYPES = ['inf1'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        INFERENTIA_INSTANCETYPES.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
//# sourceMappingURL=data:application/json;base64,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