import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import * as sfn from '@aws-cdk/aws-stepfunctions';
import { Construct } from 'constructs';
/**
 * An entry to be sent to EventBridge.
 *
 * @see https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEventsRequestEntry.html
 * @stability stable
 */
export interface EventBridgePutEventsEntry {
    /**
     * The event body.
     *
     * Can either be provided as an object or as a JSON-serialized string
     *
     * @stability stable
     * @example
     *
     * sfn.TaskInput.fromText('{"instance-id": "i-1234567890abcdef0", "state": "terminated"}')
     * sfn.TaskInput.fromObject({ Message: 'Hello from Step Functions' })
     * sfn.TaskInput.fromJsonPathAt('$.EventDetail')
     */
    readonly detail: sfn.TaskInput;
    /**
     * Used along with the source field to help identify the fields and values expected in the detail field.
     *
     * For example, events by CloudTrail have detail type "AWS API Call via CloudTrail"
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-events.html
     * @stability stable
     */
    readonly detailType: string;
    /**
     * The event bus the entry will be sent to.
     *
     * @default - event is sent to account's default event bus
     * @stability stable
     */
    readonly eventBus?: events.IEventBus;
    /**
     * The service or application that caused this event to be generated.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-events.html
     * @stability stable
     * @example
     *
     * 'com.example.service'
     */
    readonly source: string;
}
/**
 * Properties for sending events with PutEvents.
 *
 * @stability stable
 */
export interface EventBridgePutEventsProps extends sfn.TaskStateBaseProps {
    /**
     * The entries that will be sent (must be at least 1).
     *
     * @stability stable
     */
    readonly entries: EventBridgePutEventsEntry[];
}
/**
 * A StepFunctions Task to send events to an EventBridge event bus.
 *
 * @stability stable
 */
export declare class EventBridgePutEvents extends sfn.TaskStateBase {
    private readonly props;
    private static readonly SUPPORTED_INTEGRATION_PATTERNS;
    /**
     * @stability stable
     */
    protected readonly taskMetrics?: sfn.TaskMetricsConfig;
    /**
     * @stability stable
     */
    protected readonly taskPolicies?: iam.PolicyStatement[];
    private readonly integrationPattern;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: EventBridgePutEventsProps);
    /**
     * Returns an array of EventBusArn strings based on this.props.entries
     */
    private get eventBusArns();
    /**
     * Provides the EventBridge put events service integration task configuration
     * @internal
     */
    protected _renderTask(): any;
    private renderEntries;
    private validateEntries;
}
