from __future__ import annotations

import dataclasses
import os
import string
import textwrap
from typing import Literal, Union

import black
import inflection
import isort.settings

from zinc.api import codegen_api as api


@dataclasses.dataclass(frozen=True)
class ModuleImport:
    module: str
    type: Literal["module"] = "module"


@dataclasses.dataclass(frozen=True)
class FromImport:
    module: str
    field: str
    type: Literal["from"] = "from"


Import = Union[ModuleImport, FromImport]


def get_isort_config() -> isort.Config:
    return isort.settings.Config(profile="black", line_length=100)


def generate_objects(specs: list[api.ClassSpec]) -> str:
    imports: set[Import] = set()
    after_import_output = ""
    specs_by_name = {spec.name: spec for spec in specs}
    # Need to put alias and union types at the end of the file since they may reference other types in the global scope
    for spec in sorted(
        specs, key=lambda spec: (spec.type == "union" or spec.type == "alias", spec.name)
    ):
        if spec.type == "object":
            output, more_imports = generate_complex_object(spec, specs_by_name)
        elif spec.type == "enum":
            output, more_imports = generate_enum(spec)
        elif spec.type == "union":
            output, more_imports = generate_union(spec)
        elif spec.type == "alias":
            output, more_imports = generate_alias(spec)
        else:
            raise Exception(f"Unexpected class spec type: {spec.type}")
        imports.update(more_imports)
        after_import_output += output + "\n\n"

    output = "# Generated by codegen.py\n"
    output += "from __future__ import annotations\n\n"
    output += "from zinc.runtime import codegen_runtime as cr\n\n"
    for import_ in sorted(imports, key=lambda key: (key.type == "module", key.module)):
        if import_.type == "module":
            output += f"import {import_.module}\n"
        else:
            output += f"from {import_.module} import {import_.field}\n"
    output += "\n\n"
    output += after_import_output

    return isort.api.sort_code_string(
        black.format_str(output, mode=black.Mode(line_length=100)), config=get_isort_config()
    )


def generate_complex_object(
    spec: api.ObjectSpec, specs_by_name: dict[api.ObjectTypeName, api.ClassSpec]
) -> tuple[str, set[Import]]:
    imports: set[Import] = set()
    output = fix_indent(
        f"""\
    @dataclasses.dataclass(frozen=True)
    class {spec.name}:
    """,
        0,
    )
    output += _description_as_docstring(spec.description, True, 4)
    imports.add(ModuleImport("dataclasses"))
    sorted_fields = sorted(
        spec.fields.items(), key=lambda entry: compare_field_spec(entry[0], entry[1])
    )
    for field_name, field_spec in sorted_fields:
        # TODO(markelliot): we don't allow lists and dicts to contain optionals,
        # but we should probably enforce that somewhere reasonable rather than
        # here in the type system.
        if field_spec.field_type == "literal":
            imports.add(FromImport("typing", "Literal"))
            output += fix_indent(
                f"""{to_python(field_name)}: Literal["{field_spec.value}"] = "{field_spec.value}"\n""",
                4,
            )
        else:
            real_type, more_imports = _as_real_type(field_spec)
            imports.update(more_imports)
            output += fix_indent(
                f"""{to_python(field_name)}: {real_type}{"" if field_spec.required else " | None"}{generate_field_initializer(field_spec)}\n""",
                4,
            )
        output += _description_as_docstring(field_spec.description, False, 4)
    output += "\n"
    output += fix_indent(
        f"""\
        def to_dict(self) -> dict[str, Any]:
            result: dict[str, Any] = {"{"}{"}"}
        """,
        4,
    )
    imports.add(FromImport("typing", "Any"))
    for field_name, field_spec in sorted_fields:
        if field_spec.required:
            output += fix_indent(
                f"""\
                result["{field_name}"] = {generate_field_serializer("self." + to_python(field_name), field_spec, specs_by_name)}
                """,
                8,
            )
        else:
            output += fix_indent(
                f"""\
            if self.{to_python(field_name)} is not None:
                result["{field_name}"] = {generate_field_serializer("self." + to_python(field_name), field_spec, specs_by_name)}
            """,
                8,
            )

    output += fix_indent("return result\n", 8)

    output += fix_indent(
        f"""\
        @classmethod
        def from_dict(
            cls, data: dict[str, Any], *, ignore_unknown_properties: bool = False
        ) -> cr.Result[{spec.name}, cr.SerDeError]:
            errors = set()
        """,
        4,
    )

    for field_name, field_spec in sorted_fields:
        output += fix_indent(
            f"""
            __{to_python(field_name)} = {generate_field_deserializer(field_name, field_spec, specs_by_name).lstrip()}
            if __{to_python(field_name)}.type == "error":
                errors.add(__{to_python(field_name)}.error)
            """,
            8,
        )

    props_list = ",".join(['"' + k + '"' for k, v in sorted_fields])
    assignments = ",".join(
        [
            f"{to_python(k)}=cast(cr.Ok, __{to_python(k)}).value"
            for k, v in sorted_fields
            if v.field_type != "literal"
        ]
    )
    imports.add(FromImport("typing", "cast"))

    output += fix_indent(
        f"""
        if len(errors) > 0:
            return cr.Error(cr.SerDeError(locator=cr.ObjectName("{spec.name}"), problem=frozenset(errors)))
        
        if not ignore_unknown_properties:
            unknown_props = data.keys() - {"{"}
                {props_list}
            {"}"}
            if len(unknown_props) > 0:
                return cr.Error(cr.SerDeError(
                    locator=cr.ObjectName("{spec.name}"),
                    problem=f"Unexpected extra properties {"{"}unknown_props{"}"}",
                ))

        return cr.Ok(
            {spec.name}(
                {assignments}
            ))
        """,
        8,
    )

    return output, imports


def generate_enum(spec: api.EnumSpec) -> tuple[str, set[Import]]:
    imports: set[Import] = set()
    output = fix_indent(
        f"""\
        class {spec.name}(enum.Enum):
        """,
        0,
    )
    output += _description_as_docstring(spec.description, True, 4)
    imports.add(ModuleImport("enum"))
    for value in spec.values:
        output += fix_indent(
            f"""\
                {to_python_identifier(value)} = "{value}"
            """,
            4,
        )
    output += "\n"

    output += fix_indent(
        f"""\
            @classmethod
            def from_str(cls, data: str) -> cr.Result[{spec.name}, cr.SerDeError]:
                normalized_data = data.upper()
                for variant in {spec.name}.__members__.values():
                    if variant.value.upper() == normalized_data:
                        return cr.Ok(variant)
                return cr.Error(cr.SerDeError(
                    locator=cr.ObjectName("{spec.name}"),
                    problem=f"Unexpected value {"{"}data{"}"}. Value must be one of '{"{"}{spec.name}.__members__.values(){"}"}",
                ))
            """,
        4,
    )
    return output, imports


def generate_union(spec: api.UnionSpec) -> tuple[str, set[Import]]:
    imports: set[Import] = set()
    output = fix_indent(
        f"""\
    {spec.name}Type = Union[{",".join(sorted(spec.union.values()))}]
    
    @dataclasses.dataclass(frozen=True)
    class {spec.name}:
        value: {spec.name}Type
    """,
        0,
    )
    imports.add(ModuleImport("dataclasses"))
    imports.add(FromImport("typing", "Union"))
    imports.add(FromImport("typing", "Any"))

    output += fix_indent(
        """\
        def to_dict(self) -> dict[str, Any]:
            return self.value.to_dict()
        """,
        4,
    )

    output += fix_indent(
        f"""
        @classmethod
        def from_dict(
            cls, data: dict[str, Any], *, ignore_unknown_properties: bool = False
        ) -> cr.Result[{spec.name}, cr.SerDeError]:
            __{spec.discriminator_field} = cr.get_field(data, "{spec.discriminator_field}", cr.get_string)
            if __{spec.discriminator_field}.type == "error":
                return cr.Error(cr.SerDeError(locator=cr.ObjectName("{spec.name}"), problem=frozenset((__{spec.discriminator_field}.error,))))
    """,
        4,
    )

    for key, variant in sorted(spec.union.items(), key=lambda entry: entry[0]):
        output += fix_indent(
            f"""\
        elif __{spec.discriminator_field}.value == "{key}":
            return cr.map_result({variant}.from_dict(data, ignore_unknown_properties=ignore_unknown_properties), {spec.name})
        """,
            8,
        )

    escape = '"""'
    output += fix_indent(
        f"""\
        else:
            return cr.Error(
                cr.SerDeError(
                    locator=cr.ObjectName("{spec.name}"), 
                    problem=f{escape}Unexpected value "{"{"}__{spec.discriminator_field}.value{"}"}" for "type\\"{escape}
                )
            )
    """,
        8,
    )
    return output, imports


def generate_alias(alias: api.AliasSpec) -> tuple[str, set[Import]]:
    type_name, imports = _as_real_type(alias.field)
    output = f"""{alias.name} = NewType("{alias.name}", {type_name})"""
    imports.add(FromImport("typing", "NewType"))
    # TODO: how do we do docstrings on root fields?
    # output += _description_as_docstring(alias.description, True, 0)
    return output, imports


def generate_field_initializer(field_spec: api.FieldSpec) -> str:
    if field_spec.field_type == "list":
        return " = dataclasses.field(default_factory=lambda: [])"
    elif field_spec.field_type == "dict":
        return " = dataclasses.field(default_factory=lambda: {})"
    elif not field_spec.required:
        return " = None"
    return ""


def generate_field_serializer(
    locator: str,
    field_spec: api.FieldSpec,
    specs_by_name: dict[api.ObjectTypeName, api.ClassSpec],
) -> str:
    if field_spec.field_type == "object":
        object_spec = specs_by_name[field_spec.object_type]
        if object_spec.type == "object" or object_spec.type == "union":
            return f"{locator}.to_dict()"
        elif object_spec.type == "enum":
            return f"{locator}.value"
        elif object_spec.type == "alias":
            return generate_field_serializer(locator, object_spec.field, specs_by_name)
    elif field_spec.field_type == "list":
        return f"""[{generate_field_serializer("value", field_spec.element_type, specs_by_name)} for value in {locator}]"""
    elif field_spec.field_type == "dict":
        return f"""{"{"}key: {generate_field_serializer("value", field_spec.value_type, specs_by_name)} for key, value in {locator}.items(){"}"}"""
    elif field_spec.field_type == "uuid":
        return f"str({locator})"
    elif field_spec.field_type == "date" or field_spec.field_type == "datetime":
        return f"{locator}.isoformat()"

    return locator


def generate_field_deserializer(
    field_name: str,
    field_spec: api.FieldSpec,
    specs_by_name: dict[api.ObjectTypeName, api.ClassSpec],
) -> str:
    if field_spec.required:
        output = f"""cr.get_field(data, "{field_name}", """
        if field_spec.field_type == "list" or field_spec.field_type == "dict":
            output += "nullable=True, "
    else:
        output = f"""cr.get_optional_field(data, "{field_name}", """
    return output + "delegate=" + generate_value_deserializer(0, field_spec, specs_by_name) + ")"


def generate_value_deserializer(
    depth: int,
    field_spec: api.FieldSpec,
    specs_by_name: dict[api.ObjectTypeName, api.ClassSpec],
) -> str:
    cur_var = f"var{depth}"
    # TODO(forozco): use arg list instead of string concatenation
    if field_spec.field_type == "object":
        object_spec = specs_by_name[field_spec.object_type]
        if object_spec.type == "object" or object_spec.type == "union":
            return f"lambda {cur_var}: cr.get_object({cur_var}, object_factory={field_spec.object_type}.from_dict, ignore_unknown_properties=ignore_unknown_properties)"
        elif object_spec.type == "enum":
            return f"lambda {cur_var}: cr.get_enum({cur_var}, enum_factory={field_spec.object_type}.from_str)"
        elif object_spec.type == "alias":
            return generate_value_deserializer(depth, object_spec.field, specs_by_name)
    elif field_spec.field_type == "list":
        args = ""
        if field_spec.min_count is not None:
            args += f", min_count={field_spec.min_count}"
        if field_spec.max_count is not None:
            args += f", max_count={field_spec.max_count}"
        if field_spec.allow_single_value_as_array:
            args += ", allow_single_value=True"

        return f"lambda {cur_var}: cr.get_list({cur_var}{args}, element_deser={generate_value_deserializer(depth + 1, field_spec.element_type, specs_by_name)})"
    elif field_spec.field_type == "dict":
        args = ""
        if field_spec.min_count is not None:
            args += f", min_count={field_spec.min_count}"
        if field_spec.max_count is not None:
            args += f", max_count={field_spec.max_count}"
        return f"lambda {cur_var}: cr.get_dict({cur_var}{args}, value_deser={generate_value_deserializer(depth + 1, field_spec.value_type, specs_by_name)})"
    elif field_spec.field_type == "literal":
        return f"""lambda {cur_var}: cr.check_string_literal({cur_var}, literal="{field_spec.value}")"""
    elif field_spec.field_type == "int" or field_spec.field_type == "number":
        args = ""
        if field_spec.minimum is not None:
            if field_spec.exclusive_min:
                args += f", exclusive_min={field_spec.minimum}"
            else:
                args += f", inclusive_min={field_spec.minimum}"

        if field_spec.maximum is not None:
            if field_spec.exclusive_max:
                args += f", exclusive_max={field_spec.maximum}"
            else:
                args += f", inclusive_max={field_spec.maximum}"

        if field_spec.field_type == "int" and field_spec.allow_number:
            args += ", allow_number=True"

        if args:
            return f"lambda {cur_var}: cr.get_{field_spec.field_type}({cur_var}{args})"
    return f"cr.get_{field_spec.field_type}"


def _as_real_type(v: api.FieldSpec) -> tuple[str, set[Import]]:
    if v.field_type == "int":
        return "int", set()
    elif v.field_type == "number":
        return "decimal.Decimal", {ModuleImport("decimal")}
    elif v.field_type == "datetime":
        return "datetime.datetime", {ModuleImport("datetime")}
    elif v.field_type == "date":
        return "datetime.date", {ModuleImport("datetime")}
    elif v.field_type == "string":
        return "str", set()
    elif v.field_type == "uuid":
        return "uuid.UUID", {ModuleImport("uuid")}
    elif v.field_type == "boolean":
        return "bool", set()
    elif v.field_type == "list":
        inner, imports = _as_real_type(v.element_type)
        return f"list[{inner}]", imports
    elif v.field_type == "dict":
        inner, imports = _as_real_type(v.value_type)
        return f"dict[str, {inner}]", imports
    elif v.field_type == "object":
        return v.object_type, set()
    raise Exception("Should never happen")


def to_python(field_name: str) -> str:
    return inflection.underscore(field_name)


def to_python_identifier(value: str) -> str:
    for special_char in string.punctuation + string.whitespace:
        value = value.replace(special_char, "_")
    if value[0] in string.digits:
        value = "_" + value
    return value


def fix_indent(lines: str, indent: int) -> str:
    return textwrap.indent(textwrap.dedent(lines), " " * indent)


def compare_field_spec(field_name: str, field_spec: api.FieldSpec) -> tuple[bool, str]:
    has_default_value = (
        not field_spec.required
        or field_spec.field_type == "literal"
        or field_spec.field_type == "dict"
        or field_spec.field_type == "list"
    )
    return has_default_value, field_name


def _description_as_docstring(
    description: str | None, quotes_on_own_line: bool, indent: int
) -> str:
    if description is None or len(description) == 0:
        return ""

    quote_lb = "\n" if quotes_on_own_line else ""
    output = '"""' + quote_lb
    if description.endswith('"'):
        output += description[:-1] + '\\"'
    else:
        output += description
    output += quote_lb + '"""\n'
    return fix_indent(output, indent)
