"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchMetric = exports.metricPeriod = exports.metricKey = void 0;
const core_1 = require("@aws-cdk/core");
const METRICKEY_SYMBOL = Symbol('@aws-cdk/aws-cloudwatch.MetricKey');
/**
 * Return a unique string representation for this metric.
 *
 * Can be used to determine as a hash key to determine if 2 Metric objects
 * represent the same metric. Excludes rendering properties.
 */
function metricKey(metric) {
    // Cache on the object itself. This is safe because Metric objects are immutable.
    if (metric.hasOwnProperty(METRICKEY_SYMBOL)) {
        return metric[METRICKEY_SYMBOL];
    }
    const parts = new Array();
    const conf = metric.toMetricConfig();
    if (conf.mathExpression) {
        parts.push(conf.mathExpression.expression);
        for (const id of Object.keys(conf.mathExpression.usingMetrics).sort()) {
            parts.push(id);
            parts.push(metricKey(conf.mathExpression.usingMetrics[id]));
        }
    }
    if (conf.metricStat) {
        parts.push(conf.metricStat.namespace);
        parts.push(conf.metricStat.metricName);
        for (const dim of conf.metricStat.dimensions || []) {
            parts.push(dim.name);
            parts.push(dim.value);
        }
        if (conf.metricStat.statistic) {
            parts.push(conf.metricStat.statistic);
        }
        if (conf.metricStat.period) {
            parts.push(`${conf.metricStat.period.toSeconds()}`);
        }
        if (conf.metricStat.region) {
            parts.push(conf.metricStat.region);
        }
        if (conf.metricStat.account) {
            parts.push(conf.metricStat.account);
        }
    }
    const ret = parts.join('|');
    Object.defineProperty(metric, METRICKEY_SYMBOL, { value: ret });
    return ret;
}
exports.metricKey = metricKey;
/**
 * Return the period of a metric
 *
 * For a stat metric, return the immediate period.
 *
 * For an expression metric, all metrics used in it have been made to have the
 * same period, so we return the period of the first inner metric.
 */
function metricPeriod(metric) {
    return dispatchMetric(metric, {
        withStat(stat) {
            return stat.period;
        },
        withExpression() {
            return metric.period || core_1.Duration.minutes(5);
        },
    });
}
exports.metricPeriod = metricPeriod;
/**
 * Given a metric object, inspect it and call the correct function for the type of output
 *
 * In addition to the metric object itself, takes a callback object with two
 * methods, to be invoked for the particular type of metric.
 *
 * If the metric represent a metric query (nominally generated through an
 * instantiation of `Metric` but can be generated by any class that implements
 * `IMetric`) a particular field in its `toMetricConfig()` output will be set
 * (to wit, `metricStat`) and the `withStat()` callback will be called with
 * that object.
 *
 * If the metric represents an expression (usually by instantiating `MathExpression`
 * but users can implement `IMetric` arbitrarily) the `mathExpression` field
 * will be set in the object returned from `toMetricConfig` and the callback
 * called `withExpression` will be applied to that object.
 *
 * Will return the values returned by the callbacks.
 *
 * To be used as such:
 *
 * ```ts
 * const ret = dispatchMetric(someMetric, {
 *   withStat(stat) {
 *     // do something with stat
 *     return 1;
 *   },
 *   withExpression(expr) {
 *     // do something with expr
 *     return 2;
 *   },
 * });
 * ```
 *
 * This function encapsulates some type analysis that would otherwise have to be
 * repeated in all places where code needs to make a distinction on the type
 * of metric object that is being passed.
 */
// tslint:disable-next-line:max-line-length
function dispatchMetric(metric, fns) {
    const conf = metric.toMetricConfig();
    if (conf.metricStat && conf.mathExpression) {
        throw new Error('Metric object must not produce both \'metricStat\' and \'mathExpression\'');
    }
    else if (conf.metricStat) {
        return fns.withStat(conf.metricStat, conf);
    }
    else if (conf.mathExpression) {
        return fns.withExpression(conf.mathExpression, conf);
    }
    else {
        throw new Error('Metric object must have either \'metricStat\' or \'mathExpression\'');
    }
}
exports.dispatchMetric = dispatchMetric;
//# sourceMappingURL=data:application/json;base64,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