"""
h2o2_mk2012_ad.py

Hydrogen peroxide, H2O2, ground state surface from
Ref [1]_. The coefficients are available from the references
supplementary information as the 'adiabatic PES', which 
corresponds to the "V+C+R+H+D" results.

The surface is implemented in internal coordinates.

X1 ... O1 -- H1 bond length (Angstroms)
X2 ... O2 -- H2 bond length ( "    "  )
X3 ... O1 -- O2 bond length ( "    "  )
X4 ... O2-O1-H1 bond angle (degrees)
X5 ... O1-O2-H2 bond angle ( "   " )
X6 ... dihedral angle ( "   " )

References
----------

.. [1] P. Malyszek and J. Koput. J. Comp. Chem. 34, 337-344 (2013).
       https://doi.org/10.1002/jcc.23137

"""


import nitrogen as n2
import nitrogen.autodiff.forward as adf
import numpy as np 


def Vfun(X, deriv = 0, out = None, var = None):
    """
    expected order : r1, r2, R, a1, a2, tau
    """
    x = n2.dfun.X2adf(X, deriv, var)
    
    r1 = x[0]
    r2 = x[1]
    R = x[2]
    a1 = x[3]
    a2 = x[4]
    tau = x[5]
    
    # Define reference values 
    Re = 1.45538654 # Angstroms
    re = 0.96257063 # Angstroms
    ae = 101.08307909 # degrees 
    
    q1 = (r1 - re) / r1  # Simons-Parr-Finlan coordinates
    q2 = (r2 - re) / r2 
    q3 = (R - Re) / R 
    q4 = (a1 - ae) * np.pi/180.0  # radians
    q5 = (a2 - ae) * np.pi/180.0  # radians
    q6 = tau * np.pi/180.0 # radians 
    
    # Calculate surface 
    
    v = calcsurf([q1,q2,q3,q4,q5,q6]) * n2.constants.Eh
  
    return n2.dfun.adf2array([v], out)

######################################
# 
# Define module-scope PES DFun object
#
PES = n2.dfun.DFun(Vfun, nf = 1, nx = 6)
#
#
######################################

def calcsurf(q):
    
    max_pow = [5,5,5,6,6,6] # max_pow[5] is really the max freq. of dihedral
    
    qpow = []
    for i in range(5):
        qi = [adf.const_like(1.0, q[i]), q[i]]
        for p in range(2,max_pow[i]+1):
            qi.append(qi[1] * qi[p-1]) # qi ** p 
        qpow.append(qi)
    
    # Calculate cos(n*q6)
    cosq = [ adf.cos(n * q[5]) for n in range(max_pow[5] + 1)]
    qpow.append(cosq) 

    v = 0.0
    
    nterms = powers.shape[0]
    
    for i in range(nterms):
        c = coeffs[i] 
        
        v += c * \
            qpow[0][powers[i,0]] * \
            qpow[1][powers[i,1]] * \
            qpow[2][powers[i,2]] * \
            qpow[3][powers[i,3]] * \
            qpow[4][powers[i,4]] * \
            qpow[5][powers[i,5]]
    
    return v



powers = np.array([
    [0, 0, 0, 0, 0, 0],
[0, 0, 0, 0, 0, 1],
[0, 0, 0, 0, 0, 2],
[0, 0, 0, 0, 0, 3],
[0, 0, 0, 0, 0, 4],
[0, 0, 0, 0, 0, 5],
[0, 0, 0, 0, 0, 6],
[0, 0, 2, 0, 0, 0],
[2, 0, 0, 0, 0, 0],
[0, 2, 0, 0, 0, 0],
[0, 0, 0, 2, 0, 0],
[0, 0, 0, 0, 2, 0],
[1, 0, 1, 0, 0, 0],
[0, 1, 1, 0, 0, 0],
[0, 0, 1, 1, 0, 0],
[0, 0, 1, 0, 1, 0],
[1, 1, 0, 0, 0, 0],
[1, 0, 0, 1, 0, 0],
[0, 1, 0, 0, 1, 0],
[1, 0, 0, 0, 1, 0],
[0, 1, 0, 1, 0, 0],
[0, 0, 0, 1, 1, 0],
[0, 0, 3, 0, 0, 0],
[3, 0, 0, 0, 0, 0],
[0, 3, 0, 0, 0, 0],
[0, 0, 0, 3, 0, 0],
[0, 0, 0, 0, 3, 0],
[1, 0, 2, 0, 0, 0],
[0, 1, 2, 0, 0, 0],
[0, 0, 2, 1, 0, 0],
[0, 0, 2, 0, 1, 0],
[2, 0, 1, 0, 0, 0],
[0, 2, 1, 0, 0, 0],
[0, 0, 1, 2, 0, 0],
[0, 0, 1, 0, 2, 0],
[1, 2, 0, 0, 0, 0],
[2, 1, 0, 0, 0, 0],
[1, 0, 0, 2, 0, 0],
[0, 1, 0, 0, 2, 0],
[2, 0, 0, 1, 0, 0],
[0, 2, 0, 0, 1, 0],
[1, 0, 0, 0, 2, 0],
[0, 1, 0, 2, 0, 0],
[2, 0, 0, 0, 1, 0],
[0, 2, 0, 1, 0, 0],
[0, 0, 0, 1, 2, 0],
[0, 0, 0, 2, 1, 0],
[1, 1, 1, 0, 0, 0],
[1, 0, 1, 1, 0, 0],
[0, 1, 1, 0, 1, 0],
[1, 0, 1, 0, 1, 0],
[0, 1, 1, 1, 0, 0],
[0, 0, 1, 1, 1, 0],
[1, 1, 0, 1, 0, 0],
[1, 1, 0, 0, 1, 0],
[1, 0, 0, 1, 1, 0],
[0, 1, 0, 1, 1, 0],
[0, 0, 4, 0, 0, 0],
[4, 0, 0, 0, 0, 0],
[0, 4, 0, 0, 0, 0],
[0, 0, 0, 4, 0, 0],
[0, 0, 0, 0, 4, 0],
[2, 0, 2, 0, 0, 0],
[0, 2, 2, 0, 0, 0],
[0, 0, 2, 2, 0, 0],
[0, 0, 2, 0, 2, 0],
[2, 2, 0, 0, 0, 0],
[2, 0, 0, 2, 0, 0],
[0, 2, 0, 0, 2, 0],
[0, 0, 0, 2, 2, 0],
[1, 0, 3, 0, 0, 0],
[0, 1, 3, 0, 0, 0],
[0, 0, 3, 1, 0, 0],
[0, 0, 3, 0, 1, 0],
[3, 0, 0, 1, 0, 0],
[0, 3, 0, 0, 1, 0],
[3, 0, 1, 0, 0, 0],
[0, 3, 1, 0, 0, 0],
[0, 0, 1, 3, 0, 0],
[0, 0, 1, 0, 3, 0],
[1, 3, 0, 0, 0, 0],
[3, 1, 0, 0, 0, 0],
[1, 0, 0, 3, 0, 0],
[0, 1, 0, 0, 3, 0],
[1, 0, 0, 0, 3, 0],
[0, 1, 0, 3, 0, 0],
[0, 0, 0, 1, 3, 0],
[0, 0, 0, 3, 1, 0],
[1, 1, 2, 0, 0, 0],
[1, 0, 2, 1, 0, 0],
[0, 1, 2, 0, 1, 0],
[1, 0, 2, 0, 1, 0],
[0, 1, 2, 1, 0, 0],
[0, 0, 2, 1, 1, 0],
[2, 0, 0, 1, 1, 0],
[0, 2, 0, 1, 1, 0],
[1, 0, 1, 2, 0, 0],
[0, 1, 1, 0, 2, 0],
[1, 0, 0, 1, 2, 0],
[0, 1, 0, 2, 1, 0],
[1, 0, 0, 2, 1, 0],
[0, 1, 0, 1, 2, 0],
[0, 0, 5, 0, 0, 0],
[5, 0, 0, 0, 0, 0],
[0, 5, 0, 0, 0, 0],
[0, 0, 0, 5, 0, 0],
[0, 0, 0, 0, 5, 0],
[0, 0, 0, 6, 0, 0],
[0, 0, 0, 0, 6, 0],
[0, 0, 0, 4, 1, 0],
[0, 0, 0, 1, 4, 0],
[0, 0, 0, 3, 2, 0],
[0, 0, 0, 2, 3, 0],
[0, 0, 1, 4, 0, 0],
[0, 0, 1, 0, 4, 0],
[0, 0, 2, 3, 0, 0],
[0, 0, 2, 0, 3, 0],
[1, 0, 0, 4, 0, 0],
[0, 1, 0, 0, 4, 0],
[2, 0, 0, 3, 0, 0],
[0, 2, 0, 0, 3, 0],
[0, 0, 1, 0, 0, 1],
[1, 0, 0, 0, 0, 1],
[0, 1, 0, 0, 0, 1],
[0, 0, 0, 1, 0, 1],
[0, 0, 0, 0, 1, 1],
[0, 0, 2, 0, 0, 1],
[2, 0, 0, 0, 0, 1],
[0, 2, 0, 0, 0, 1],
[0, 0, 0, 2, 0, 1],
[0, 0, 0, 0, 2, 1],
[1, 0, 1, 0, 0, 1],
[0, 1, 1, 0, 0, 1],
[0, 0, 1, 1, 0, 1],
[0, 0, 1, 0, 1, 1],
[1, 1, 0, 0, 0, 1],
[1, 0, 0, 1, 0, 1],
[0, 1, 0, 0, 1, 1],
[1, 0, 0, 0, 1, 1],
[0, 1, 0, 1, 0, 1],
[0, 0, 0, 1, 1, 1],
[0, 0, 3, 0, 0, 1],
[3, 0, 0, 0, 0, 1],
[0, 3, 0, 0, 0, 1],
[0, 0, 0, 3, 0, 1],
[0, 0, 0, 0, 3, 1],
[1, 0, 2, 0, 0, 1],
[0, 1, 2, 0, 0, 1],
[0, 0, 2, 1, 0, 1],
[0, 0, 2, 0, 1, 1],
[0, 0, 1, 2, 0, 1],
[0, 0, 1, 0, 2, 1],
[1, 2, 0, 0, 0, 1],
[2, 1, 0, 0, 0, 1],
[1, 0, 0, 2, 0, 1],
[0, 1, 0, 0, 2, 1],
[1, 0, 0, 0, 2, 1],
[0, 1, 0, 2, 0, 1],
[0, 0, 0, 1, 2, 1],
[0, 0, 0, 2, 1, 1],
[1, 1, 1, 0, 0, 1],
[1, 0, 0, 1, 1, 1],
[0, 1, 0, 1, 1, 1],
[0, 0, 0, 4, 0, 1],
[0, 0, 0, 0, 4, 1],
[0, 0, 0, 5, 0, 1],
[0, 0, 0, 0, 5, 1],
[0, 0, 1, 3, 0, 1],
[0, 0, 1, 0, 3, 1],
[0, 0, 2, 2, 0, 1],
[0, 0, 2, 0, 2, 1],
[0, 0, 0, 1, 3, 1],
[0, 0, 0, 3, 1, 1],
[0, 0, 0, 2, 2, 1],
[1, 0, 0, 3, 0, 1],
[0, 1, 0, 0, 3, 1],
[1, 0, 0, 0, 3, 1],
[0, 1, 0, 3, 0, 1],
[2, 0, 0, 2, 0, 1],
[0, 2, 0, 0, 2, 1],
[2, 0, 0, 0, 2, 1],
[0, 2, 0, 2, 0, 1],
[1, 0, 2, 1, 0, 1],
[0, 1, 2, 0, 1, 1],
[2, 0, 1, 1, 0, 1],
[0, 2, 1, 0, 1, 1],
[1, 0, 1, 2, 0, 1],
[0, 1, 1, 0, 2, 1],
[0, 0, 1, 0, 0, 2],
[1, 0, 0, 0, 0, 2],
[0, 1, 0, 0, 0, 2],
[0, 0, 0, 1, 0, 2],
[0, 0, 0, 0, 1, 2],
[0, 0, 2, 0, 0, 2],
[2, 0, 0, 0, 0, 2],
[0, 2, 0, 0, 0, 2],
[0, 0, 0, 2, 0, 2],
[0, 0, 0, 0, 2, 2],
[1, 0, 1, 0, 0, 2],
[0, 1, 1, 0, 0, 2],
[0, 0, 1, 1, 0, 2],
[0, 0, 1, 0, 1, 2],
[1, 1, 0, 0, 0, 2],
[1, 0, 0, 1, 0, 2],
[0, 1, 0, 0, 1, 2],
[1, 0, 0, 0, 1, 2],
[0, 1, 0, 1, 0, 2],
[0, 0, 0, 1, 1, 2],
[0, 0, 3, 0, 0, 2],
[3, 0, 0, 0, 0, 2],
[0, 3, 0, 0, 0, 2],
[0, 0, 0, 3, 0, 2],
[0, 0, 0, 0, 3, 2],
[0, 0, 0, 2, 1, 2],
[0, 0, 0, 1, 2, 2],
[0, 0, 1, 2, 0, 2],
[0, 0, 1, 0, 2, 2],
[1, 0, 2, 0, 0, 2],
[0, 1, 2, 0, 0, 2],
[2, 0, 1, 0, 0, 2],
[0, 2, 1, 0, 0, 2],
[0, 0, 0, 4, 0, 2],
[0, 0, 0, 0, 4, 2],
[0, 0, 0, 1, 3, 2],
[0, 0, 0, 3, 1, 2],
[0, 0, 0, 2, 2, 2],
[2, 0, 0, 1, 0, 2],
[0, 2, 0, 0, 1, 2],
[1, 0, 0, 2, 0, 2],
[0, 1, 0, 0, 2, 2],
[1, 0, 0, 0, 2, 2],
[0, 1, 0, 2, 0, 2],
[1, 0, 1, 1, 0, 2],
[0, 1, 1, 0, 1, 2],
[1, 0, 1, 0, 1, 2],
[0, 1, 1, 1, 0, 2],
[0, 0, 1, 3, 0, 2],
[0, 0, 1, 0, 3, 2],
[0, 0, 1, 0, 0, 3],
[1, 0, 0, 0, 0, 3],
[0, 1, 0, 0, 0, 3],
[0, 0, 0, 1, 0, 3],
[0, 0, 0, 0, 1, 3],
[0, 0, 2, 0, 0, 3],
[2, 0, 0, 0, 0, 3],
[0, 2, 0, 0, 0, 3],
[0, 0, 0, 2, 0, 3],
[0, 0, 0, 0, 2, 3],
[0, 0, 0, 1, 1, 3],
[0, 0, 3, 0, 0, 3],
[0, 0, 0, 3, 0, 3],
[0, 0, 0, 0, 3, 3],
[0, 0, 0, 1, 2, 3],
[0, 0, 0, 2, 1, 3],
[0, 0, 1, 1, 0, 3],
[0, 0, 1, 0, 1, 3],
[1, 0, 0, 1, 0, 3],
[0, 1, 0, 0, 1, 3],
[1, 0, 0, 0, 1, 3],
[0, 1, 0, 1, 0, 3],
[0, 0, 2, 1, 0, 3],
[0, 0, 2, 0, 1, 3],
[0, 0, 1, 0, 0, 4],
[1, 0, 0, 0, 0, 4],
[0, 1, 0, 0, 0, 4],
[0, 0, 0, 1, 0, 4],
[0, 0, 0, 0, 1, 4],
[0, 0, 2, 0, 0, 4],
[0, 0, 0, 2, 0, 4],
[0, 0, 0, 0, 2, 4],
[0, 0, 0, 1, 1, 4],
[0, 0, 1, 1, 0, 4],
[0, 0, 1, 0, 1, 4],
[0, 0, 1, 0, 0, 5],
[1, 0, 0, 0, 0, 5],
[0, 1, 0, 0, 0, 5],
[0, 0, 0, 1, 0, 5],
[0, 0, 0, 0, 1, 5]
])
    

coeffs = np.array([
      0.00396159  ,
  0.00481490  ,
  0.00318934  ,
  0.00027018  ,
  0.00005307  ,
  0.00001047  ,
  0.00000198  ,
  1.07103383  ,
  0.85671785  ,
  0.85671785  ,
  0.11105339  ,
  0.11105339  ,
  -0.03876908 ,
  -0.03876908 ,
  0.18430247  ,
  0.18430247  ,
  0.00036727  ,
  -0.00663756 ,
  -0.00663756 ,
  -0.00196944 ,
  -0.00196944 ,
  0.01747081  ,
  -1.18343510 ,
  -0.23735539 ,
  -0.23735539 ,
  -0.02611900 ,
  -0.02611900 ,
  -0.15438002 ,
  -0.15438002 ,
  -0.35516368 ,
  -0.35516368 ,
  0.07899067  ,
  0.07899067  ,
  -0.26776532 ,
  -0.26776532 ,
  -0.00406083 ,
  -0.00406083 ,
  -0.01925971 ,
  -0.01925971 ,
  -0.01107079 ,
  -0.01107079 ,
  -0.00816282 ,
  -0.00816282 ,
  0.00337183  ,
  0.00337183  ,
  -0.01352772 ,
  -0.01352772 ,
  0.01289325  ,
  -0.07449808 ,
  -0.07449808 ,
  -0.03379136 ,
  -0.03379136 ,
  -0.01672271 ,
  -0.00495469 ,
  -0.00495469 ,
  -0.00453600 ,
  -0.00453600 ,
  -0.91033894 ,
  -0.38779590 ,
  -0.38779590 ,
  -0.00503640 ,
  -0.00503640 ,
  -0.46416302 ,
  -0.46416302 ,
  0.07527264  ,
  0.07527264  ,
  -0.00799835 ,
  -0.04029912 ,
  -0.04029912 ,
  0.00364088  ,
  0.47561739  ,
  0.47561739  ,
  -0.41647359 ,
  -0.41647359 ,
  -0.06425296 ,
  -0.06425296 ,
  0.26125142  ,
  0.26125142  ,
  0.10336257  ,
  0.10336257  ,
  -0.01680055 ,
  -0.01680055 ,
  0.04984239  ,
  0.04984239  ,
  0.00354416  ,
  0.00354416  ,
  0.00452574  ,
  0.00452574  ,
  -0.05423804 ,
  0.06564708  ,
  0.06564708  ,
  0.03801095  ,
  0.03801095  ,
  -0.09161667 ,
  -0.01589965 ,
  -0.01589965 ,
  0.01341203  ,
  0.01341203  ,
  -0.01342635 ,
  -0.01342635 ,
  -0.00671149 ,
  -0.00671149 ,
  -0.73562441 ,
  -0.30455894 ,
  -0.30455894 ,
  0.00582616  ,
  0.00582616  ,
  -0.00547701 ,
  -0.00547701 ,
  0.00280896  ,
  0.00280896  ,
  0.00674263  ,
  0.00674263  ,
  0.06845098  ,
  0.06845098  ,
  0.04193747  ,
  0.04193747  ,
  -0.05190213 ,
  -0.05190213 ,
  0.04168912  ,
  0.04168912  ,
  -0.01682379 ,
  -0.00098759 ,
  -0.00098759 ,
  -0.01176361 ,
  -0.01176361 ,
  0.01742527  ,
  -0.00533832 ,
  -0.00533832 ,
  0.00542779  ,
  0.00542779  ,
  0.00263732  ,
  0.00263732  ,
  0.01859551  ,
  0.01859551  ,
  0.00511361  ,
  -0.00973834 ,
  -0.00973834 ,
  -0.00511467 ,
  -0.00511467 ,
  -0.01356281 ,
  0.00352911  ,
  -0.00964293 ,
  -0.00964293 ,
  -0.00113452 ,
  -0.00113452 ,
  0.01028106  ,
  0.01028106  ,
  -0.03748145 ,
  -0.03748145 ,
  -0.00708628 ,
  -0.00708628 ,
  0.00742831  ,
  0.00742831  ,
  0.00419281  ,
  0.00419281  ,
  -0.00555253 ,
  -0.00555253 ,
  -0.02044897 ,
  -0.02044897 ,
  -0.02429936 ,
  0.00148383  ,
  0.00148383  ,
  0.00050075  ,
  0.00050075  ,
  0.00149142  ,
  0.00149142  ,
  0.02232416  ,
  0.02232416  ,
  0.07164353  ,
  0.07164353  ,
  0.01644870  ,
  0.01644870  ,
  0.01815537  ,
  0.01605919  ,
  0.01605919  ,
  0.00735028  ,
  0.00735028  ,
  0.02670612  ,
  0.02670612  ,
  0.01548269  ,
  0.01548269  ,
  -0.13042235 ,
  -0.13042235 ,
  0.07364926  ,
  0.07364926  ,
  -0.08874645 ,
  -0.08874645 ,
  -0.01177248 ,
  0.00172223  ,
  0.00172223  ,
  -0.00154074 ,
  -0.00154074 ,
  0.01965194  ,
  0.00409752  ,
  0.00409752  ,
  0.00301573  ,
  0.00301573  ,
  -0.00734859 ,
  -0.00734859 ,
  0.00350247  ,
  0.00350247  ,
  -0.00037121 ,
  0.00249543  ,
  0.00249543  ,
  -0.00168725 ,
  -0.00168725 ,
  0.00914785  ,
  -0.02015559 ,
  0.00925238  ,
  0.00925238  ,
  -0.00593037 ,
  -0.00593037 ,
  -0.01230679 ,
  -0.01230679 ,
  0.00829575  ,
  0.00829575  ,
  0.03735453  ,
  0.03735453  ,
  -0.04328977 ,
  -0.04328977 ,
  0.00458548  ,
  0.00458548  ,
  0.00364501  ,
  0.00364501  ,
  0.00986809  ,
  0.01437361  ,
  0.01437361  ,
  0.00072674  ,
  0.00072674  ,
  -0.00158409 ,
  -0.00158409 ,
  -0.03961996 ,
  -0.03961996 ,
  -0.01732246 ,
  -0.01732246 ,
  0.02668498  ,
  0.02668498  ,
  -0.00188286 ,
  0.00052265  ,
  0.00052265  ,
  -0.00089442 ,
  -0.00089442 ,
  0.00481644  ,
  0.00031496  ,
  0.00031496  ,
  0.00103249  ,
  0.00103249  ,
  0.00224998  ,
  -0.00366693 ,
  -0.00033429 ,
  -0.00033429 ,
  -0.00319598 ,
  -0.00319598 ,
  0.00447145  ,
  0.00447145  ,
  -0.00147544 ,
  -0.00147544 ,
  -0.00085521 ,
  -0.00085521 ,
  -0.01099915 ,
  -0.01099915 ,
  -0.00042972 ,
  0.00013538  ,
  0.00013538  ,
  -0.00019221 ,
  -0.00019221 ,
  0.00121114  ,
  0.00026755  ,
  0.00026755  ,
  0.00054596  ,
  0.00057513  ,
  0.00057513  ,
  -0.00009041 ,
  0.00002274  ,
  0.00002274  ,
  -0.00004075 ,
  -0.00004075 
    ])