from logging import getLogger
from typing import Dict, Iterator, NamedTuple, Optional, Sequence

from black import Mode, TargetVersion, format_str
from django.apps import apps
from django.db.migrations.autodetector import MigrationAutodetector
from django.db.migrations.executor import MigrationExecutor
from django.db.migrations.loader import MigrationLoader
from django.db.migrations.migration import Migration
from django.db.migrations.questioner import NonInteractiveMigrationQuestioner
from django.db.migrations.state import ProjectState
from django.db.migrations.writer import MigrationWriter
from django.utils.timezone import now
from google.cloud import bigquery

from .bigqwery import BigQwery
from .config import BaseConfigProvider, default_settings
from .extractor import Schema

logger = getLogger(__name__)


class Coconut:
    """
    Coconuts are transported by Swallows during migrations. They contain the
    new schemas to be created in BigQuery at this stage of the (Django)
    migration process.
    """

    def __init__(
        self, schemas: Sequence[Schema], settings: Optional[BaseConfigProvider] = None
    ):
        """
        Dumb init

        Parameters
        ----------
        schemas
            Schemas that have to be applied with this migration
        settings
            Settings to be used
        """

        self.schemas: Sequence[Schema] = schemas
        self.settings: BaseConfigProvider = settings if settings else default_settings
        self._bq: Optional[BigQwery] = None

    @property
    def bq(self) -> BigQwery:
        """
        Cached BigQwery client
        """

        if not self._bq:
            self._bq = BigQwery(self.settings)

        return self._bq

    def generate_django_migration(self, dependencies) -> str:
        """
        Generates the textual Python code which contains this migration. The
        strategy is to supply a simple serializable data structure (the schemas
        attribute) to the Coconut object and then let the code in this file
        take care of running the migrations. This limits greatly the amount of
        Python code to be generated (versus having all the BigQuery calls
        directly within the migrations files, which would also be possible).

        Parameters
        ----------
        dependencies
            Django migrations dependencies list, as computed by the caller
        """

        return f"""# Generated by Smolqwery on {now().isoformat()}
from django.db import migrations
from smolqwery.swallow import Coconut

class Migration(migrations.Migration):
    dependencies = {dependencies!r}

    coconut = Coconut({[*self.schemas]!r})

    operations = [migrations.RunPython(coconut.forward, coconut.backward)]
"""

    def generate_django_migration_name(self) -> str:
        """
        Generates a migration name based on the tables that are being created
        """

        return f'smolqwery_{"_".join(x["table_name"] for x in self.schemas)}'

    def list_created_tables(self) -> Iterator[str]:
        """
        Returns a list of created tables/views. This is mostly used for display
        """

        for schema in self.schemas:
            yield self.bq.get_table_id(schema["table_name"])

            if schema["delta_fields"]:
                yield self.bq.get_delta_view_id(schema["table_name"])

    def ensure_dataset(self):
        """
        Ensures that the dataset exists. The dataset isn't really part of the
        migration per se, in the sense that if you unapply a migration it will
        not be deleted. However, for convenience (and making sure that ACLs
        and storage location are set) we create the dataset if required. This
        is done for every migration even though only the first one will in
        effect created it.
        """

        dataset = bigquery.Dataset(self.bq.get_dataset_id())
        dataset.location = self.settings.dataset_location
        dataset.access_entries = [
            bigquery.AccessEntry(
                role="WRITER",
                entity_type="groupByEmail",
                entity_id=group_email,
            )
            for group_email in self.settings.acl_groups
        ]

        self.bq.client.create_dataset(dataset, exists_ok=True)

    def create_table(self, schema: Schema):
        """
        Creates the table based on the schema

        Parameters
        ----------
        schema
            Schema derived from the dataclass of the extractor
        """

        bq_schema = [
            bigquery.SchemaField(
                f["name"],
                f["type"],
                mode=("NULLABLE" if f["nullable"] else "REQUIRED"),
            )
            for f in schema["regular_fields"]
        ]
        # noinspection PyTypeChecker
        table = bigquery.Table(
            self.bq.get_table_id(schema["table_name"]), schema=bq_schema
        )
        self.bq.client.create_table(table)

    def create_delta(self, schema: Schema):
        """
        Creates the delta view, which is a view where some fields are
        differentiated from one day to the other. This allows to have for
        example a "users_created" field that gives the absolute number of users
        in the DB while the differentiated view will give how many were created
        on this day.

        Parameters
        ----------
        schema
            Schema derived from the dataclass
        """

        if not schema["delta_fields"]:
            return

        diff_fields = [
            f"t1.`{f}` - coalesce(t0.`{f}`, 0) as `{f}`" for f in schema["delta_fields"]
        ]

        query = f"""
            select
                t1.date as date,
                {','.join(diff_fields)}
            from `{self.bq.get_table_id(schema['table_name'])}` t1
            left join `{self.bq.get_table_id(schema['table_name'])}` t0
                on t1.date = date_add(t0.date, interval 1 day)
        """

        # noinspection PyTypeChecker
        view = bigquery.Table(self.bq.get_delta_view_id(schema["table_name"]))
        view.view_query = query

        self.bq.client.create_table(view)

    def drop_schema(self, schema: Schema):
        """
        For the backwards migration, drop everything we created

        Parameters
        ----------
        schema
            Schema derived from the dataclasses
        """

        if schema["delta_fields"]:
            self.bq.client.delete_table(self.bq.get_delta_view_id(schema["table_name"]))

        self.bq.client.delete_table(self.bq.get_table_id(schema["table_name"]))

    def forward(self, *_):
        """
        Runs the migration forwards
        """

        self.ensure_dataset()

        for schema in self.schemas:
            self.create_table(schema)
            self.create_delta(schema)

    def backward(self, *_):
        """
        Runs the migration backwards
        """

        for schema in self.schemas:
            self.drop_schema(schema)


class SchemaDiff(NamedTuple):
    """
    Computed diff between two schemas

    Other Parameters
    ----------------
    new_tables
        New tables to create and their schema
    changed_tables
        Name of tables that were changed (and that shouldn't)
    """

    new_tables: Sequence[Schema]
    changed_tables: Sequence[str]


class Swallow:
    """
    That's our migrations manager (swallows are migratory birds, got it?).
    Mostly we leverage Django's internal migrations management system in order
    to plug our things into it.
    """

    def __init__(self, settings: Optional[BaseConfigProvider] = None):
        self.settings: BaseConfigProvider = settings if settings else default_settings
        self.schemas: Dict[str, Schema] = {}

    def load_coconut(self, coconut: Coconut) -> None:
        """
        Adding a coconut to the known schemas. Used when successively reading
        all old migrations.

        Parameters
        ----------
        coconut
            Coconut to account
        """

        assert coconut.settings is self.settings

        for schema in coconut.schemas:
            if schema["table_name"] in self.schemas:
                raise Exception(f'Re-declaring schema "{schema["table_name"]}"')

            self.schemas[schema["table_name"]] = schema

    def list_migrations(self) -> Iterator[Migration]:
        """
        Makes a list of existing Django migrations
        """

        loader = MigrationLoader(None, ignore_no_migrations=True)
        executor = MigrationExecutor(None)

        for migration, _ in executor.migration_plan(loader.graph.leaf_nodes()):
            yield migration

    def list_coconuts(self) -> Iterator[Coconut]:
        """
        Makes a list of all coconuts embedded in existing migrations
        """

        for migration in self.list_migrations():
            if isinstance(coconut := getattr(migration, "coconut", None), Coconut):
                yield coconut

    def full_schema(self) -> Dict[str, Schema]:
        """
        Computes the currently expected full schema, based on active extractors
        """

        out = {}
        bq = BigQwery(settings=self.settings)

        for extractor_class in self.settings.get_extractors():
            extractor = extractor_class(bq, self.settings)
            schema = extractor.get_schema()

            if schema["table_name"] in out:
                raise Exception(f'Re-declaring schema "{schema["table_name"]}"')

            out[schema["table_name"]] = schema

        return out

    def diff(
        self, current: Dict[str, Schema], expected: Dict[str, Schema]
    ) -> SchemaDiff:
        """
        Makes the (interesting part of the) diff between to schemas to help
        knowing what to put in the next Coconut.

        Parameters
        ----------
        current
            Current schema observed in migrations
        expected
            Expected full schema
        """

        expected_tables = set(expected.keys())
        current_tables = set(current.keys())

        new_tables = expected_tables - current_tables
        changed_tables = []

        for test_table in expected_tables & current_tables:
            if current[test_table] != expected[test_table]:
                changed_tables.append(test_table)

        return SchemaDiff([expected[t] for t in new_tables], changed_tables)

    def diff_from_migrations(self) -> SchemaDiff:
        """
        Computes the content of the next Coconut based on current migrations
        and expected schema
        """

        for coconut in self.list_coconuts():
            self.load_coconut(coconut)

        expected = self.full_schema()
        diff = self.diff(self.schemas, expected)

        return diff

    def write_coconut_migration(self, schemas: Sequence[Schema]) -> Iterator[Migration]:
        """
        Writes a new migration with a Coconut that will apply the provided
        schema.

        That new migration's name and location is determined by Django's
        migrations system. The code here is a simplification of makemigrations
        with a bunch of overrides (notably the part where the code of the
        migration is generated, since we're doing something entirely different)

        Parameters
        ----------
        schemas
            Schemas to be included in the migration
        """

        coconut = Coconut(schemas, settings=self.settings)
        loader = MigrationLoader(None, ignore_no_migrations=True)
        questioner = NonInteractiveMigrationQuestioner(
            specified_apps=set(), dry_run=False
        )
        auto_detector = MigrationAutodetector(
            loader.project_state(),
            ProjectState.from_apps(apps),
            questioner,
        )
        changes = auto_detector.arrange_for_graph(
            changes={
                self.settings.django_app: [
                    Migration(
                        "smolqwery",
                        self.settings.django_app,
                    )
                ]
            },
            graph=loader.graph,
            migration_name=coconut.generate_django_migration_name(),
        )

        migration = changes[self.settings.django_app][0]
        code = format_str(
            coconut.generate_django_migration(migration.dependencies),
            mode=Mode(
                target_versions={TargetVersion.PY38},
            ),
        )
        writer = MigrationWriter(migration)

        with open(writer.path, "w", encoding="utf-8") as f:
            logger.info("Creating migration %s", migration)
            f.write(code)
            setattr(migration, "coconut", coconut)
            yield migration

    def make_migrations(self) -> Iterator[Migration]:
        """
        Equivalent of Django's makemigrations but for Smolqwery migrations
        """

        diff = self.diff_from_migrations()

        if diff.changed_tables:
            raise Exception(
                f'Table(s) "{", ".join(diff.changed_tables)}" was/were '
                f"changed, which is forbidden. Create new tables instead."
            )

        if diff.new_tables:
            yield from self.write_coconut_migration(diff.new_tables)
