def propose_adducts(ID, df, masses, method = "pearson", threshold = 0.90, mass_error = 0.005):
    
    """Suggests highly correlated mass traces for an input XIC without prior knowledge of identity. 
    An untargeted approach of searching for adducts and in-source fragments in a dataset. 
    Highly correlated XICs to an input features will be identified.
    Potential annotations will be given from an internal reference data base of 
    frequent adducts / fragments.
     
    Parameters
    ----------    
    ID : integer
         The index of the input feature to suggest potential adducts for.
    
    df: pd.DataFrame
        A two-dimensional DataFrame containing aligned mass spectrometric features
        e.g. generated by align_spectra().
        
    masses : pd.series 
             A series of m/z values for specification of adduct or fragment types. 
            Use of theoretic masses is strongly recommended for higher precission.
            If theoretic masses are not available, adapt mass_error 
            in accordance to the type of your mass analyzer.
        
    method : str, {'pearson’, ‘spearman’, ‘kendall’}
             Correlation method for pointwise compariation of XIC traces. 
            
    threshold : float
                Correlation treshold to associate two XIC. Default is 0.9.
               
    mass_error : int or float
                 Tolerance of the mass spectrometer in Da. Default is 0.005 Da.
                 Is used to suggest potential annotations for the candidate.
    
    Returns
    -------
    Output is a pd.DataFrame containing XIC IDs, correlation coefficients,
    mass difference of correlated ion traces and potential anootation of these. 
    
    See Also
    --------
    DBDIpy.align_spectra() : For preparation of the input data.
    DBDIpy.identify_aducts() : For identification of multiple ion species from one compound
    with prior knowledge of adduct type.
    
    """
    
    import math
    import numpy as np
    import pandas as pd
    
   
#%%check if user imput is valid
        
    if not isinstance(ID, int):
        raise TypeError("Argument ID should be an integer.")
        
    if not isinstance(threshold, float):
        raise TypeError("Invalid input for arggument treshold.")
        
    if not isinstance(mass_error, float):
        raise TypeError("Invalid input for arggument mass_error.")
        
    if not isinstance(df, pd.DataFrame):
        raise TypeError("Argument df should of instance pd.DataFrame.")  
        
    if df.isnull().values.any():
        raise ValueError("Input DataFrame contains missing values.")
        
    if ID is None:
        raise ValueError("Input DataFrame contains missing values.")
        
#%% define lookup df with potential annotations
          
    annotations = pd.DataFrame({'mass_diff': [18.010565, 15.994915, 31.98983],
                                'annotation': ["[M-H2O+H]+", "[M+O+H]+", "[M+2O+H]+"]})
        
#%% setup results df and correlation matrix calculation  
          
    res = pd.DataFrame({'match_IDs': [],
                        'cor': [],
                        'mass_diff': [],
                        'annotation': []})
     
    cormat = df.T.corr(method = method)
    
#%% identify highly correlated features

    featnow = cormat.iloc[:,ID]
    featnow = featnow[featnow >= threshold] 
    featnow = featnow[featnow < 1]       
    res["match_IDs"] = featnow.index
    res["cor"] = featnow.values
    
    massnow = masses[ID]
    featmasses = masses[featnow.index]
    featmassdiff = abs(featmasses - massnow)
    res["mass_diff"] = featmassdiff.values
    
    for a in range(len(annotations)):
       idsi = res[(np.isclose(res["mass_diff"], annotations.loc[a, "mass_diff"]))]
       res.loc[idsi.index, "annotation"] = annotations.loc[a, "annotation"]
    
  
    return res