#!/bin/bash
# Copyright (c) Facebook, Inc. and its affiliates.
#
# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

# Exit if any error occurs
set -e

usage() {
  echo "Usage: $0 [-b] [-o] [-r] [-t] [-k kernel_name]"
  echo ""
  echo "Build Ax documentation. Must be executed from root of Ax repository."
  echo ""
  echo "  -b   Build static version of documentation (otherwise start server)."
  echo "  -o   Only Docusaurus (skip Sphinx, tutorials). Useful when just make change to Docusaurus settings."
  echo "  -t   Execute tutorials (instead of just converting)."
  echo "  -r   Convert backtick-quoted class or function names in .md files into links to API documentation."
  echo "  -k   Kernel name to use for executing tutorials. Use Jupyter default if not set."
  echo ""
  exit 1
}

BUILD_STATIC=false
ONLY_DOCUSAURUS=false
BUILD_TUTORIALS=false
KERNEL_NAME=false
INSERT_API_REFS=false

while getopts 'hbotrk:' flag; do
  case "${flag}" in
    h)
      usage
      ;;
    b)
      BUILD_STATIC=true
      ;;
    o)
      ONLY_DOCUSAURUS=true
      ;;
    t)
      BUILD_TUTORIALS=true
      ;;
    r)
      INSERT_API_REFS=true
      ;;
    k)
      KERNEL_NAME=${OPTARG}
      ;;
    *)
      usage
      ;;
  esac
done

if [[ $ONLY_DOCUSAURUS == false ]]; then
  # generate Sphinx documentation
  echo "-----------------------------------"
  echo "Generating API reference via Sphinx"
  echo "-----------------------------------"
  cd sphinx || exit
  make html
  cd .. || exit
fi

# init Docusaurus deps
echo "-----------------------------------"
echo "Getting Docusaurus deps"
echo "-----------------------------------"
cd website || exit
yarn

if [[ $ONLY_DOCUSAURUS == false ]]; then
  # run script to parse html generated by sphinx
  echo "--------------------------------------------"
  echo "Parsing Sphinx docs and moving to Docusaurus"
  echo "--------------------------------------------"
  cd ..
  mkdir -p "website/pages/api/"

  cwd=$(pwd)
  python3 scripts/parse_sphinx.py -i "${cwd}/sphinx/build/html/" -o "${cwd}/website/pages/api/"

  SPHINX_JS_DIR='sphinx/build/html/_static/'
  DOCUSAURUS_JS_DIR='website/static/js/'

  mkdir -p $DOCUSAURUS_JS_DIR

  # move JS files from /sphinx/build/html/_static/*:
  cp "${SPHINX_JS_DIR}documentation_options.js" "${DOCUSAURUS_JS_DIR}documentation_options.js"
  cp "${SPHINX_JS_DIR}jquery.js" "${DOCUSAURUS_JS_DIR}jquery.js"
  cp "${SPHINX_JS_DIR}underscore.js" "${DOCUSAURUS_JS_DIR}underscore.js"
  cp "${SPHINX_JS_DIR}doctools.js" "${DOCUSAURUS_JS_DIR}doctools.js"
  cp "${SPHINX_JS_DIR}language_data.js" "${DOCUSAURUS_JS_DIR}language_data.js"
  cp "${SPHINX_JS_DIR}searchtools.js" "${DOCUSAURUS_JS_DIR}searchtools.js"

  # searchindex.js is not static util
  cp "sphinx/build/html/searchindex.js" "${DOCUSAURUS_JS_DIR}searchindex.js"

  # copy module sources
  cp -r "sphinx/build/html/_sources/" "website/static/_sphinx-sources/"

  echo "-----------------------------------"
  echo "Generating tutorials"
  echo "-----------------------------------"
  # mkdir -p "website/_tutorials"
  # mkdir -p "website/static/files"
  if [[ $BUILD_TUTORIALS == true ]]; then
    if [[ $KERNEL_NAME == false ]]; then
      python3 scripts/make_tutorials.py -w "${cwd}" -e
    else
      python3 scripts/make_tutorials.py -w "${cwd}" -e -k "${KERNEL_NAME}"
    fi
  else
    python3 scripts/make_tutorials.py -w "${cwd}"
  fi

cd website || exit
fi  # end of not only Docusaurus block

if [[ $INSERT_API_REFS == true ]]; then
  echo "-----------------------------------"
  echo "Inserting API reference links in Markdown files"
  echo "-----------------------------------"
  cd ..
  cwd=$(pwd)
  python3 scripts/insert_api_refs.py --source_path "${cwd}/ax" --docs_path "${cwd}/docs"
  cd - || exit
fi

if [[ $BUILD_STATIC == true ]]; then
  echo "-----------------------------------"
  echo "Building static Docusaurus site"
  echo "-----------------------------------"
  yarn build
else
  echo "-----------------------------------"
  echo "Starting local server"
  echo "-----------------------------------"
  yarn start
fi
