"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addForwarderToLogGroups = exports.addForwarder = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_logs_destinations_1 = require("@aws-cdk/aws-logs-destinations");
const cdk = require("@aws-cdk/core");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
function getForwarder(scope, forwarderArn) {
    const forwarderConstructId = index_1.generateForwarderConstructId(forwarderArn);
    if (scope.node.tryFindChild(forwarderConstructId)) {
        return scope.node.tryFindChild(forwarderConstructId);
    }
    else {
        return lambda.Function.fromFunctionArn(scope, forwarderConstructId, forwarderArn);
    }
}
function addForwarder(scope, lambdaFunctions, forwarderArn) {
    const forwarder = getForwarder(scope, forwarderArn);
    const forwarderDestination = new aws_logs_destinations_1.LambdaDestination(forwarder);
    lambdaFunctions.forEach((lam) => {
        const subscriptionFilterName = index_1.generateSubscriptionFilterName(cdk.Names.uniqueId(lam), forwarderArn);
        loglevel_1.default.debug(`Adding log subscription ${subscriptionFilterName} for ${lam.functionName}`);
        lam.logGroup.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarder = addForwarder;
function addForwarderToLogGroups(scope, logGroups, forwarderArn) {
    const forwarder = getForwarder(scope, forwarderArn);
    const forwarderDestination = new aws_logs_destinations_1.LambdaDestination(forwarder);
    logGroups.forEach((group) => {
        const subscriptionFilterName = index_1.generateSubscriptionFilterName(cdk.Names.nodeUniqueId(group.node), forwarderArn);
        group.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarderToLogGroups = addForwarderToLogGroups;
//# sourceMappingURL=data:application/json;base64,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