"use strict";
const cluster_1 = require("../lib/cluster-resource-handler/cluster");
const mocks = require("./cluster-resource-handler-mocks");
module.exports = {
    setUp(callback) {
        mocks.reset();
        callback();
    },
    create: {
        async 'onCreate: minimal defaults (vpc + role)'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', mocks.MOCK_PROPS));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.configureAssumeRoleRequest, {
                RoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                RoleSessionName: 'AWSCDK.EKSCluster.Create.fake-request-id',
            });
            test.deepEqual(mocks.actualRequest.createClusterRequest, {
                roleArn: 'arn:of:role',
                resourcesVpcConfig: {
                    subnetIds: ['subnet1', 'subnet2'],
                    securityGroupIds: ['sg1', 'sg2', 'sg3'],
                },
                name: 'MyResourceId-fakerequestid',
            });
            test.done();
        },
        async 'generated cluster name does not exceed 100 characters'(test) {
            var _a, _b;
            // GIVEN
            const req = {
                StackId: 'fake-stack-id',
                RequestId: '602c078a-6181-4352-9676-4f00352445aa',
                ResourceType: 'Custom::EKSCluster',
                ServiceToken: 'boom',
                LogicalResourceId: 'hello'.repeat(30),
                ResponseURL: 'http://response-url',
                RequestType: 'Create',
                ResourceProperties: {
                    ServiceToken: 'boom',
                    Config: mocks.MOCK_PROPS,
                    AssumeRoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                },
            };
            // WHEN
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
            await handler.onEvent();
            // THEN
            test.equal((_a = mocks.actualRequest.createClusterRequest) === null || _a === void 0 ? void 0 : _a.name.length, 100);
            test.deepEqual((_b = mocks.actualRequest.createClusterRequest) === null || _b === void 0 ? void 0 : _b.name, 'hellohellohellohellohellohellohellohellohellohellohellohellohellohe-602c078a6181435296764f00352445aa');
            test.done();
        },
        async 'onCreate: explicit cluster name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                name: 'ExplicitCustomName',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.name, 'ExplicitCustomName');
            test.done();
        },
        async 'with no specific version'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                version: '12.34.56',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.version, '12.34.56');
            test.done();
        },
        async 'isCreateComplete still not complete if cluster is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'CREATING';
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { IsComplete: false });
            test.done();
        },
        async 'isCreateComplete throws if cluster is FAILED'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'FAILED';
            try {
                await handler.isComplete();
                test.ok(false, 'expected error to be thrown');
            }
            catch (err) {
                test.equal(err.message, 'Cluster is in a FAILED status');
            }
            test.done();
        },
        async 'isUpdateComplete throws if cluster is FAILED'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'FAILED';
            try {
                await handler.isComplete();
                test.ok(false, 'expected error to be thrown');
            }
            catch (err) {
                test.equal(err.message, 'Cluster is in a FAILED status');
            }
            test.done();
        },
        async 'isCreateComplete is complete when cluster is ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp, {
                IsComplete: true,
                Data: {
                    Name: 'physical-resource-id',
                    Endpoint: 'http://endpoint',
                    Arn: 'arn:cluster-arn',
                    CertificateAuthorityData: 'certificateAuthority-data',
                    ClusterSecurityGroupId: '',
                    EncryptionConfigKeyArn: '',
                    OpenIdConnectIssuerUrl: '',
                    OpenIdConnectIssuer: '',
                },
            });
            test.done();
        },
    },
    delete: {
        async 'returns correct physical name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.onEvent();
            test.deepEqual(mocks.actualRequest.deleteClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { PhysicalResourceId: 'physical-resource-id' });
            test.done();
        },
        async 'onDelete ignores ResourceNotFoundException'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.deleteClusterErrorCode = 'ResourceNotFoundException';
            await handler.onEvent();
            test.done();
        },
        async 'isDeleteComplete returns false as long as describeCluster succeeds'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.ok(!resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete returns true when describeCluster throws a ResourceNotFound exception'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'ResourceNotFoundException';
            const resp = await handler.isComplete();
            test.ok(resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete propagates other errors'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'OtherException';
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            test.equal(error.code, 'OtherException');
            test.done();
        },
    },
    update: {
        async 'no change'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', mocks.MOCK_PROPS, mocks.MOCK_PROPS));
            const resp = await handler.onEvent();
            test.equal(resp, undefined);
            test.equal(mocks.actualRequest.createClusterRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterConfigRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
            test.done();
        },
        async 'isUpdateComplete is not complete when status is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'UPDATING';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, false);
            test.done();
        },
        async 'isUpdateComplete waits for ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, true);
            test.done();
        },
        'requires replacement': {
            'name change': {
                async 'explicit name change'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: 'new-cluster-name-1234',
                    }, {
                        name: 'old-cluster-name',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'new-cluster-name-1234',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'new-cluster-name-1234' });
                    test.done();
                },
                async 'from auto-gen name to explicit name'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: undefined,
                    }, {
                        name: 'explicit',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'MyResourceId-fakerequestid',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                    test.done();
                },
            },
            async 'subnets or security groups requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                }, {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1'],
                        securityGroupIds: ['sg2'],
                    },
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'arn:of:role',
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                });
                test.done();
            },
            async '"roleArn" requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                }, {
                    roleArn: 'old-arn',
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'fails if cluster has an explicit "name" that is the same as the old "name"'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // THEN
                let err;
                try {
                    await handler.onEvent();
                }
                catch (e) {
                    err = e;
                }
                test.equal(err === null || err === void 0 ? void 0 : err.message, 'Cannot replace cluster "explicit-cluster-name" since it has an explicit physical name. Either rename the cluster or remove the "name" configuration');
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it does not'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: undefined,
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it has a different name'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'new-explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'old-explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'new-explicit-cluster-name' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'new-explicit-cluster-name',
                    roleArn: 'new-arn',
                });
                test.done();
            },
        },
        'isUpdateComplete with EKS update ID': {
            async 'with "Failed" status'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Failed';
                mocks.simulateResponse.describeUpdateResponseMockErrors = [
                    {
                        errorMessage: 'errorMessageMock',
                        errorCode: 'errorCodeMock',
                        resourceIds: [
                            'foo', 'bar',
                        ],
                    },
                ];
                let error;
                try {
                    await isCompleteHandler.isComplete();
                }
                catch (e) {
                    error = e;
                }
                test.ok(error);
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(error.message, 'cluster update id "foobar" failed with errors: [{"errorMessage":"errorMessageMock","errorCode":"errorCodeMock","resourceIds":["foo","bar"]}]');
                test.done();
            },
            async 'with "InProgress" status, returns IsComplete=false'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'InProgress';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(response.IsComplete, false);
                test.done();
            },
            async 'with "Successful" status, returns IsComplete=true with "Data"'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Successful';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.deepEqual(response, {
                    IsComplete: true,
                    Data: {
                        Name: 'physical-resource-id',
                        Endpoint: 'http://endpoint',
                        Arn: 'arn:cluster-arn',
                        CertificateAuthorityData: 'certificateAuthority-data',
                        ClusterSecurityGroupId: '',
                        EncryptionConfigKeyArn: '',
                        OpenIdConnectIssuerUrl: '',
                        OpenIdConnectIssuer: '',
                    },
                });
                test.done();
            },
        },
        'in-place': {
            'version change': {
                async 'from undefined to a specific value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '12.34',
                    }, {
                        version: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '12.34',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from a specific value to another value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '2.0',
                    }, {
                        version: '1.1',
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '2.0',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'to a new value that is already the current version'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '1.0',
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.describeClusterRequest, { name: 'physical-resource-id' });
                    test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'fails from specific value to undefined'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: undefined,
                    }, {
                        version: '1.2',
                    }));
                    let error;
                    try {
                        await handler.onEvent();
                    }
                    catch (e) {
                        error = e;
                    }
                    test.equal(error.message, 'Cannot remove cluster version configuration. Current version is 1.2');
                    test.done();
                },
            },
            'logging or access change': {
                async 'from undefined to partial logging enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api'],
                                    enabled: true,
                                },
                            ],
                        },
                    }, {
                        logging: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api'],
                                    enabled: true,
                                },
                            ],
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from partial vpc configuration to only private access enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        resourcesVpcConfig: {
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                            endpointPrivateAccess: true,
                        },
                    }, {
                        resourcesVpcConfig: {
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: undefined,
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: undefined,
                            publicAccessCidrs: undefined,
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from undefined to both logging and access fully enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api', 'audit', 'authenticator', 'controllerManager', 'scheduler'],
                                    enabled: true,
                                },
                            ],
                        },
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: true,
                            publicAccessCidrs: ['0.0.0.0/0'],
                        },
                    }, {
                        logging: undefined,
                        resourcesVpcConfig: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api', 'audit', 'authenticator', 'controllerManager', 'scheduler'],
                                    enabled: true,
                                },
                            ],
                        },
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: true,
                            publicAccessCidrs: ['0.0.0.0/0'],
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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