"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const a = new lib_1.Metric({ namespace: 'Test', metricName: 'ACount' });
const b = new lib_1.Metric({ namespace: 'Test', metricName: 'BCount', statistic: 'Average' });
const c = new lib_1.Metric({ namespace: 'Test', metricName: 'CCount' });
const b99 = new lib_1.Metric({ namespace: 'Test', metricName: 'BCount', statistic: 'p99' });
let stack;
function graphMetricsAre(test, w, metrics) {
    test.deepEqual(stack.resolve(w.toJson()), [{
            type: 'metric',
            width: 6,
            height: 6,
            properties: { view: 'timeSeries',
                region: { Ref: 'AWS::Region' },
                metrics,
                yAxis: {} }
        }]);
}
function alarmMetricsAre(metrics) {
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
        Metrics: metrics
    }));
}
module.exports = {
    'setUp'(cb) {
        stack = new core_1.Stack();
        cb();
    },
    'can not use invalid variable names in MathExpression'(test) {
        test.throws(() => {
            new lib_1.MathExpression({
                expression: 'HAPPY + JOY',
                usingMetrics: {
                    HAPPY: a,
                    JOY: b
                }
            });
        }, /Invalid variable names in expression/);
        test.done();
    },
    'cannot reuse variable names in nested MathExpressions'(test) {
        // WHEN
        test.throws(() => {
            new lib_1.MathExpression({
                expression: 'a + e',
                usingMetrics: {
                    a,
                    e: new lib_1.MathExpression({
                        expression: 'a + c',
                        usingMetrics: { a: b, c }
                    })
                }
            });
        }, /The ID 'a' used for two metrics in the expression: 'BCount' and 'ACount'. Rename one/);
        test.done();
    },
    'can not use invalid period in MathExpression'(test) {
        test.throws(() => {
            new lib_1.MathExpression({
                expression: 'a+b',
                usingMetrics: { a, b },
                period: core_1.Duration.seconds(20)
            });
        }, /'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received 20/);
        test.done();
    },
    'MathExpression optimization: "with" with the same period returns the same object'(test) {
        const m = new lib_1.MathExpression({ expression: 'SUM(METRICS())', usingMetrics: {}, period: core_1.Duration.minutes(10) });
        // Note: object equality, NOT deep equality on purpose
        test.equals(m.with({}), m);
        test.equals(m.with({ period: core_1.Duration.minutes(10) }), m);
        test.notEqual(m.with({ period: core_1.Duration.minutes(5) }), m);
        test.done();
    },
    'in graphs': {
        'MathExpressions can be added to a graph'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + b',
                        usingMetrics: { a, b }
                    })
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ expression: 'a + b', label: 'a + b' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                ['Test', 'BCount', { visible: false, id: 'b' }],
            ]);
            test.done();
        },
        'can nest MathExpressions in a graph'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'b + c',
                                usingMetrics: { b, c }
                            })
                        }
                    })
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ label: 'a + e', expression: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'b + c', visible: false, id: 'e' }],
                ['Test', 'BCount', { visible: false, id: 'b' }],
                ['Test', 'CCount', { visible: false, id: 'c' }]
            ]);
            test.done();
        },
        'can add the same metric under different ids'(test) {
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'b + c',
                                usingMetrics: { b: a, c }
                            })
                        }
                    })
                ],
            });
            graphMetricsAre(test, graph, [
                [{ label: 'a + e', expression: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'b + c', visible: false, id: 'e' }],
                ['Test', 'ACount', { visible: false, id: 'b' }],
                ['Test', 'CCount', { visible: false, id: 'c' }]
            ]);
            test.done();
        },
        'can reuse identifiers in MathExpressions if metrics are the same'(test) {
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'a + c',
                                usingMetrics: { a, c }
                            })
                        }
                    })
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ label: 'a + e', expression: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'a + c', visible: false, id: 'e' }],
                ['Test', 'CCount', { visible: false, id: 'c' }]
            ]);
            test.done();
        },
        'MathExpression and its constituent metrics can both be added to a graph'(test) {
            const graph = new lib_1.GraphWidget({
                left: [
                    a,
                    new lib_1.MathExpression({
                        expression: 'a + b',
                        usingMetrics: { a, b }
                    })
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                ['Test', 'ACount', { id: 'a' }],
                [{ label: 'a + b', expression: 'a + b' }],
                ['Test', 'BCount', { visible: false, id: 'b' }]
            ]);
            test.done();
        },
        'MathExpression controls period of metrics directly used in it'(test) {
            // Check that if we add A with { period: 10s } to a mathexpression of period 5m
            // then two metric lines are added for A, one at 10s and one at 5m
            const graph = new lib_1.GraphWidget({
                left: [
                    a.with({ period: core_1.Duration.seconds(10) }),
                    new lib_1.MathExpression({
                        expression: 'a + b',
                        usingMetrics: { a: a.with({ period: core_1.Duration.seconds(10) }), b }
                    })
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                ['Test', 'ACount', { period: 10 }],
                [{ label: 'a + b', expression: 'a + b' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                ['Test', 'BCount', { visible: false, id: 'b' }]
            ]);
            test.done();
        },
        'MathExpression controls period of metrics transitively used in it'(test) {
            // Same as the previous test, but recursively
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'a + b',
                                period: core_1.Duration.minutes(1),
                                usingMetrics: { a, b }
                            })
                        }
                    })
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ expression: 'a + e', label: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'a + b', visible: false, id: 'e' }],
                ['Test', 'BCount', { visible: false, id: 'b' }]
            ]);
            test.done();
        },
        'can use percentiles in expression metrics in graphs'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + b99',
                        usingMetrics: { a, b99 }
                    })
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ expression: 'a + b99', label: 'a + b99' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                ['Test', 'BCount', { visible: false, id: 'b99', stat: 'p99' }],
            ]);
            test.done();
        },
    },
    'in alarms': {
        'MathExpressions can be used for an alarm'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1, evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + b',
                    usingMetrics: { a, b }
                })
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: "a + b",
                    Id: "expr_1"
                },
                {
                    Id: "a",
                    MetricStat: {
                        Metric: {
                            MetricName: "ACount",
                            Namespace: "Test"
                        },
                        Period: 300,
                        Stat: "Average"
                    },
                    ReturnData: false
                },
                {
                    Id: "b",
                    MetricStat: {
                        Metric: {
                            MetricName: "BCount",
                            Namespace: "Test"
                        },
                        Period: 300,
                        Stat: "Average"
                    },
                    ReturnData: false
                }
            ]);
            test.done();
        },
        'can nest MathExpressions in an alarm'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1, evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + e',
                    usingMetrics: {
                        a,
                        e: new lib_1.MathExpression({
                            expression: 'b + c',
                            usingMetrics: { b, c }
                        })
                    }
                })
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: "a + e",
                    Id: "expr_1"
                },
                {
                    Id: "a",
                    MetricStat: {
                        Metric: {
                            MetricName: "ACount",
                            Namespace: "Test"
                        },
                        Period: 300,
                        Stat: "Average"
                    },
                    ReturnData: false
                },
                {
                    Expression: "b + c",
                    Id: "e",
                    ReturnData: false
                },
                {
                    Id: "b",
                    MetricStat: {
                        Metric: {
                            MetricName: "BCount",
                            Namespace: "Test"
                        },
                        Period: 300,
                        Stat: "Average"
                    },
                    ReturnData: false
                },
                {
                    Id: "c",
                    MetricStat: {
                        Metric: {
                            MetricName: "CCount",
                            Namespace: "Test"
                        },
                        Period: 300,
                        Stat: "Average"
                    },
                    ReturnData: false
                }
            ]);
            test.done();
        },
        'MathExpression controls period of metrics transitively used in it with alarms'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1, evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + e',
                    usingMetrics: {
                        a,
                        e: new lib_1.MathExpression({
                            expression: 'b + c',
                            usingMetrics: { b, c },
                            period: core_1.Duration.minutes(1)
                        })
                    },
                    period: core_1.Duration.seconds(30)
                })
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: "a + e",
                    Id: "expr_1"
                },
                {
                    Id: "a",
                    MetricStat: {
                        Metric: {
                            MetricName: "ACount",
                            Namespace: "Test"
                        },
                        Period: 30,
                        Stat: "Average"
                    },
                    ReturnData: false
                },
                {
                    Expression: "b + c",
                    Id: "e",
                    ReturnData: false
                },
                {
                    Id: "b",
                    MetricStat: {
                        Metric: {
                            MetricName: "BCount",
                            Namespace: "Test"
                        },
                        Period: 30,
                        Stat: "Average"
                    },
                    ReturnData: false
                },
                {
                    Id: "c",
                    MetricStat: {
                        Metric: {
                            MetricName: "CCount",
                            Namespace: "Test"
                        },
                        Period: 30,
                        Stat: "Average"
                    },
                    ReturnData: false
                }
            ]);
            test.done();
        },
        'annotation for a mathexpression alarm is calculated based upon constituent metrics'(test) {
            // GIVEN
            const alarm = new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1, evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    period: core_1.Duration.minutes(10),
                    expression: 'a + b',
                    usingMetrics: { a, b: b.with({ period: core_1.Duration.minutes(20) }) } // This is overridden
                })
            });
            // WHEN
            const alarmLabel = alarm.toAnnotation().label;
            // THEN
            test.equals(alarmLabel, 'a + b >= 1 for 1 datapoints within 10 minutes');
            test.done();
        },
        'can use percentiles in expression metrics in alarms'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1, evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + b99',
                    usingMetrics: { a, b99 }
                })
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: "a + b99",
                    Id: "expr_1"
                },
                {
                    Id: "a",
                    MetricStat: {
                        Metric: {
                            MetricName: "ACount",
                            Namespace: "Test"
                        },
                        Period: 300,
                        Stat: "Average"
                    },
                    ReturnData: false
                },
                {
                    Id: "b99",
                    MetricStat: {
                        Metric: {
                            MetricName: "BCount",
                            Namespace: "Test"
                        },
                        Period: 300,
                        Stat: "p99"
                    },
                    ReturnData: false
                }
            ]);
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5tZXRyaWMtbWF0aC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QubWV0cmljLW1hdGgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUEyRDtBQUMzRCx3Q0FBZ0Q7QUFFaEQsZ0NBQTZFO0FBRTdFLE1BQU0sQ0FBQyxHQUFHLElBQUksWUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztBQUNsRSxNQUFNLENBQUMsR0FBRyxJQUFJLFlBQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLFFBQVEsRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQztBQUN4RixNQUFNLENBQUMsR0FBRyxJQUFJLFlBQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7QUFDbEUsTUFBTSxHQUFHLEdBQUcsSUFBSSxZQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxRQUFRLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7QUFFdEYsSUFBSSxLQUFZLENBQUM7QUFrZ0JqQixTQUFTLGVBQWUsQ0FBQyxJQUFVLEVBQUUsQ0FBVSxFQUFFLE9BQWM7SUFDN0QsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLENBQUU7WUFDMUMsSUFBSSxFQUFFLFFBQVE7WUFDZCxLQUFLLEVBQUUsQ0FBQztZQUNSLE1BQU0sRUFBRSxDQUFDO1lBQ1QsVUFBVSxFQUNWLEVBQUUsSUFBSSxFQUFFLFlBQVk7Z0JBQ2xCLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7Z0JBQzlCLE9BQU87Z0JBQ1AsS0FBSyxFQUFFLEVBQUUsRUFBRTtTQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ3RCLENBQUM7QUFFRCxTQUFTLGVBQWUsQ0FBQyxPQUFjO0lBQ3JDLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsd0JBQXdCLEVBQUU7UUFDMUQsT0FBTyxFQUFFLE9BQU87S0FDakIsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDO0FBamhCRCxpQkFBUztJQUNQLE9BQU8sQ0FBQyxFQUFjO1FBQ3BCLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQ3BCLEVBQUUsRUFBRSxDQUFDO0lBQ1AsQ0FBQztJQUVELHNEQUFzRCxDQUFDLElBQVU7UUFDL0QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixJQUFJLG9CQUFjLENBQUM7Z0JBQ2pCLFVBQVUsRUFBRSxhQUFhO2dCQUN6QixZQUFZLEVBQUU7b0JBQ1osS0FBSyxFQUFFLENBQUM7b0JBQ1IsR0FBRyxFQUFFLENBQUM7aUJBQ1A7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLEVBQUUsc0NBQXNDLENBQUMsQ0FBQztRQUUzQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdURBQXVELENBQUMsSUFBVTtRQUNoRSxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixJQUFJLG9CQUFjLENBQUM7Z0JBQ2pCLFVBQVUsRUFBRSxPQUFPO2dCQUNuQixZQUFZLEVBQUU7b0JBQ1osQ0FBQztvQkFDRCxDQUFDLEVBQUUsSUFBSSxvQkFBYyxDQUFDO3dCQUNwQixVQUFVLEVBQUUsT0FBTzt3QkFDbkIsWUFBWSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUU7cUJBQzFCLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLEVBQUUsc0ZBQXNGLENBQUMsQ0FBQztRQUUzRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOENBQThDLENBQUMsSUFBVTtRQUN2RCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLElBQUksb0JBQWMsQ0FBQztnQkFDakIsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLFlBQVksRUFBRSxFQUFDLENBQUMsRUFBRSxDQUFDLEVBQUM7Z0JBQ3BCLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQzthQUM3QixDQUFDLENBQUM7UUFDTCxDQUFDLEVBQUUseUVBQXlFLENBQUMsQ0FBQztRQUU5RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0ZBQWtGLENBQUMsSUFBVTtRQUMzRixNQUFNLENBQUMsR0FBRyxJQUFJLG9CQUFjLENBQUMsRUFBRSxVQUFVLEVBQUUsZ0JBQWdCLEVBQUUsWUFBWSxFQUFFLEVBQUUsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFL0csc0RBQXNEO1FBQ3RELElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUMzQixJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFekQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRTFELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxXQUFXLEVBQUU7UUFDWCx5Q0FBeUMsQ0FBQyxJQUFVO1lBQ2xELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLGlCQUFXLENBQUM7Z0JBQzVCLElBQUksRUFBRTtvQkFDSixJQUFJLG9CQUFjLENBQUM7d0JBQ2pCLFVBQVUsRUFBRSxPQUFPO3dCQUNuQixZQUFZLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFO3FCQUN2QixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQWUsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO2dCQUMzQixDQUFFLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLENBQUU7Z0JBQzNDLENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFFO2dCQUNqRCxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTthQUNsRCxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQscUNBQXFDLENBQUMsSUFBVTtZQUM5QyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxpQkFBVyxDQUFDO2dCQUM1QixJQUFJLEVBQUU7b0JBQ0osSUFBSSxvQkFBYyxDQUFDO3dCQUNqQixVQUFVLEVBQUUsT0FBTzt3QkFDbkIsWUFBWSxFQUFFOzRCQUNaLENBQUM7NEJBQ0QsQ0FBQyxFQUFFLElBQUksb0JBQWMsQ0FBQztnQ0FDcEIsVUFBVSxFQUFFLE9BQU87Z0NBQ25CLFlBQVksRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUU7NkJBQ3ZCLENBQUM7eUJBQ0g7cUJBQ0YsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFlLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRTtnQkFDM0IsQ0FBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxDQUFFO2dCQUMzQyxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTtnQkFDakQsQ0FBRSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUU7Z0JBQ3BELENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFFO2dCQUNqRCxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTthQUNsRCxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNkNBQTZDLENBQUMsSUFBVTtZQUN0RCxNQUFNLEtBQUssR0FBRyxJQUFJLGlCQUFXLENBQUM7Z0JBQzVCLElBQUksRUFBRTtvQkFDSixJQUFJLG9CQUFjLENBQUM7d0JBQ2pCLFVBQVUsRUFBRSxPQUFPO3dCQUNuQixZQUFZLEVBQUU7NEJBQ1osQ0FBQzs0QkFDRCxDQUFDLEVBQUUsSUFBSSxvQkFBYyxDQUFDO2dDQUNwQixVQUFVLEVBQUUsT0FBTztnQ0FDbkIsWUFBWSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUU7NkJBQzFCLENBQUM7eUJBQ0g7cUJBQ0YsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztZQUVILGVBQWUsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO2dCQUMzQixDQUFFLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLENBQUU7Z0JBQzNDLENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFFO2dCQUNqRCxDQUFFLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTtnQkFDcEQsQ0FBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUU7Z0JBQ2pELENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFFO2FBQ2xELENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxrRUFBa0UsQ0FBQyxJQUFVO1lBQzNFLE1BQU0sS0FBSyxHQUFHLElBQUksaUJBQVcsQ0FBQztnQkFDNUIsSUFBSSxFQUFFO29CQUNKLElBQUksb0JBQWMsQ0FBQzt3QkFDakIsVUFBVSxFQUFFLE9BQU87d0JBQ25CLFlBQVksRUFBRTs0QkFDWixDQUFDOzRCQUNELENBQUMsRUFBRSxJQUFJLG9CQUFjLENBQUM7Z0NBQ3BCLFVBQVUsRUFBRSxPQUFPO2dDQUNuQixZQUFZLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFOzZCQUN2QixDQUFDO3lCQUNIO3FCQUNGLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBZSxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUU7Z0JBQzNCLENBQUUsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsQ0FBRTtnQkFDM0MsQ0FBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUU7Z0JBQ2pELENBQUUsRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFFO2dCQUNwRCxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTthQUNsRCxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQseUVBQXlFLENBQUMsSUFBVTtZQUNsRixNQUFNLEtBQUssR0FBRyxJQUFJLGlCQUFXLENBQUM7Z0JBQzVCLElBQUksRUFBRTtvQkFDSixDQUFDO29CQUNELElBQUksb0JBQWMsQ0FBQzt3QkFDakIsVUFBVSxFQUFFLE9BQU87d0JBQ25CLFlBQVksRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUU7cUJBQ3ZCLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBZSxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUU7Z0JBQzNCLENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTtnQkFDakMsQ0FBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxDQUFFO2dCQUMzQyxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTthQUNsRCxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsK0RBQStELENBQUMsSUFBVTtZQUN4RSwrRUFBK0U7WUFDL0Usa0VBQWtFO1lBQ2xFLE1BQU0sS0FBSyxHQUFHLElBQUksaUJBQVcsQ0FBQztnQkFDNUIsSUFBSSxFQUFFO29CQUNKLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDO29CQUN4QyxJQUFJLG9CQUFjLENBQUM7d0JBQ2pCLFVBQVUsRUFBRSxPQUFPO3dCQUNuQixZQUFZLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUU7cUJBQ2pFLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBZSxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUU7Z0JBQzNCLENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLE1BQU0sRUFBRSxFQUFFLEVBQUUsQ0FBRTtnQkFDcEMsQ0FBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxDQUFFO2dCQUMzQyxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTtnQkFDakQsQ0FBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUU7YUFDbEQsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG1FQUFtRSxDQUFDLElBQVU7WUFDNUUsNkNBQTZDO1lBRTdDLE1BQU0sS0FBSyxHQUFHLElBQUksaUJBQVcsQ0FBQztnQkFDNUIsSUFBSSxFQUFFO29CQUNKLElBQUksb0JBQWMsQ0FBQzt3QkFDakIsVUFBVSxFQUFFLE9BQU87d0JBQ25CLFlBQVksRUFBRTs0QkFDWixDQUFDOzRCQUNELENBQUMsRUFBRSxJQUFJLG9CQUFjLENBQUM7Z0NBQ3BCLFVBQVUsRUFBRSxPQUFPO2dDQUNuQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7Z0NBQzNCLFlBQVksRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUU7NkJBQ3ZCLENBQUM7eUJBQ0g7cUJBQ0YsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFlLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRTtnQkFDM0IsQ0FBRSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxDQUFFO2dCQUMzQyxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBRTtnQkFDakQsQ0FBRSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUU7Z0JBQ3BELENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFFO2FBQ2xELENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxxREFBcUQsQ0FBQyxJQUFVO1lBQzlELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLGlCQUFXLENBQUM7Z0JBQzVCLElBQUksRUFBRTtvQkFDSixJQUFJLG9CQUFjLENBQUM7d0JBQ2pCLFVBQVUsRUFBRSxTQUFTO3dCQUNyQixZQUFZLEVBQUUsRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFO3FCQUN6QixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQWUsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO2dCQUMzQixDQUFFLEVBQUUsVUFBVSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLENBQUU7Z0JBQy9DLENBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFFO2dCQUNqRCxDQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxDQUFFO2FBQ2pFLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELFdBQVcsRUFBRTtRQUNYLDBDQUEwQyxDQUFDLElBQVU7WUFDbkQsUUFBUTtZQUNSLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7Z0JBQ3hCLFNBQVMsRUFBRSxDQUFDLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQztnQkFDbEMsTUFBTSxFQUFFLElBQUksb0JBQWMsQ0FBQztvQkFDekIsVUFBVSxFQUFFLE9BQU87b0JBQ25CLFlBQVksRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUU7aUJBQ3ZCLENBQUM7YUFDSCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBZSxDQUFDO2dCQUNkO29CQUNFLFVBQVUsRUFBRSxPQUFPO29CQUNuQixFQUFFLEVBQUUsUUFBUTtpQkFDYjtnQkFDRDtvQkFDRSxFQUFFLEVBQUUsR0FBRztvQkFDUCxVQUFVLEVBQUU7d0JBQ1YsTUFBTSxFQUFFOzRCQUNOLFVBQVUsRUFBRSxRQUFROzRCQUNwQixTQUFTLEVBQUUsTUFBTTt5QkFDbEI7d0JBQ0QsTUFBTSxFQUFFLEdBQUc7d0JBQ1gsSUFBSSxFQUFFLFNBQVM7cUJBQ2hCO29CQUNELFVBQVUsRUFBRSxLQUFLO2lCQUNsQjtnQkFDRDtvQkFDRSxFQUFFLEVBQUUsR0FBRztvQkFDUCxVQUFVLEVBQUU7d0JBQ1YsTUFBTSxFQUFFOzRCQUNOLFVBQVUsRUFBRSxRQUFROzRCQUNwQixTQUFTLEVBQUUsTUFBTTt5QkFDbEI7d0JBQ0QsTUFBTSxFQUFFLEdBQUc7d0JBQ1gsSUFBSSxFQUFFLFNBQVM7cUJBQ2hCO29CQUNELFVBQVUsRUFBRSxLQUFLO2lCQUNsQjthQUVGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxzQ0FBc0MsQ0FBQyxJQUFVO1lBQy9DLFFBQVE7WUFDUixJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUN4QixTQUFTLEVBQUUsQ0FBQyxFQUFFLGlCQUFpQixFQUFFLENBQUM7Z0JBQ2xDLE1BQU0sRUFBRSxJQUFJLG9CQUFjLENBQUM7b0JBQ3pCLFVBQVUsRUFBRSxPQUFPO29CQUNuQixZQUFZLEVBQUU7d0JBQ1osQ0FBQzt3QkFDRCxDQUFDLEVBQUUsSUFBSSxvQkFBYyxDQUFDOzRCQUNwQixVQUFVLEVBQUUsT0FBTzs0QkFDbkIsWUFBWSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRTt5QkFDdkIsQ0FBQztxQkFDSDtpQkFDRixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQWUsQ0FBQztnQkFDZDtvQkFDRSxVQUFVLEVBQUUsT0FBTztvQkFDbkIsRUFBRSxFQUFFLFFBQVE7aUJBQ2I7Z0JBQ0Q7b0JBQ0UsRUFBRSxFQUFFLEdBQUc7b0JBQ1AsVUFBVSxFQUFFO3dCQUNWLE1BQU0sRUFBRTs0QkFDTixVQUFVLEVBQUUsUUFBUTs0QkFDcEIsU0FBUyxFQUFFLE1BQU07eUJBQ2xCO3dCQUNELE1BQU0sRUFBRSxHQUFHO3dCQUNYLElBQUksRUFBRSxTQUFTO3FCQUNoQjtvQkFDRCxVQUFVLEVBQUUsS0FBSztpQkFDbEI7Z0JBQ0Q7b0JBQ0UsVUFBVSxFQUFFLE9BQU87b0JBQ25CLEVBQUUsRUFBRSxHQUFHO29CQUNQLFVBQVUsRUFBRSxLQUFLO2lCQUNsQjtnQkFDRDtvQkFDRSxFQUFFLEVBQUUsR0FBRztvQkFDUCxVQUFVLEVBQUU7d0JBQ1YsTUFBTSxFQUFFOzRCQUNOLFVBQVUsRUFBRSxRQUFROzRCQUNwQixTQUFTLEVBQUUsTUFBTTt5QkFDbEI7d0JBQ0QsTUFBTSxFQUFFLEdBQUc7d0JBQ1gsSUFBSSxFQUFFLFNBQVM7cUJBQ2hCO29CQUNELFVBQVUsRUFBRSxLQUFLO2lCQUNsQjtnQkFDRDtvQkFDRSxFQUFFLEVBQUUsR0FBRztvQkFDUCxVQUFVLEVBQUU7d0JBQ1YsTUFBTSxFQUFFOzRCQUNOLFVBQVUsRUFBRSxRQUFROzRCQUNwQixTQUFTLEVBQUUsTUFBTTt5QkFDbEI7d0JBQ0QsTUFBTSxFQUFFLEdBQUc7d0JBQ1gsSUFBSSxFQUFFLFNBQVM7cUJBQ2hCO29CQUNELFVBQVUsRUFBRSxLQUFLO2lCQUNsQjthQUNGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwrRUFBK0UsQ0FBQyxJQUFVO1lBQ3hGLFFBQVE7WUFDUixJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUN4QixTQUFTLEVBQUUsQ0FBQyxFQUFFLGlCQUFpQixFQUFFLENBQUM7Z0JBQ2xDLE1BQU0sRUFBRSxJQUFJLG9CQUFjLENBQUM7b0JBQ3pCLFVBQVUsRUFBRSxPQUFPO29CQUNuQixZQUFZLEVBQUU7d0JBQ1osQ0FBQzt3QkFDRCxDQUFDLEVBQUUsSUFBSSxvQkFBYyxDQUFDOzRCQUNwQixVQUFVLEVBQUUsT0FBTzs0QkFDbkIsWUFBWSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRTs0QkFDdEIsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO3lCQUM1QixDQUFDO3FCQUNIO29CQUNELE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztpQkFDN0IsQ0FBQzthQUNILENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFlLENBQUM7Z0JBQ2Q7b0JBQ0UsVUFBVSxFQUFFLE9BQU87b0JBQ25CLEVBQUUsRUFBRSxRQUFRO2lCQUNiO2dCQUNEO29CQUNFLEVBQUUsRUFBRSxHQUFHO29CQUNQLFVBQVUsRUFBRTt3QkFDVixNQUFNLEVBQUU7NEJBQ04sVUFBVSxFQUFFLFFBQVE7NEJBQ3BCLFNBQVMsRUFBRSxNQUFNO3lCQUNsQjt3QkFDRCxNQUFNLEVBQUUsRUFBRTt3QkFDVixJQUFJLEVBQUUsU0FBUztxQkFDaEI7b0JBQ0QsVUFBVSxFQUFFLEtBQUs7aUJBQ2xCO2dCQUNEO29CQUNFLFVBQVUsRUFBRSxPQUFPO29CQUNuQixFQUFFLEVBQUUsR0FBRztvQkFDUCxVQUFVLEVBQUUsS0FBSztpQkFDbEI7Z0JBQ0Q7b0JBQ0UsRUFBRSxFQUFFLEdBQUc7b0JBQ1AsVUFBVSxFQUFFO3dCQUNWLE1BQU0sRUFBRTs0QkFDTixVQUFVLEVBQUUsUUFBUTs0QkFDcEIsU0FBUyxFQUFFLE1BQU07eUJBQ2xCO3dCQUNELE1BQU0sRUFBRSxFQUFFO3dCQUNWLElBQUksRUFBRSxTQUFTO3FCQUNoQjtvQkFDRCxVQUFVLEVBQUUsS0FBSztpQkFDbEI7Z0JBQ0Q7b0JBQ0UsRUFBRSxFQUFFLEdBQUc7b0JBQ1AsVUFBVSxFQUFFO3dCQUNWLE1BQU0sRUFBRTs0QkFDTixVQUFVLEVBQUUsUUFBUTs0QkFDcEIsU0FBUyxFQUFFLE1BQU07eUJBQ2xCO3dCQUNELE1BQU0sRUFBRSxFQUFFO3dCQUNWLElBQUksRUFBRSxTQUFTO3FCQUNoQjtvQkFDRCxVQUFVLEVBQUUsS0FBSztpQkFDbEI7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsb0ZBQW9GLENBQUMsSUFBVTtZQUM3RixRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDdEMsU0FBUyxFQUFFLENBQUMsRUFBRSxpQkFBaUIsRUFBRSxDQUFDO2dCQUNsQyxNQUFNLEVBQUUsSUFBSSxvQkFBYyxDQUFDO29CQUN6QixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7b0JBQzVCLFVBQVUsRUFBRSxPQUFPO29CQUNuQixZQUFZLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUI7aUJBQ3ZGLENBQUM7YUFDSCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFlBQVksRUFBRSxDQUFDLEtBQUssQ0FBQztZQUU5QyxPQUFPO1lBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsK0NBQStDLENBQUMsQ0FBQztZQUV6RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQscURBQXFELENBQUMsSUFBVTtZQUM5RCxRQUFRO1lBQ1IsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDeEIsU0FBUyxFQUFFLENBQUMsRUFBRSxpQkFBaUIsRUFBRSxDQUFDO2dCQUNsQyxNQUFNLEVBQUUsSUFBSSxvQkFBYyxDQUFDO29CQUN6QixVQUFVLEVBQUUsU0FBUztvQkFDckIsWUFBWSxFQUFFLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRTtpQkFDekIsQ0FBQzthQUNILENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFlLENBQUM7Z0JBQ2Q7b0JBQ0UsVUFBVSxFQUFFLFNBQVM7b0JBQ3JCLEVBQUUsRUFBRSxRQUFRO2lCQUNiO2dCQUNEO29CQUNFLEVBQUUsRUFBRSxHQUFHO29CQUNQLFVBQVUsRUFBRTt3QkFDVixNQUFNLEVBQUU7NEJBQ04sVUFBVSxFQUFFLFFBQVE7NEJBQ3BCLFNBQVMsRUFBRSxNQUFNO3lCQUNsQjt3QkFDRCxNQUFNLEVBQUUsR0FBRzt3QkFDWCxJQUFJLEVBQUUsU0FBUztxQkFDaEI7b0JBQ0QsVUFBVSxFQUFFLEtBQUs7aUJBQ2xCO2dCQUNEO29CQUNFLEVBQUUsRUFBRSxLQUFLO29CQUNULFVBQVUsRUFBRTt3QkFDVixNQUFNLEVBQUU7NEJBQ04sVUFBVSxFQUFFLFFBQVE7NEJBQ3BCLFNBQVMsRUFBRSxNQUFNO3lCQUNsQjt3QkFDRCxNQUFNLEVBQUUsR0FBRzt3QkFDWCxJQUFJLEVBQUUsS0FBSztxQkFDWjtvQkFDRCxVQUFVLEVBQUUsS0FBSztpQkFDbEI7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2VMaWtlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCB7IER1cmF0aW9uLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCB7IEFsYXJtLCBHcmFwaFdpZGdldCwgSVdpZGdldCwgTWF0aEV4cHJlc3Npb24sIE1ldHJpYyB9IGZyb20gJy4uL2xpYic7XG5cbmNvbnN0IGEgPSBuZXcgTWV0cmljKHsgbmFtZXNwYWNlOiAnVGVzdCcsIG1ldHJpY05hbWU6ICdBQ291bnQnIH0pO1xuY29uc3QgYiA9IG5ldyBNZXRyaWMoeyBuYW1lc3BhY2U6ICdUZXN0JywgbWV0cmljTmFtZTogJ0JDb3VudCcsIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnIH0pO1xuY29uc3QgYyA9IG5ldyBNZXRyaWMoeyBuYW1lc3BhY2U6ICdUZXN0JywgbWV0cmljTmFtZTogJ0NDb3VudCcgfSk7XG5jb25zdCBiOTkgPSBuZXcgTWV0cmljKHsgbmFtZXNwYWNlOiAnVGVzdCcsIG1ldHJpY05hbWU6ICdCQ291bnQnLCBzdGF0aXN0aWM6ICdwOTknIH0pO1xuXG5sZXQgc3RhY2s6IFN0YWNrO1xuZXhwb3J0ID0ge1xuICAnc2V0VXAnKGNiOiAoKSA9PiB2b2lkKSB7XG4gICAgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjYigpO1xuICB9LFxuXG4gICdjYW4gbm90IHVzZSBpbnZhbGlkIHZhcmlhYmxlIG5hbWVzIGluIE1hdGhFeHByZXNzaW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgZXhwcmVzc2lvbjogJ0hBUFBZICsgSk9ZJyxcbiAgICAgICAgdXNpbmdNZXRyaWNzOiB7XG4gICAgICAgICAgSEFQUFk6IGEsXG4gICAgICAgICAgSk9ZOiBiXG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH0sIC9JbnZhbGlkIHZhcmlhYmxlIG5hbWVzIGluIGV4cHJlc3Npb24vKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW5ub3QgcmV1c2UgdmFyaWFibGUgbmFtZXMgaW4gbmVzdGVkIE1hdGhFeHByZXNzaW9ucycodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICBleHByZXNzaW9uOiAnYSArIGUnLFxuICAgICAgICB1c2luZ01ldHJpY3M6IHtcbiAgICAgICAgICBhLFxuICAgICAgICAgIGU6IG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAgICAgICBleHByZXNzaW9uOiAnYSArIGMnLFxuICAgICAgICAgICAgdXNpbmdNZXRyaWNzOiB7IGE6IGIsIGMgfVxuICAgICAgICAgIH0pXG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH0sIC9UaGUgSUQgJ2EnIHVzZWQgZm9yIHR3byBtZXRyaWNzIGluIHRoZSBleHByZXNzaW9uOiAnQkNvdW50JyBhbmQgJ0FDb3VudCcuIFJlbmFtZSBvbmUvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW4gbm90IHVzZSBpbnZhbGlkIHBlcmlvZCBpbiBNYXRoRXhwcmVzc2lvbicodGVzdDogVGVzdCkge1xuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAgIGV4cHJlc3Npb246ICdhK2InLFxuICAgICAgICB1c2luZ01ldHJpY3M6IHthLCBifSxcbiAgICAgICAgcGVyaW9kOiBEdXJhdGlvbi5zZWNvbmRzKDIwKVxuICAgICAgfSk7XG4gICAgfSwgLydwZXJpb2QnIG11c3QgYmUgMSwgNSwgMTAsIDMwLCBvciBhIG11bHRpcGxlIG9mIDYwIHNlY29uZHMsIHJlY2VpdmVkIDIwLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnTWF0aEV4cHJlc3Npb24gb3B0aW1pemF0aW9uOiBcIndpdGhcIiB3aXRoIHRoZSBzYW1lIHBlcmlvZCByZXR1cm5zIHRoZSBzYW1lIG9iamVjdCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IG0gPSBuZXcgTWF0aEV4cHJlc3Npb24oeyBleHByZXNzaW9uOiAnU1VNKE1FVFJJQ1MoKSknLCB1c2luZ01ldHJpY3M6IHt9LCBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMTApIH0pO1xuXG4gICAgLy8gTm90ZTogb2JqZWN0IGVxdWFsaXR5LCBOT1QgZGVlcCBlcXVhbGl0eSBvbiBwdXJwb3NlXG4gICAgdGVzdC5lcXVhbHMobS53aXRoKHt9KSwgbSk7XG4gICAgdGVzdC5lcXVhbHMobS53aXRoKHsgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDEwKSB9KSwgbSk7XG5cbiAgICB0ZXN0Lm5vdEVxdWFsKG0ud2l0aCh7IHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSB9KSwgbSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW4gZ3JhcGhzJzoge1xuICAgICdNYXRoRXhwcmVzc2lvbnMgY2FuIGJlIGFkZGVkIHRvIGEgZ3JhcGgnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBncmFwaCA9IG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgICAgZXhwcmVzc2lvbjogJ2EgKyBiJyxcbiAgICAgICAgICAgIHVzaW5nTWV0cmljczogeyBhLCBiIH1cbiAgICAgICAgICB9KVxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGdyYXBoTWV0cmljc0FyZSh0ZXN0LCBncmFwaCwgW1xuICAgICAgICBbIHsgZXhwcmVzc2lvbjogJ2EgKyBiJywgbGFiZWw6ICdhICsgYicgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0FDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0JDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYicgfSBdLFxuICAgICAgXSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIG5lc3QgTWF0aEV4cHJlc3Npb25zIGluIGEgZ3JhcGgnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBncmFwaCA9IG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgICAgZXhwcmVzc2lvbjogJ2EgKyBlJyxcbiAgICAgICAgICAgIHVzaW5nTWV0cmljczoge1xuICAgICAgICAgICAgICBhLFxuICAgICAgICAgICAgICBlOiBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgICAgICAgIGV4cHJlc3Npb246ICdiICsgYycsXG4gICAgICAgICAgICAgICAgdXNpbmdNZXRyaWNzOiB7IGIsIGMgfVxuICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0pXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZ3JhcGhNZXRyaWNzQXJlKHRlc3QsIGdyYXBoLCBbXG4gICAgICAgIFsgeyBsYWJlbDogJ2EgKyBlJywgZXhwcmVzc2lvbjogJ2EgKyBlJyB9IF0sXG4gICAgICAgIFsgJ1Rlc3QnLCAnQUNvdW50JywgeyB2aXNpYmxlOiBmYWxzZSwgaWQ6ICdhJyB9IF0sXG4gICAgICAgIFsgeyBleHByZXNzaW9uOiAnYiArIGMnLCB2aXNpYmxlOiBmYWxzZSwgaWQ6ICdlJyB9IF0sXG4gICAgICAgIFsgJ1Rlc3QnLCAnQkNvdW50JywgeyB2aXNpYmxlOiBmYWxzZSwgaWQ6ICdiJyB9IF0sXG4gICAgICAgIFsgJ1Rlc3QnLCAnQ0NvdW50JywgeyB2aXNpYmxlOiBmYWxzZSwgaWQ6ICdjJyB9IF1cbiAgICAgIF0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBhZGQgdGhlIHNhbWUgbWV0cmljIHVuZGVyIGRpZmZlcmVudCBpZHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IGdyYXBoID0gbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgbGVmdDogW1xuICAgICAgICAgIG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAgICAgICBleHByZXNzaW9uOiAnYSArIGUnLFxuICAgICAgICAgICAgdXNpbmdNZXRyaWNzOiB7XG4gICAgICAgICAgICAgIGEsXG4gICAgICAgICAgICAgIGU6IG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAgICAgICAgICAgZXhwcmVzc2lvbjogJ2IgKyBjJyxcbiAgICAgICAgICAgICAgICB1c2luZ01ldHJpY3M6IHsgYjogYSwgYyB9XG4gICAgICAgICAgICAgIH0pXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSlcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICBncmFwaE1ldHJpY3NBcmUodGVzdCwgZ3JhcGgsIFtcbiAgICAgICAgWyB7IGxhYmVsOiAnYSArIGUnLCBleHByZXNzaW9uOiAnYSArIGUnIH0gXSxcbiAgICAgICAgWyAnVGVzdCcsICdBQ291bnQnLCB7IHZpc2libGU6IGZhbHNlLCBpZDogJ2EnIH0gXSxcbiAgICAgICAgWyB7IGV4cHJlc3Npb246ICdiICsgYycsIHZpc2libGU6IGZhbHNlLCBpZDogJ2UnIH0gXSxcbiAgICAgICAgWyAnVGVzdCcsICdBQ291bnQnLCB7IHZpc2libGU6IGZhbHNlLCBpZDogJ2InIH0gXSxcbiAgICAgICAgWyAnVGVzdCcsICdDQ291bnQnLCB7IHZpc2libGU6IGZhbHNlLCBpZDogJ2MnIH0gXVxuICAgICAgXSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIHJldXNlIGlkZW50aWZpZXJzIGluIE1hdGhFeHByZXNzaW9ucyBpZiBtZXRyaWNzIGFyZSB0aGUgc2FtZScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3QgZ3JhcGggPSBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICAgIGV4cHJlc3Npb246ICdhICsgZScsXG4gICAgICAgICAgICB1c2luZ01ldHJpY3M6IHtcbiAgICAgICAgICAgICAgYSxcbiAgICAgICAgICAgICAgZTogbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICAgICAgICBleHByZXNzaW9uOiAnYSArIGMnLFxuICAgICAgICAgICAgICAgIHVzaW5nTWV0cmljczogeyBhLCBjIH1cbiAgICAgICAgICAgICAgfSlcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9KVxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGdyYXBoTWV0cmljc0FyZSh0ZXN0LCBncmFwaCwgW1xuICAgICAgICBbIHsgbGFiZWw6ICdhICsgZScsIGV4cHJlc3Npb246ICdhICsgZScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0FDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYScgfSBdLFxuICAgICAgICBbIHsgZXhwcmVzc2lvbjogJ2EgKyBjJywgdmlzaWJsZTogZmFsc2UsIGlkOiAnZScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0NDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYycgfSBdXG4gICAgICBdKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdNYXRoRXhwcmVzc2lvbiBhbmQgaXRzIGNvbnN0aXR1ZW50IG1ldHJpY3MgY2FuIGJvdGggYmUgYWRkZWQgdG8gYSBncmFwaCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3QgZ3JhcGggPSBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgYSxcbiAgICAgICAgICBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgICAgZXhwcmVzc2lvbjogJ2EgKyBiJyxcbiAgICAgICAgICAgIHVzaW5nTWV0cmljczogeyBhLCBiIH1cbiAgICAgICAgICB9KVxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGdyYXBoTWV0cmljc0FyZSh0ZXN0LCBncmFwaCwgW1xuICAgICAgICBbICdUZXN0JywgJ0FDb3VudCcsIHsgaWQ6ICdhJyB9IF0sXG4gICAgICAgIFsgeyBsYWJlbDogJ2EgKyBiJywgZXhwcmVzc2lvbjogJ2EgKyBiJyB9IF0sXG4gICAgICAgIFsgJ1Rlc3QnLCAnQkNvdW50JywgeyB2aXNpYmxlOiBmYWxzZSwgaWQ6ICdiJyB9IF1cbiAgICAgIF0pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdNYXRoRXhwcmVzc2lvbiBjb250cm9scyBwZXJpb2Qgb2YgbWV0cmljcyBkaXJlY3RseSB1c2VkIGluIGl0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBDaGVjayB0aGF0IGlmIHdlIGFkZCBBIHdpdGggeyBwZXJpb2Q6IDEwcyB9IHRvIGEgbWF0aGV4cHJlc3Npb24gb2YgcGVyaW9kIDVtXG4gICAgICAvLyB0aGVuIHR3byBtZXRyaWMgbGluZXMgYXJlIGFkZGVkIGZvciBBLCBvbmUgYXQgMTBzIGFuZCBvbmUgYXQgNW1cbiAgICAgIGNvbnN0IGdyYXBoID0gbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgbGVmdDogW1xuICAgICAgICAgIGEud2l0aCh7IHBlcmlvZDogRHVyYXRpb24uc2Vjb25kcygxMCkgfSksXG4gICAgICAgICAgbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICAgIGV4cHJlc3Npb246ICdhICsgYicsXG4gICAgICAgICAgICB1c2luZ01ldHJpY3M6IHsgYTogYS53aXRoKHsgcGVyaW9kOiBEdXJhdGlvbi5zZWNvbmRzKDEwKSB9KSwgYiB9XG4gICAgICAgICAgfSlcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBncmFwaE1ldHJpY3NBcmUodGVzdCwgZ3JhcGgsIFtcbiAgICAgICAgWyAnVGVzdCcsICdBQ291bnQnLCB7IHBlcmlvZDogMTAgfSBdLFxuICAgICAgICBbIHsgbGFiZWw6ICdhICsgYicsIGV4cHJlc3Npb246ICdhICsgYicgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0FDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0JDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYicgfSBdXG4gICAgICBdKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnTWF0aEV4cHJlc3Npb24gY29udHJvbHMgcGVyaW9kIG9mIG1ldHJpY3MgdHJhbnNpdGl2ZWx5IHVzZWQgaW4gaXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIFNhbWUgYXMgdGhlIHByZXZpb3VzIHRlc3QsIGJ1dCByZWN1cnNpdmVseVxuXG4gICAgICBjb25zdCBncmFwaCA9IG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgICAgZXhwcmVzc2lvbjogJ2EgKyBlJyxcbiAgICAgICAgICAgIHVzaW5nTWV0cmljczoge1xuICAgICAgICAgICAgICBhLFxuICAgICAgICAgICAgICBlOiBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgICAgICAgIGV4cHJlc3Npb246ICdhICsgYicsXG4gICAgICAgICAgICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgICAgICAgICAgIHVzaW5nTWV0cmljczogeyBhLCBiIH1cbiAgICAgICAgICAgICAgfSlcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9KVxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGdyYXBoTWV0cmljc0FyZSh0ZXN0LCBncmFwaCwgW1xuICAgICAgICBbIHsgZXhwcmVzc2lvbjogJ2EgKyBlJywgbGFiZWw6ICdhICsgZScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0FDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYScgfSBdLFxuICAgICAgICBbIHsgZXhwcmVzc2lvbjogJ2EgKyBiJywgdmlzaWJsZTogZmFsc2UsIGlkOiAnZScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0JDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYicgfSBdXG4gICAgICBdKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIHVzZSBwZXJjZW50aWxlcyBpbiBleHByZXNzaW9uIG1ldHJpY3MgaW4gZ3JhcGhzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgZ3JhcGggPSBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICBsZWZ0OiBbXG4gICAgICAgICAgbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICAgIGV4cHJlc3Npb246ICdhICsgYjk5JyxcbiAgICAgICAgICAgIHVzaW5nTWV0cmljczogeyBhLCBiOTkgfVxuICAgICAgICAgIH0pXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZ3JhcGhNZXRyaWNzQXJlKHRlc3QsIGdyYXBoLCBbXG4gICAgICAgIFsgeyBleHByZXNzaW9uOiAnYSArIGI5OScsIGxhYmVsOiAnYSArIGI5OScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0FDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYScgfSBdLFxuICAgICAgICBbICdUZXN0JywgJ0JDb3VudCcsIHsgdmlzaWJsZTogZmFsc2UsIGlkOiAnYjk5Jywgc3RhdDogJ3A5OScgfSBdLFxuICAgICAgXSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG5cbiAgJ2luIGFsYXJtcyc6IHtcbiAgICAnTWF0aEV4cHJlc3Npb25zIGNhbiBiZSB1c2VkIGZvciBhbiBhbGFybScodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIG5ldyBBbGFybShzdGFjaywgJ0FsYXJtJywge1xuICAgICAgICB0aHJlc2hvbGQ6IDEsIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgICAgICBtZXRyaWM6IG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAgICAgZXhwcmVzc2lvbjogJ2EgKyBiJyxcbiAgICAgICAgICB1c2luZ01ldHJpY3M6IHsgYSwgYiB9XG4gICAgICAgIH0pXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgYWxhcm1NZXRyaWNzQXJlKFtcbiAgICAgICAge1xuICAgICAgICAgIEV4cHJlc3Npb246IFwiYSArIGJcIixcbiAgICAgICAgICBJZDogXCJleHByXzFcIlxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgSWQ6IFwiYVwiLFxuICAgICAgICAgIE1ldHJpY1N0YXQ6IHtcbiAgICAgICAgICAgIE1ldHJpYzoge1xuICAgICAgICAgICAgICBNZXRyaWNOYW1lOiBcIkFDb3VudFwiLFxuICAgICAgICAgICAgICBOYW1lc3BhY2U6IFwiVGVzdFwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgUGVyaW9kOiAzMDAsXG4gICAgICAgICAgICBTdGF0OiBcIkF2ZXJhZ2VcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmV0dXJuRGF0YTogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIElkOiBcImJcIixcbiAgICAgICAgICBNZXRyaWNTdGF0OiB7XG4gICAgICAgICAgICBNZXRyaWM6IHtcbiAgICAgICAgICAgICAgTWV0cmljTmFtZTogXCJCQ291bnRcIixcbiAgICAgICAgICAgICAgTmFtZXNwYWNlOiBcIlRlc3RcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBlcmlvZDogMzAwLFxuICAgICAgICAgICAgU3RhdDogXCJBdmVyYWdlXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFJldHVybkRhdGE6IGZhbHNlXG4gICAgICAgIH1cblxuICAgICAgXSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIG5lc3QgTWF0aEV4cHJlc3Npb25zIGluIGFuIGFsYXJtJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgbmV3IEFsYXJtKHN0YWNrLCAnQWxhcm0nLCB7XG4gICAgICAgIHRocmVzaG9sZDogMSwgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgICAgIG1ldHJpYzogbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICBleHByZXNzaW9uOiAnYSArIGUnLFxuICAgICAgICAgIHVzaW5nTWV0cmljczoge1xuICAgICAgICAgICAgYSxcbiAgICAgICAgICAgIGU6IG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAgICAgICAgIGV4cHJlc3Npb246ICdiICsgYycsXG4gICAgICAgICAgICAgIHVzaW5nTWV0cmljczogeyBiLCBjIH1cbiAgICAgICAgICAgIH0pXG4gICAgICAgICAgfVxuICAgICAgICB9KVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGFsYXJtTWV0cmljc0FyZShbXG4gICAgICAgIHtcbiAgICAgICAgICBFeHByZXNzaW9uOiBcImEgKyBlXCIsXG4gICAgICAgICAgSWQ6IFwiZXhwcl8xXCJcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIElkOiBcImFcIixcbiAgICAgICAgICBNZXRyaWNTdGF0OiB7XG4gICAgICAgICAgICBNZXRyaWM6IHtcbiAgICAgICAgICAgICAgTWV0cmljTmFtZTogXCJBQ291bnRcIixcbiAgICAgICAgICAgICAgTmFtZXNwYWNlOiBcIlRlc3RcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBlcmlvZDogMzAwLFxuICAgICAgICAgICAgU3RhdDogXCJBdmVyYWdlXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFJldHVybkRhdGE6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBFeHByZXNzaW9uOiBcImIgKyBjXCIsXG4gICAgICAgICAgSWQ6IFwiZVwiLFxuICAgICAgICAgIFJldHVybkRhdGE6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBJZDogXCJiXCIsXG4gICAgICAgICAgTWV0cmljU3RhdDoge1xuICAgICAgICAgICAgTWV0cmljOiB7XG4gICAgICAgICAgICAgIE1ldHJpY05hbWU6IFwiQkNvdW50XCIsXG4gICAgICAgICAgICAgIE5hbWVzcGFjZTogXCJUZXN0XCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBQZXJpb2Q6IDMwMCxcbiAgICAgICAgICAgIFN0YXQ6IFwiQXZlcmFnZVwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBSZXR1cm5EYXRhOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgSWQ6IFwiY1wiLFxuICAgICAgICAgIE1ldHJpY1N0YXQ6IHtcbiAgICAgICAgICAgIE1ldHJpYzoge1xuICAgICAgICAgICAgICBNZXRyaWNOYW1lOiBcIkNDb3VudFwiLFxuICAgICAgICAgICAgICBOYW1lc3BhY2U6IFwiVGVzdFwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgUGVyaW9kOiAzMDAsXG4gICAgICAgICAgICBTdGF0OiBcIkF2ZXJhZ2VcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmV0dXJuRGF0YTogZmFsc2VcbiAgICAgICAgfVxuICAgICAgXSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnTWF0aEV4cHJlc3Npb24gY29udHJvbHMgcGVyaW9kIG9mIG1ldHJpY3MgdHJhbnNpdGl2ZWx5IHVzZWQgaW4gaXQgd2l0aCBhbGFybXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBuZXcgQWxhcm0oc3RhY2ssICdBbGFybScsIHtcbiAgICAgICAgdGhyZXNob2xkOiAxLCBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgICAgbWV0cmljOiBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgIGV4cHJlc3Npb246ICdhICsgZScsXG4gICAgICAgICAgdXNpbmdNZXRyaWNzOiB7XG4gICAgICAgICAgICBhLFxuICAgICAgICAgICAgZTogbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICAgICAgZXhwcmVzc2lvbjogJ2IgKyBjJyxcbiAgICAgICAgICAgICAgdXNpbmdNZXRyaWNzOiB7IGIsIGMgfSxcbiAgICAgICAgICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDEpXG4gICAgICAgICAgICB9KVxuICAgICAgICAgIH0sXG4gICAgICAgICAgcGVyaW9kOiBEdXJhdGlvbi5zZWNvbmRzKDMwKVxuICAgICAgICB9KVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGFsYXJtTWV0cmljc0FyZShbXG4gICAgICAgIHtcbiAgICAgICAgICBFeHByZXNzaW9uOiBcImEgKyBlXCIsXG4gICAgICAgICAgSWQ6IFwiZXhwcl8xXCJcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIElkOiBcImFcIixcbiAgICAgICAgICBNZXRyaWNTdGF0OiB7XG4gICAgICAgICAgICBNZXRyaWM6IHtcbiAgICAgICAgICAgICAgTWV0cmljTmFtZTogXCJBQ291bnRcIixcbiAgICAgICAgICAgICAgTmFtZXNwYWNlOiBcIlRlc3RcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBlcmlvZDogMzAsXG4gICAgICAgICAgICBTdGF0OiBcIkF2ZXJhZ2VcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmV0dXJuRGF0YTogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIEV4cHJlc3Npb246IFwiYiArIGNcIixcbiAgICAgICAgICBJZDogXCJlXCIsXG4gICAgICAgICAgUmV0dXJuRGF0YTogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIElkOiBcImJcIixcbiAgICAgICAgICBNZXRyaWNTdGF0OiB7XG4gICAgICAgICAgICBNZXRyaWM6IHtcbiAgICAgICAgICAgICAgTWV0cmljTmFtZTogXCJCQ291bnRcIixcbiAgICAgICAgICAgICAgTmFtZXNwYWNlOiBcIlRlc3RcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBlcmlvZDogMzAsXG4gICAgICAgICAgICBTdGF0OiBcIkF2ZXJhZ2VcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmV0dXJuRGF0YTogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIElkOiBcImNcIixcbiAgICAgICAgICBNZXRyaWNTdGF0OiB7XG4gICAgICAgICAgICBNZXRyaWM6IHtcbiAgICAgICAgICAgICAgTWV0cmljTmFtZTogXCJDQ291bnRcIixcbiAgICAgICAgICAgICAgTmFtZXNwYWNlOiBcIlRlc3RcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBlcmlvZDogMzAsXG4gICAgICAgICAgICBTdGF0OiBcIkF2ZXJhZ2VcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmV0dXJuRGF0YTogZmFsc2VcbiAgICAgICAgfVxuICAgICAgXSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYW5ub3RhdGlvbiBmb3IgYSBtYXRoZXhwcmVzc2lvbiBhbGFybSBpcyBjYWxjdWxhdGVkIGJhc2VkIHVwb24gY29uc3RpdHVlbnQgbWV0cmljcycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGFsYXJtID0gbmV3IEFsYXJtKHN0YWNrLCAnQWxhcm0nLCB7XG4gICAgICAgIHRocmVzaG9sZDogMSwgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgICAgIG1ldHJpYzogbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgICBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMTApLFxuICAgICAgICAgIGV4cHJlc3Npb246ICdhICsgYicsXG4gICAgICAgICAgdXNpbmdNZXRyaWNzOiB7IGEsIGI6IGIud2l0aCh7IHBlcmlvZDogRHVyYXRpb24ubWludXRlcygyMCkgfSkgfSAvLyBUaGlzIGlzIG92ZXJyaWRkZW5cbiAgICAgICAgfSlcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBhbGFybUxhYmVsID0gYWxhcm0udG9Bbm5vdGF0aW9uKCkubGFiZWw7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZXF1YWxzKGFsYXJtTGFiZWwsICdhICsgYiA+PSAxIGZvciAxIGRhdGFwb2ludHMgd2l0aGluIDEwIG1pbnV0ZXMnKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdjYW4gdXNlIHBlcmNlbnRpbGVzIGluIGV4cHJlc3Npb24gbWV0cmljcyBpbiBhbGFybXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBuZXcgQWxhcm0oc3RhY2ssICdBbGFybScsIHtcbiAgICAgICAgdGhyZXNob2xkOiAxLCBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgICAgbWV0cmljOiBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICAgIGV4cHJlc3Npb246ICdhICsgYjk5JyxcbiAgICAgICAgICB1c2luZ01ldHJpY3M6IHsgYSwgYjk5IH1cbiAgICAgICAgfSlcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBhbGFybU1ldHJpY3NBcmUoW1xuICAgICAgICB7XG4gICAgICAgICAgRXhwcmVzc2lvbjogXCJhICsgYjk5XCIsXG4gICAgICAgICAgSWQ6IFwiZXhwcl8xXCJcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIElkOiBcImFcIixcbiAgICAgICAgICBNZXRyaWNTdGF0OiB7XG4gICAgICAgICAgICBNZXRyaWM6IHtcbiAgICAgICAgICAgICAgTWV0cmljTmFtZTogXCJBQ291bnRcIixcbiAgICAgICAgICAgICAgTmFtZXNwYWNlOiBcIlRlc3RcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBlcmlvZDogMzAwLFxuICAgICAgICAgICAgU3RhdDogXCJBdmVyYWdlXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFJldHVybkRhdGE6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBJZDogXCJiOTlcIixcbiAgICAgICAgICBNZXRyaWNTdGF0OiB7XG4gICAgICAgICAgICBNZXRyaWM6IHtcbiAgICAgICAgICAgICAgTWV0cmljTmFtZTogXCJCQ291bnRcIixcbiAgICAgICAgICAgICAgTmFtZXNwYWNlOiBcIlRlc3RcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBlcmlvZDogMzAwLFxuICAgICAgICAgICAgU3RhdDogXCJwOTlcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmV0dXJuRGF0YTogZmFsc2VcbiAgICAgICAgfVxuICAgICAgXSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH1cbn07XG5cbmZ1bmN0aW9uIGdyYXBoTWV0cmljc0FyZSh0ZXN0OiBUZXN0LCB3OiBJV2lkZ2V0LCBtZXRyaWNzOiBhbnlbXSkge1xuICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHcudG9Kc29uKCkpLCBbIHtcbiAgICB0eXBlOiAnbWV0cmljJyxcbiAgICB3aWR0aDogNixcbiAgICBoZWlnaHQ6IDYsXG4gICAgcHJvcGVydGllczpcbiAgICB7IHZpZXc6ICd0aW1lU2VyaWVzJyxcbiAgICAgIHJlZ2lvbjogeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgIG1ldHJpY3MsXG4gICAgICB5QXhpczoge30gfSB9XSk7XG59XG5cbmZ1bmN0aW9uIGFsYXJtTWV0cmljc0FyZShtZXRyaWNzOiBhbnlbXSkge1xuICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q2xvdWRXYXRjaDo6QWxhcm0nLCB7XG4gICAgTWV0cmljczogbWV0cmljc1xuICB9KSk7XG59XG4iXX0=