"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const a = new lib_1.Metric({ namespace: 'Test', metricName: 'ACount' });
let stack1;
let stack2;
function graphMetricsAre(test, stack, w, metrics) {
    test.deepEqual(stack.resolve(w.toJson()), [{
            type: 'metric',
            width: 6,
            height: 6,
            properties: { view: 'timeSeries',
                region: { Ref: 'AWS::Region' },
                metrics,
                yAxis: {} }
        }]);
}
module.exports = {
    'setUp'(cb) {
        stack1 = new core_1.Stack(undefined, undefined, { env: { region: 'pluto', account: '1234' } });
        stack2 = new core_1.Stack(undefined, undefined, { env: { region: 'mars', account: '5678' } });
        cb();
    },
    'in graphs': {
        'metric attached to stack1 will not render region and account in stack1'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.attachTo(stack1)
                ],
            });
            // THEN
            graphMetricsAre(test, stack1, graph, [
                ['Test', 'ACount'],
            ]);
            test.done();
        },
        'metric attached to stack1 will render region and account in stack2'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.attachTo(stack1)
                ],
            });
            // THEN
            graphMetricsAre(test, stack2, graph, [
                ['Test', 'ACount', { region: 'pluto', accountId: '1234' }],
            ]);
            test.done();
        },
        'metric with explicit account and region will render in environment agnostic stack'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.with({ account: '1234', region: 'us-north-5' })
                ],
            });
            // THEN
            graphMetricsAre(test, new core_1.Stack(), graph, [
                ['Test', 'ACount', { accountId: '1234', region: 'us-north-5' }],
            ]);
            test.done();
        },
        'metric attached to agnostic stack will not render in agnostic stack'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.attachTo(new core_1.Stack()),
                ],
            });
            // THEN
            graphMetricsAre(test, new core_1.Stack(), graph, [
                ['Test', 'ACount'],
            ]);
            test.done();
        },
    },
    'in alarms': {
        'metric attached to stack1 will not render region and account in stack1'(test) {
            // GIVEN
            new lib_1.Alarm(stack1, 'Alarm', {
                threshold: 1, evaluationPeriods: 1,
                metric: a.attachTo(stack1),
            });
            // THEN
            assert_1.expect(stack1).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
                MetricName: "ACount",
                Namespace: "Test",
                Period: 300,
            }));
            test.done();
        },
        'metric attached to stack1 will throw in stack2'(test) {
            // Cross-region/cross-account metrics are supported in Dashboards but not in Alarms
            // GIVEN
            test.throws(() => {
                new lib_1.Alarm(stack2, 'Alarm', {
                    threshold: 1, evaluationPeriods: 1,
                    metric: a.attachTo(stack1),
                });
            }, /Cannot create an Alarm in region 'mars' based on metric 'ACount' in 'pluto'/);
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,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