"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const metric_types_1 = require("../metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.AVERAGE,
        avg: metric_types_1.Statistic.AVERAGE,
        minimum: metric_types_1.Statistic.MINIMUM,
        min: metric_types_1.Statistic.MINIMUM,
        maximum: metric_types_1.Statistic.MAXIMUM,
        max: metric_types_1.Statistic.MAXIMUM,
        samplecount: metric_types_1.Statistic.SAMPLE_COUNT,
        n: metric_types_1.Statistic.SAMPLE_COUNT,
        sum: metric_types_1.Statistic.SUM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat]
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1])
        };
    }
    throw new Error(`Not a valid statistic: '${stat}', must be one of Average | Minimum | Maximum | SampleCount | Sum | pNN.NN`);
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,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