"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const drop_empty_object_at_the_end_of_an_array_token_1 = require("./drop-empty-object-at-the-end-of-an-array-token");
const env_tokens_1 = require("./env-tokens");
const metric_util_1 = require("./metric-util");
const object_1 = require("./object");
/**
 * Return the JSON structure which represents these metrics in a graph.
 *
 * Depending on the metric type (stat or expression), one `Metric` object
 * can render to multiple time series.
 *
 * - Top-level metrics will be rendered visibly, additionally added metrics will
 *   be rendered invisibly.
 * - IDs used in math expressions need to be either globally unique, or refer to the same
 *   metric object.
 *
 * This will be called by GraphWidget, no need for clients to call this.
 */
function allMetricsGraphJson(left, right) {
    // Add metrics to a set which will automatically expand them recursively,
    // making sure to retain conflicting the visible one on conflicting metrics objects.
    const mset = new MetricSet();
    mset.addTopLevel('left', ...left);
    mset.addTopLevel('right', ...right);
    // Render all metrics from the set.
    return mset.entries.map(entry => new drop_empty_object_at_the_end_of_an_array_token_1.DropEmptyObjectAtTheEndOfAnArray(metricGraphJson(entry.metric, entry.tag, entry.id)));
}
exports.allMetricsGraphJson = allMetricsGraphJson;
function metricGraphJson(metric, yAxis, id) {
    const config = metric.toMetricConfig();
    const ret = [];
    const options = { ...config.renderingProperties };
    metric_util_1.dispatchMetric(metric, {
        withStat(stat) {
            ret.push(stat.namespace, stat.metricName);
            // Dimensions
            for (const dim of (stat.dimensions || [])) {
                ret.push(dim.name, dim.value);
            }
            // Metric attributes that are rendered to graph options
            if (stat.account) {
                options.accountId = env_tokens_1.accountIfDifferentFromStack(stat.account);
            }
            if (stat.region) {
                options.region = env_tokens_1.regionIfDifferentFromStack(stat.region);
            }
            if (stat.period && stat.period.toSeconds() !== 300) {
                options.period = stat.period.toSeconds();
            }
            if (stat.statistic && stat.statistic !== 'Average') {
                options.stat = stat.statistic;
            }
        },
        withExpression(expr) {
            options.expression = expr.expression;
        }
    });
    // Options
    if (!yAxis) {
        options.visible = false;
    }
    if (yAxis !== 'left') {
        options.yAxis = yAxis;
    }
    if (id) {
        options.id = id;
    }
    // If math expressions don't have a label (or an ID), they'll render with an unelegant
    // autogenerated id ("metric_alias0"). Our ids may in the future also be autogenerated,
    // so if an ME doesn't have a label, use its toString() as the label (renders the expression).
    if (options.visible !== false && options.expression && !options.label) {
        options.label = metric.toString();
    }
    const renderedOpts = object_1.dropUndefined(options);
    if (Object.keys(renderedOpts).length !== 0) {
        ret.push(renderedOpts);
    }
    return ret;
}
/**
 * Contain a set of metrics, expanding math expressions
 *
 * "Primary" metrics (added via a top-level call) can be tagged with an additional value.
 */
class MetricSet {
    constructor() {
        this.metrics = new Array();
        this.metricById = new Map();
        this.metricByKey = new Map();
    }
    /**
     * Add the given set of metrics to this set
     */
    addTopLevel(tag, ...metrics) {
        for (const metric of metrics) {
            this.addOne(metric, tag);
        }
    }
    /**
     * Access all the accumulated timeseries entries
     */
    get entries() {
        return this.metrics;
    }
    /**
     * Add a metric into the set
     *
     * The id may not be the same as a previous metric added, unless it's the same metric.
     *
     * It can be made visible, in which case the new "metric" object replaces the old
     * one (and the new ones "renderingPropertieS" will be honored instead of the old
     * one's).
     */
    addOne(metric, tag, id) {
        var _a;
        const key = metric_util_1.metricKey(metric);
        let existingEntry;
        // Try lookup existing by id if we have one
        if (id) {
            existingEntry = this.metricById.get(id);
            if (existingEntry && metric_util_1.metricKey(existingEntry.metric) !== key) {
                throw new Error(`Can't happen, already checked elsewhere`);
            }
        }
        if (!existingEntry) {
            // Try lookup by metric if we didn't find one by id
            existingEntry = this.metricByKey.get(key);
            // If the one we found already has an id, it must be different from the id
            // we're trying to add and we want to add a new metric. Pretend we didn't
            // find one.
            if (((_a = existingEntry) === null || _a === void 0 ? void 0 : _a.id) && id) {
                existingEntry = undefined;
            }
        }
        // Create a new entry if we didn't find one so far
        let entry;
        if (existingEntry) {
            entry = existingEntry;
        }
        else {
            entry = { metric };
            this.metrics.push(entry);
            this.metricByKey.set(key, entry);
        }
        // If it didn't have an id but now we do, add one
        if (!entry.id && id) {
            entry.id = id;
            this.metricById.set(id, entry);
        }
        // If it didn't have a tag but now we do, add one
        if (!entry.tag && tag) {
            entry.tag = tag;
        }
        // Recurse and add children
        const conf = metric.toMetricConfig();
        if (conf.mathExpression) {
            for (const [subId, subMetric] of Object.entries(conf.mathExpression.usingMetrics)) {
                this.addOne(subMetric, undefined, subId);
            }
        }
    }
}
exports.MetricSet = MetricSet;
//# sourceMappingURL=data:application/json;base64,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