"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ShellScriptAction = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
/**
 * Validate a revision using shell commands
 */
class ShellScriptAction {
    constructor(props) {
        var _a, _b;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: props.actionName,
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [],
            outputs: [],
        };
        if (Object.keys((_a = props.useOutputs) !== null && _a !== void 0 ? _a : {}).length + ((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).length === 0) {
            throw new Error('You must supply either \'useOutputs\' or \'additionalArtifacts\', since a CodeBuild Action must always have at least one input artifact.');
        }
    }
    /**
     * The CodeBuild Project's principal
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d, _e;
        const inputs = new Array();
        inputs.push(...(_a = this.props.additionalArtifacts) !== null && _a !== void 0 ? _a : []);
        const envVarCommands = new Array();
        const bashOptions = (_b = this.props.bashOptions) !== null && _b !== void 0 ? _b : '-eu';
        if (bashOptions) {
            envVarCommands.push(`set ${bashOptions}`);
        }
        for (const [varName, output] of Object.entries((_c = this.props.useOutputs) !== null && _c !== void 0 ? _c : {})) {
            const outputArtifact = output.artifactFile;
            // Add the artifact to the list of inputs, if it's not in there already. Determine
            // the location where CodeBuild is going to stick it based on whether it's the first (primary)
            // input or an 'extra input', then parse.
            let artifactIndex = inputs.findIndex(a => a.artifactName === outputArtifact.artifact.artifactName);
            if (artifactIndex === -1) {
                artifactIndex = inputs.push(outputArtifact.artifact) - 1;
            }
            const dirEnv = artifactIndex === 0 ? 'CODEBUILD_SRC_DIR' : `CODEBUILD_SRC_DIR_${outputArtifact.artifact.artifactName}`;
            envVarCommands.push(`export ${varName}="$(node -pe 'require(process.env.${dirEnv} + "/${outputArtifact.fileName}")["${output.outputName}"]')"`);
        }
        this._project = new codebuild.PipelineProject(scope, 'Project', {
            environment: { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0 },
            vpc: this.props.vpc,
            subnetSelection: this.props.subnetSelection,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    build: {
                        commands: [
                            ...envVarCommands,
                            ...this.props.commands,
                        ],
                    },
                },
            }),
        });
        for (const statement of (_d = this.props.rolePolicyStatements) !== null && _d !== void 0 ? _d : []) {
            this._project.addToRolePolicy(statement);
        }
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.props.actionName,
            input: inputs[0],
            extraInputs: inputs.slice(1),
            runOrder: (_e = this.props.runOrder) !== null && _e !== void 0 ? _e : 100,
            project: this._project,
        });
        // Replace the placeholder actionProperties at the last minute
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
    }
    /**
     * Project generated to run the shell script in
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after ShellScriptAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.ShellScriptAction = ShellScriptAction;
//# sourceMappingURL=data:application/json;base64,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