"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const crypto = require("crypto");
const path = require("path");
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const construct_internals_1 = require("../private/construct-internals");
const _util_1 = require("./_util");
/**
 * A standard synth with a generated buildspec
 */
class SimpleSynthAction {
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        if (this.props.installCommand && this.props.installCommands) {
            throw new Error('Pass either \'installCommand\' or \'installCommands\', but not both');
        }
        if (this.props.buildCommand && this.props.buildCommands) {
            throw new Error('Pass either \'buildCommand\' or \'buildCommands\', but not both');
        }
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * Create a standard NPM synth action
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * Create a standard Yarn synth action
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Project generated to run the synth command
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after SimpleSynthAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d;
        const buildCommands = (_a = this.props.buildCommands) !== null && _a !== void 0 ? _a : [this.props.buildCommand];
        const installCommands = (_b = this.props.installCommands) !== null && _b !== void 0 ? _b : [this.props.installCommand];
        const testCommands = (_c = this.props.testCommands) !== null && _c !== void 0 ? _c : [];
        const synthCommand = this.props.synthCommand;
        const buildSpec = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    commands: _util_1.filterEmpty([
                        this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                        ...installCommands,
                    ]),
                },
                build: {
                    commands: _util_1.filterEmpty([
                        ...buildCommands,
                        ...testCommands,
                        synthCommand,
                    ]),
                },
            },
            artifacts: renderArtifacts(this),
        });
        const environment = { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0, ...this.props.environment };
        const environmentVariables = {
            ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
            ...this.props.environmentVariables,
        };
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = hash(core_1.Stack.of(scope).resolve({
            environment,
            buildSpecString: buildSpec.toBuildSpec(),
            environmentVariables,
        }));
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: this.props.projectName,
            environment,
            vpc: this.props.vpc,
            subnetSelection: this.props.subnetSelection,
            buildSpec,
            environmentVariables,
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._project = project;
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_d = this.props.additionalArtifacts) !== null && _d !== void 0 ? _d : []).map(a => a.artifact)],
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: {
                _PROJECT_CONFIG_HASH: { value: projectConfigHash },
            },
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            var _a;
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * The CodeBuild Project's principal
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
function hash(obj) {
    const d = crypto.createHash('sha256');
    d.update(JSON.stringify(obj));
    return d.digest('hex');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2ltcGxlLXN5bnRoLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNpbXBsZS1zeW50aC1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsaUNBQWlDO0FBQ2pDLDZCQUE2QjtBQUM3QixvREFBb0QsQ0FBQyx5REFBeUQ7QUFDOUcsMERBQTBELENBQUMsNERBQTREO0FBQ3ZILDBFQUEwRSxDQUFDLG9FQUFvRTtBQUkvSSx3Q0FBaUQsQ0FBQyxnREFBZ0Q7QUFDbEcsd0VBQTJFO0FBQzNFLG1DQUFnRTtBQTZKaEU7O0dBRUc7QUFDSCxNQUFhLGlCQUFpQjtJQW9DMUIsWUFBNkIsS0FBNkI7O1FBQTdCLFVBQUssR0FBTCxLQUFLLENBQXdCO1FBQ3RELG1GQUFtRjtRQUNuRixzRUFBc0U7UUFDdEUsRUFBRTtRQUNGLGVBQWU7UUFDZixhQUFhO1FBQ2IsV0FBVztRQUNYLGFBQWE7UUFDYixTQUFTO1FBQ1QsVUFBVTtRQUNWLElBQUksQ0FBQyxpQkFBaUIsR0FBRztZQUNyQixVQUFVLFFBQUUsS0FBSyxDQUFDLFVBQVUsbUNBQUksT0FBTztZQUN2QyxRQUFRLEVBQUUsWUFBWSxDQUFDLGNBQWMsQ0FBQyxLQUFLO1lBQzNDLFFBQVEsRUFBRSxXQUFXO1lBQ3JCLGNBQWMsRUFBRSxFQUFFLFNBQVMsRUFBRSxDQUFDLEVBQUUsU0FBUyxFQUFFLENBQUMsRUFBRSxVQUFVLEVBQUUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUU7WUFDNUUsTUFBTSxFQUFFLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUM5QixPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMscUJBQXFCLEVBQUUsR0FBRyxPQUFDLEtBQUssQ0FBQyxtQkFBbUIsbUNBQUksRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3BHLENBQUM7UUFDRixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQ3pELE1BQU0sSUFBSSxLQUFLLENBQUMscUVBQXFFLENBQUMsQ0FBQztTQUMxRjtRQUNELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO1NBQ3RGO1FBQ0QsTUFBTSxLQUFLLFNBQUcsS0FBSyxDQUFDLG1CQUFtQixtQ0FBSSxFQUFFLENBQUM7UUFDOUMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDL0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxZQUFZLEVBQUU7Z0JBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUhBQXVILENBQUMsQ0FBQzthQUM1STtZQUNELEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxFQUFFO2dCQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7b0JBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMseUdBQXlHLENBQUMsQ0FBQztpQkFDOUg7YUFDSjtTQUNKO0lBQ0wsQ0FBQztJQXRFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsT0FBZ0M7O1FBQzNELE9BQU8sSUFBSSxpQkFBaUIsQ0FBQztZQUN6QixHQUFHLE9BQU87WUFDVixjQUFjLFFBQUUsT0FBTyxDQUFDLGNBQWMsbUNBQUksUUFBUTtZQUNsRCxZQUFZLFFBQUUsT0FBTyxDQUFDLFlBQVksbUNBQUksZUFBZTtZQUNyRCxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7WUFDaEIsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlO1NBQzNDLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsaUJBQWlCLENBQUMsT0FBaUM7O1FBQzdELE9BQU8sSUFBSSxpQkFBaUIsQ0FBQztZQUN6QixHQUFHLE9BQU87WUFDVixjQUFjLFFBQUUsT0FBTyxDQUFDLGNBQWMsbUNBQUksZ0NBQWdDO1lBQzFFLFlBQVksUUFBRSxPQUFPLENBQUMsWUFBWSxtQ0FBSSxlQUFlO1lBQ3JELEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBRztZQUNoQixlQUFlLEVBQUUsT0FBTyxDQUFDLGVBQWU7U0FDM0MsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQXdDRDs7T0FFRztJQUNILElBQVcsZ0JBQWdCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDO0lBQ2xDLENBQUM7SUFDRDs7T0FFRztJQUNILElBQVcsT0FBTztRQUNkLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkVBQTZFLENBQUMsQ0FBQztTQUNsRztRQUNELE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUN6QixDQUFDO0lBQ0Q7O09BRUc7SUFDSSxJQUFJLENBQUMsS0FBZ0IsRUFBRSxLQUEwQixFQUFFLE9BQXVDOztRQUM3RixNQUFNLGFBQWEsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsbUNBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQzVFLE1BQU0sZUFBZSxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxtQ0FBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDbEYsTUFBTSxZQUFZLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLG1DQUFJLEVBQUUsQ0FBQztRQUNuRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUM3QyxNQUFNLFNBQVMsR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQztZQUM3QyxPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRTtnQkFDSixTQUFTLEVBQUU7b0JBQ1AsUUFBUSxFQUFFLG1CQUFXLENBQUM7d0JBQ2xCLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7d0JBQzlELEdBQUcsZUFBZTtxQkFDckIsQ0FBQztpQkFDTDtnQkFDRCxLQUFLLEVBQUU7b0JBQ0gsUUFBUSxFQUFFLG1CQUFXLENBQUM7d0JBQ2xCLEdBQUcsYUFBYTt3QkFDaEIsR0FBRyxZQUFZO3dCQUNmLFlBQVk7cUJBQ2YsQ0FBQztpQkFDTDthQUNKO1lBQ0QsU0FBUyxFQUFFLGVBQWUsQ0FBQyxJQUFJLENBQUM7U0FDbkMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxXQUFXLEdBQUcsRUFBRSxVQUFVLEVBQUUsU0FBUyxDQUFDLGVBQWUsQ0FBQyxZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQ3RHLE1BQU0sb0JBQW9CLEdBQUc7WUFDekIsR0FBRyxnQ0FBd0IsQ0FBQyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsd0JBQXdCLElBQUksRUFBRSxDQUFDO1lBQ3pFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxvQkFBb0I7U0FDckMsQ0FBQztRQUNGLCtFQUErRTtRQUMvRSwyRkFBMkY7UUFDM0YsNEVBQTRFO1FBQzVFLG1CQUFtQjtRQUNuQixNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNuRCxXQUFXO1lBQ1gsZUFBZSxFQUFFLFNBQVMsQ0FBQyxXQUFXLEVBQUU7WUFDeEMsb0JBQW9CO1NBQ3ZCLENBQUMsQ0FBQyxDQUFDO1FBQ0osTUFBTSxPQUFPLEdBQUcsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtZQUNwRSxXQUFXLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO1lBQ25DLFdBQVc7WUFDWCxHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHO1lBQ25CLGVBQWUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWU7WUFDM0MsU0FBUztZQUNULG9CQUFvQjtTQUN2QixDQUFDLENBQUM7UUFDSCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLEtBQUssU0FBUyxFQUFFO1lBQy9DLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxFQUFFO2dCQUN0RCxPQUFPLENBQUMsZUFBZSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1lBQzdDLENBQUMsQ0FBQyxDQUFDO1NBQ047UUFDRCxJQUFJLENBQUMsUUFBUSxHQUFHLE9BQU8sQ0FBQztRQUN4QixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksb0JBQW9CLENBQUMsZUFBZSxDQUFDO1lBQ3BELFVBQVUsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtZQUM1QyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjO1lBQ2hDLE9BQU8sRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMscUJBQXFCLEVBQUUsR0FBRyxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLG1DQUFJLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUMzRyxpRkFBaUY7WUFDakYsdURBQXVEO1lBQ3ZELHdFQUF3RTtZQUN4RSw4REFBOEQ7WUFDOUQsb0JBQW9CLEVBQUU7Z0JBQ2xCLG9CQUFvQixFQUFFLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2FBQ3JEO1lBQ0QsT0FBTztTQUNWLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNoRCxTQUFTLGVBQWUsQ0FBQyxJQUF1Qjs7WUFDNUMsa0RBQWtEO1lBQ2xELHlGQUF5RjtZQUN6RixvQ0FBb0M7WUFDcEMsTUFBTSxvQkFBb0IsR0FBRztnQkFDekIsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLElBQUksT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksbUNBQUksR0FBRyxFQUFFLCtDQUF5QixDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUM3RixPQUFPLEVBQUUsTUFBTTthQUNsQixDQUFDO1lBQ0YsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixFQUFFO2dCQUNoQyxNQUFNLFNBQVMsR0FBd0IsRUFBRSxDQUFDO2dCQUMxQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxZQUFZLEVBQUU7b0JBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsdUZBQXVGLENBQUMsQ0FBQztpQkFDNUc7Z0JBQ0QsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsb0JBQW9CLENBQUM7Z0JBQ2hGLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUU7O29CQUMzQyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7d0JBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsMENBQTBDLENBQUMsQ0FBQztxQkFDL0Q7b0JBQ0QsU0FBUyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLEdBQUc7d0JBQ25DLGdCQUFnQixFQUFFLElBQUksQ0FBQyxJQUFJLE9BQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLG1DQUFJLEdBQUcsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDO3dCQUMxRSxPQUFPLEVBQUUsTUFBTTtxQkFDbEIsQ0FBQztnQkFDTixDQUFDLENBQUMsQ0FBQztnQkFDSCxPQUFPLEVBQUUscUJBQXFCLEVBQUUsU0FBUyxFQUFFLENBQUM7YUFDL0M7WUFDRCxPQUFPLG9CQUFvQixDQUFDO1FBQ2hDLENBQUM7SUFDTCxDQUFDO0lBQ0Q7O09BRUc7SUFDSCxJQUFXLGNBQWM7UUFDckIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztJQUN2QyxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxhQUFhLENBQUMsSUFBWSxFQUFFLE1BQTJCLEVBQUUsT0FBMEI7UUFDdEYsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDZixNQUFNLElBQUksS0FBSyxDQUFDLG1CQUFtQixDQUFDLENBQUM7U0FDeEM7UUFDRCxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDN0QsQ0FBQztDQUNKO0FBeE1ELDhDQXdNQztBQXlERCxTQUFTLElBQUksQ0FBSSxHQUFNO0lBQ25CLE1BQU0sQ0FBQyxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDdEMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDOUIsT0FBTyxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzNCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSAnY3J5cHRvJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBjb2RlYnVpbGQgZnJvbSBcIi4uLy4uLy4uL2F3cy1jb2RlYnVpbGRcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlYnVpbGQnXG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSBcIi4uLy4uLy4uL2F3cy1jb2RlcGlwZWxpbmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnXG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmVfYWN0aW9ucyBmcm9tIFwiLi4vLi4vLi4vYXdzLWNvZGVwaXBlbGluZS1hY3Rpb25zXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lLWFjdGlvbnMnXG5pbXBvcnQgKiBhcyBlYzIgZnJvbSBcIi4uLy4uLy4uL2F3cy1lYzJcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInXG5pbXBvcnQgKiBhcyBldmVudHMgZnJvbSBcIi4uLy4uLy4uL2F3cy1ldmVudHNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnXG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIFN0YWNrIH0gZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBjbG91ZEFzc2VtYmx5QnVpbGRTcGVjRGlyIH0gZnJvbSAnLi4vcHJpdmF0ZS9jb25zdHJ1Y3QtaW50ZXJuYWxzJztcbmltcG9ydCB7IGNvcHlFbnZpcm9ubWVudFZhcmlhYmxlcywgZmlsdGVyRW1wdHkgfSBmcm9tICcuL191dGlsJztcbi8qKlxuICogQ29uZmlndXJhdGlvbiBvcHRpb25zIGZvciBhIFNpbXBsZVN5bnRoXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2ltcGxlU3ludGhPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgc291cmNlIGFydGlmYWN0IG9mIHRoZSBDb2RlUGlwZWxpbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBzb3VyY2VBcnRpZmFjdDogY29kZXBpcGVsaW5lLkFydGlmYWN0O1xuICAgIC8qKlxuICAgICAqIFRoZSBhcnRpZmFjdCB3aGVyZSB0aGUgQ2xvdWRBc3NlbWJseSBzaG91bGQgYmUgZW1pdHRlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNsb3VkQXNzZW1ibHlBcnRpZmFjdDogY29kZXBpcGVsaW5lLkFydGlmYWN0O1xuICAgIC8qKlxuICAgICAqIEVudmlyb25tZW50IHZhcmlhYmxlcyB0byBzZW5kIGludG8gYnVpbGRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBlbnZpcm9ubWVudCB2YXJpYWJsZXNcbiAgICAgKi9cbiAgICByZWFkb25seSBlbnZpcm9ubWVudFZhcmlhYmxlcz86IFJlY29yZDxzdHJpbmcsIGNvZGVidWlsZC5CdWlsZEVudmlyb25tZW50VmFyaWFibGU+O1xuICAgIC8qKlxuICAgICAqIEVudmlyb25tZW50IHZhcmlhYmxlcyB0byBjb3B5IG92ZXIgZnJvbSBwYXJlbnQgZW52XG4gICAgICpcbiAgICAgKiBUaGVzZSBhcmUgZW52aXJvbm1lbnQgdmFyaWFibGVzIHRoYXQgYXJlIGJlaW5nIHVzZWQgYnkgdGhlIGJ1aWxkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBlbnZpcm9ubWVudCB2YXJpYWJsZXMgY29waWVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29weUVudmlyb25tZW50VmFyaWFibGVzPzogc3RyaW5nW107XG4gICAgLyoqXG4gICAgICogTmFtZSBvZiB0aGUgYnVpbGQgYWN0aW9uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAnU3ludGgnXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWN0aW9uTmFtZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBOYW1lIG9mIHRoZSBDb2RlQnVpbGQgcHJvamVjdFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBdXRvbWF0aWNhbGx5IGdlbmVyYXRlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByb2plY3ROYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEJ1aWxkIGVudmlyb25tZW50IHRvIHVzZSBmb3IgQ29kZUJ1aWxkIGpvYlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgQnVpbGRFbnZpcm9ubWVudC5MaW51eEJ1aWxkSW1hZ2UuU1RBTkRBUkRfNF8wXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiBjb2RlYnVpbGQuQnVpbGRFbnZpcm9ubWVudDtcbiAgICAvKipcbiAgICAgKiBEaXJlY3RvcnkgaW5zaWRlIHRoZSBzb3VyY2Ugd2hlcmUgcGFja2FnZS5qc29uIGFuZCBjZGsuanNvbiBhcmUgbG9jYXRlZFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBSZXBvc2l0b3J5IHJvb3RcbiAgICAgKi9cbiAgICByZWFkb25seSBzdWJkaXJlY3Rvcnk/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogUHJvZHVjZSBhZGRpdGlvbmFsIG91dHB1dCBhcnRpZmFjdHMgYWZ0ZXIgdGhlIGJ1aWxkIGJhc2VkIG9uIHRoZSBnaXZlbiBkaXJlY3Rvcmllc1xuICAgICAqXG4gICAgICogQ2FuIGJlIHVzZWQgdG8gcHJvZHVjZSBhZGRpdGlvbmFsIGFydGlmYWN0cyBkdXJpbmcgdGhlIGJ1aWxkIHN0ZXAsXG4gICAgICogc2VwYXJhdGUgZnJvbSB0aGUgY2xvdWQgYXNzZW1ibHksIHdoaWNoIGNhbiBiZSB1c2VkIGZ1cnRoZXIgb24gaW4gdGhlXG4gICAgICogcGlwZWxpbmUuXG4gICAgICpcbiAgICAgKiBEaXJlY3RvcmllcyBhcmUgZXZhbHVhdGVkIHdpdGggcmVzcGVjdCB0byBgc3ViZGlyZWN0b3J5YC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBhcnRpZmFjdHMgZ2VuZXJhdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWRkaXRpb25hbEFydGlmYWN0cz86IEFkZGl0aW9uYWxBcnRpZmFjdFtdO1xuICAgIC8qKlxuICAgICAqIFBvbGljeSBzdGF0ZW1lbnRzIHRvIGFkZCB0byByb2xlIHVzZWQgZHVyaW5nIHRoZSBzeW50aFxuICAgICAqXG4gICAgICogQ2FuIGJlIHVzZWQgdG8gYWRkIGFjY2VzIHRvIGEgQ29kZUFydGlmYWN0IHJlcG9zaXRvcnkgZXRjLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBwb2xpY3kgc3RhdGVtZW50cyBhZGRlZCB0byBDb2RlQnVpbGQgUHJvamVjdCBSb2xlXG4gICAgICovXG4gICAgcmVhZG9ubHkgcm9sZVBvbGljeVN0YXRlbWVudHM/OiBpYW0uUG9saWN5U3RhdGVtZW50W107XG4gICAgLyoqXG4gICAgICogVGhlIFZQQyB3aGVyZSB0byBleGVjdXRlIHRoZSBTaW1wbGVTeW50aC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gVlBDXG4gICAgICovXG4gICAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG4gICAgLyoqXG4gICAgICogV2hpY2ggc3VibmV0cyB0byB1c2UuXG4gICAgICpcbiAgICAgKiBPbmx5IHVzZWQgaWYgJ3ZwYycgaXMgc3VwcGxpZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFsbCBwcml2YXRlIHN1Ym5ldHMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BzIGZvciBTaW1wbGVTeW50aEFjdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNpbXBsZVN5bnRoQWN0aW9uUHJvcHMgZXh0ZW5kcyBTaW1wbGVTeW50aE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBzeW50aCBjb21tYW5kXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3ludGhDb21tYW5kOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGluc3RhbGwgY29tbWFuZFxuICAgICAqXG4gICAgICogSWYgbm90IHByb3ZpZGVkIGJ5IHRoZSBidWlsZCBpbWFnZSBvciBhbm90aGVyIGRlcGVuZGVuY3lcbiAgICAgKiBtYW5hZ2VtZW50IHRvb2wsIGF0IGxlYXN0IGluc3RhbGwgdGhlIENESyBDTEkgaGVyZSB1c2luZ1xuICAgICAqIGBucG0gaW5zdGFsbCAtZyBhd3MtY2RrYC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gaW5zdGFsbCByZXF1aXJlZFxuICAgICAqIEBkZXByZWNhdGVkIFVzZSBgaW5zdGFsbENvbW1hbmRzYCBpbnN0ZWFkXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFsbENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGJ1aWxkIGNvbW1hbmRcbiAgICAgKlxuICAgICAqIElmIHlvdXIgcHJvZ3JhbW1pbmcgbGFuZ3VhZ2UgcmVxdWlyZXMgYSBjb21waWxhdGlvbiBzdGVwLCBwdXQgdGhlXG4gICAgICogY29tcGlsYXRpb24gY29tbWFuZCBoZXJlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBidWlsZCByZXF1aXJlZFxuICAgICAqIEBkZXByZWNhdGVkIFVzZSBgYnVpbGRDb21tYW5kc2AgaW5zdGVhZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1aWxkQ29tbWFuZD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBJbnN0YWxsIGNvbW1hbmRzXG4gICAgICpcbiAgICAgKiBJZiBub3QgcHJvdmlkZWQgYnkgdGhlIGJ1aWxkIGltYWdlIG9yIGFub3RoZXIgZGVwZW5kZW5jeVxuICAgICAqIG1hbmFnZW1lbnQgdG9vbCwgYXQgbGVhc3QgaW5zdGFsbCB0aGUgQ0RLIENMSSBoZXJlIHVzaW5nXG4gICAgICogYG5wbSBpbnN0YWxsIC1nIGF3cy1jZGtgLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBpbnN0YWxsIHJlcXVpcmVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFsbENvbW1hbmRzPzogc3RyaW5nW107XG4gICAgLyoqXG4gICAgICogVGhlIGJ1aWxkIGNvbW1hbmRzXG4gICAgICpcbiAgICAgKiBJZiB5b3VyIHByb2dyYW1taW5nIGxhbmd1YWdlIHJlcXVpcmVzIGEgY29tcGlsYXRpb24gc3RlcCwgcHV0IHRoZVxuICAgICAqIGNvbXBpbGF0aW9uIGNvbW1hbmQgaGVyZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gYnVpbGQgcmVxdWlyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBidWlsZENvbW1hbmRzPzogc3RyaW5nW107XG4gICAgLyoqXG4gICAgICogVGVzdCBjb21tYW5kc1xuICAgICAqXG4gICAgICogVGhlc2UgY29tbWFuZHMgYXJlIHJ1biBhZnRlciB0aGUgYnVpbGQgY29tbWFuZHMgYnV0IGJlZm9yZSB0aGVcbiAgICAgKiBzeW50aCBjb21tYW5kLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyB0ZXN0IGNvbW1hbmRzXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGVzdENvbW1hbmRzPzogc3RyaW5nW107XG59XG4vKipcbiAqIFNwZWNpZmljYXRpb24gb2YgYW4gYWRkaXRpb25hbCBhcnRpZmFjdCB0byBnZW5lcmF0ZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkZGl0aW9uYWxBcnRpZmFjdCB7XG4gICAgLyoqXG4gICAgICogRGlyZWN0b3J5IHRvIGJlIHBhY2thZ2VkXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGlyZWN0b3J5OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogQXJ0aWZhY3QgdG8gcmVwcmVzZW50IHRoZSBidWlsZCBkaXJlY3RvcnkgaW4gdGhlIHBpcGVsaW5lXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXJ0aWZhY3Q6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbn1cbi8qKlxuICogQSBzdGFuZGFyZCBzeW50aCB3aXRoIGEgZ2VuZXJhdGVkIGJ1aWxkc3BlY1xuICovXG5leHBvcnQgY2xhc3MgU2ltcGxlU3ludGhBY3Rpb24gaW1wbGVtZW50cyBjb2RlcGlwZWxpbmUuSUFjdGlvbiwgaWFtLklHcmFudGFibGUge1xuICAgIC8qKlxuICAgICAqIENyZWF0ZSBhIHN0YW5kYXJkIE5QTSBzeW50aCBhY3Rpb25cbiAgICAgKlxuICAgICAqIFVzZXMgYG5wbSBjaWAgdG8gaW5zdGFsbCBkZXBlbmRlbmNpZXMgYW5kIGBucHggY2RrIHN5bnRoYCB0byBzeW50aGVzaXplLlxuICAgICAqXG4gICAgICogSWYgeW91IG5lZWQgYSBidWlsZCBzdGVwLCBhZGQgYGJ1aWxkQ29tbWFuZDogJ25wbSBydW4gYnVpbGQnYC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHN0YW5kYXJkTnBtU3ludGgob3B0aW9uczogU3RhbmRhcmROcG1TeW50aE9wdGlvbnMpIHtcbiAgICAgICAgcmV0dXJuIG5ldyBTaW1wbGVTeW50aEFjdGlvbih7XG4gICAgICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICAgICAgaW5zdGFsbENvbW1hbmQ6IG9wdGlvbnMuaW5zdGFsbENvbW1hbmQgPz8gJ25wbSBjaScsXG4gICAgICAgICAgICBzeW50aENvbW1hbmQ6IG9wdGlvbnMuc3ludGhDb21tYW5kID8/ICducHggY2RrIHN5bnRoJyxcbiAgICAgICAgICAgIHZwYzogb3B0aW9ucy52cGMsXG4gICAgICAgICAgICBzdWJuZXRTZWxlY3Rpb246IG9wdGlvbnMuc3VibmV0U2VsZWN0aW9uLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlIGEgc3RhbmRhcmQgWWFybiBzeW50aCBhY3Rpb25cbiAgICAgKlxuICAgICAqIFVzZXMgYHlhcm4gaW5zdGFsbCAtLWZyb3plbi1sb2NrZmlsZWAgdG8gaW5zdGFsbCBkZXBlbmRlbmNpZXMgYW5kIGBucHggY2RrIHN5bnRoYCB0byBzeW50aGVzaXplLlxuICAgICAqXG4gICAgICogSWYgeW91IG5lZWQgYSBidWlsZCBzdGVwLCBhZGQgYGJ1aWxkQ29tbWFuZDogJ3lhcm4gYnVpbGQnYC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHN0YW5kYXJkWWFyblN5bnRoKG9wdGlvbnM6IFN0YW5kYXJkWWFyblN5bnRoT3B0aW9ucykge1xuICAgICAgICByZXR1cm4gbmV3IFNpbXBsZVN5bnRoQWN0aW9uKHtcbiAgICAgICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAgICAgICBpbnN0YWxsQ29tbWFuZDogb3B0aW9ucy5pbnN0YWxsQ29tbWFuZCA/PyAneWFybiBpbnN0YWxsIC0tZnJvemVuLWxvY2tmaWxlJyxcbiAgICAgICAgICAgIHN5bnRoQ29tbWFuZDogb3B0aW9ucy5zeW50aENvbW1hbmQgPz8gJ25weCBjZGsgc3ludGgnLFxuICAgICAgICAgICAgdnBjOiBvcHRpb25zLnZwYyxcbiAgICAgICAgICAgIHN1Ym5ldFNlbGVjdGlvbjogb3B0aW9ucy5zdWJuZXRTZWxlY3Rpb24sXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICBwcml2YXRlIF9hY3Rpb24/OiBjb2RlcGlwZWxpbmVfYWN0aW9ucy5Db2RlQnVpbGRBY3Rpb247XG4gICAgcHJpdmF0ZSBfYWN0aW9uUHJvcGVydGllczogY29kZXBpcGVsaW5lLkFjdGlvblByb3BlcnRpZXM7XG4gICAgcHJpdmF0ZSBfcHJvamVjdD86IGNvZGVidWlsZC5JUHJvamVjdDtcbiAgICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTaW1wbGVTeW50aEFjdGlvblByb3BzKSB7XG4gICAgICAgIC8vIEEgbnVtYmVyIG9mIGFjdGlvblByb3BlcnRpZXMgZ2V0IHJlYWQgYmVmb3JlIGJpbmQoKSBpcyBldmVuIGNhbGxlZCAoc28gYmVmb3JlIHdlXG4gICAgICAgIC8vIGhhdmUgbWFkZSB0aGUgUHJvamVjdCBhbmQgY2FuIGNvbnN0cnVjdCB0aGUgYWN0dWFsIENvZGVCdWlsZEFjdGlvbilcbiAgICAgICAgLy9cbiAgICAgICAgLy8gLSBhY3Rpb25OYW1lXG4gICAgICAgIC8vIC0gcmVzb3VyY2VcbiAgICAgICAgLy8gLSByZWdpb25cbiAgICAgICAgLy8gLSBjYXRlZ29yeVxuICAgICAgICAvLyAtIHJvbGVcbiAgICAgICAgLy8gLSBvd25lclxuICAgICAgICB0aGlzLl9hY3Rpb25Qcm9wZXJ0aWVzID0ge1xuICAgICAgICAgICAgYWN0aW9uTmFtZTogcHJvcHMuYWN0aW9uTmFtZSA/PyAnU3ludGgnLFxuICAgICAgICAgICAgY2F0ZWdvcnk6IGNvZGVwaXBlbGluZS5BY3Rpb25DYXRlZ29yeS5CVUlMRCxcbiAgICAgICAgICAgIHByb3ZpZGVyOiAnQ29kZUJ1aWxkJyxcbiAgICAgICAgICAgIGFydGlmYWN0Qm91bmRzOiB7IG1pbklucHV0czogMCwgbWF4SW5wdXRzOiA1LCBtaW5PdXRwdXRzOiAwLCBtYXhPdXRwdXRzOiA1IH0sXG4gICAgICAgICAgICBpbnB1dHM6IFtwcm9wcy5zb3VyY2VBcnRpZmFjdF0sXG4gICAgICAgICAgICBvdXRwdXRzOiBbcHJvcHMuY2xvdWRBc3NlbWJseUFydGlmYWN0LCAuLi4ocHJvcHMuYWRkaXRpb25hbEFydGlmYWN0cyA/PyBbXSkubWFwKGEgPT4gYS5hcnRpZmFjdCldLFxuICAgICAgICB9O1xuICAgICAgICBpZiAodGhpcy5wcm9wcy5pbnN0YWxsQ29tbWFuZCAmJiB0aGlzLnByb3BzLmluc3RhbGxDb21tYW5kcykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdQYXNzIGVpdGhlciBcXCdpbnN0YWxsQ29tbWFuZFxcJyBvciBcXCdpbnN0YWxsQ29tbWFuZHNcXCcsIGJ1dCBub3QgYm90aCcpO1xuICAgICAgICB9XG4gICAgICAgIGlmICh0aGlzLnByb3BzLmJ1aWxkQ29tbWFuZCAmJiB0aGlzLnByb3BzLmJ1aWxkQ29tbWFuZHMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUGFzcyBlaXRoZXIgXFwnYnVpbGRDb21tYW5kXFwnIG9yIFxcJ2J1aWxkQ29tbWFuZHNcXCcsIGJ1dCBub3QgYm90aCcpO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGFkZGxzID0gcHJvcHMuYWRkaXRpb25hbEFydGlmYWN0cyA/PyBbXTtcbiAgICAgICAgaWYgKE9iamVjdC5rZXlzKGFkZGxzKS5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICBpZiAoIXByb3BzLmNsb3VkQXNzZW1ibHlBcnRpZmFjdC5hcnRpZmFjdE5hbWUpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBtdXN0IGdpdmUgYWxsIG91dHB1dCBhcnRpZmFjdHMsIGluY2x1ZGluZyB0aGUgXFwnY2xvdWRBc3NlbWJseUFydGlmYWN0XFwnLCBuYW1lcyB3aGVuIHVzaW5nIFxcJ2FkZGl0aW9uYWxBcnRpZmFjdHNcXCcnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGZvciAoY29uc3QgYWRkbCBvZiBhZGRscykge1xuICAgICAgICAgICAgICAgIGlmICghYWRkbC5hcnRpZmFjdC5hcnRpZmFjdE5hbWUpIHtcbiAgICAgICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBnaXZlIGFsbCBvdXRwdXQgYXJ0aWZhY3RzIHBhc3NlZCB0byBTaW1wbGVTeW50aEFjdGlvbiBuYW1lcyB3aGVuIHVzaW5nIFxcJ2FkZGl0aW9uYWxBcnRpZmFjdHNcXCcnKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogRXhpc3RzIHRvIGltcGxlbWVudCBJQWN0aW9uXG4gICAgICovXG4gICAgcHVibGljIGdldCBhY3Rpb25Qcm9wZXJ0aWVzKCk6IGNvZGVwaXBlbGluZS5BY3Rpb25Qcm9wZXJ0aWVzIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuX2FjdGlvblByb3BlcnRpZXM7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFByb2plY3QgZ2VuZXJhdGVkIHRvIHJ1biB0aGUgc3ludGggY29tbWFuZFxuICAgICAqL1xuICAgIHB1YmxpYyBnZXQgcHJvamVjdCgpOiBjb2RlYnVpbGQuSVByb2plY3Qge1xuICAgICAgICBpZiAoIXRoaXMuX3Byb2plY3QpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUHJvamVjdCBiZWNvbWVzIGF2YWlsYWJsZSBhZnRlciBTaW1wbGVTeW50aEFjdGlvbiBoYXMgYmVlbiBib3VuZCB0byBhIHN0YWdlJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX3Byb2plY3Q7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEV4aXN0cyB0byBpbXBsZW1lbnQgSUFjdGlvblxuICAgICAqL1xuICAgIHB1YmxpYyBiaW5kKHNjb3BlOiBDb25zdHJ1Y3QsIHN0YWdlOiBjb2RlcGlwZWxpbmUuSVN0YWdlLCBvcHRpb25zOiBjb2RlcGlwZWxpbmUuQWN0aW9uQmluZE9wdGlvbnMpOiBjb2RlcGlwZWxpbmUuQWN0aW9uQ29uZmlnIHtcbiAgICAgICAgY29uc3QgYnVpbGRDb21tYW5kcyA9IHRoaXMucHJvcHMuYnVpbGRDb21tYW5kcyA/PyBbdGhpcy5wcm9wcy5idWlsZENvbW1hbmRdO1xuICAgICAgICBjb25zdCBpbnN0YWxsQ29tbWFuZHMgPSB0aGlzLnByb3BzLmluc3RhbGxDb21tYW5kcyA/PyBbdGhpcy5wcm9wcy5pbnN0YWxsQ29tbWFuZF07XG4gICAgICAgIGNvbnN0IHRlc3RDb21tYW5kcyA9IHRoaXMucHJvcHMudGVzdENvbW1hbmRzID8/IFtdO1xuICAgICAgICBjb25zdCBzeW50aENvbW1hbmQgPSB0aGlzLnByb3BzLnN5bnRoQ29tbWFuZDtcbiAgICAgICAgY29uc3QgYnVpbGRTcGVjID0gY29kZWJ1aWxkLkJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgICAgICAgIHZlcnNpb246ICcwLjInLFxuICAgICAgICAgICAgcGhhc2VzOiB7XG4gICAgICAgICAgICAgICAgcHJlX2J1aWxkOiB7XG4gICAgICAgICAgICAgICAgICAgIGNvbW1hbmRzOiBmaWx0ZXJFbXB0eShbXG4gICAgICAgICAgICAgICAgICAgICAgICB0aGlzLnByb3BzLnN1YmRpcmVjdG9yeSA/IGBjZCAke3RoaXMucHJvcHMuc3ViZGlyZWN0b3J5fWAgOiAnJyxcbiAgICAgICAgICAgICAgICAgICAgICAgIC4uLmluc3RhbGxDb21tYW5kcyxcbiAgICAgICAgICAgICAgICAgICAgXSksXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBidWlsZDoge1xuICAgICAgICAgICAgICAgICAgICBjb21tYW5kczogZmlsdGVyRW1wdHkoW1xuICAgICAgICAgICAgICAgICAgICAgICAgLi4uYnVpbGRDb21tYW5kcyxcbiAgICAgICAgICAgICAgICAgICAgICAgIC4uLnRlc3RDb21tYW5kcyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHN5bnRoQ29tbWFuZCxcbiAgICAgICAgICAgICAgICAgICAgXSksXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBhcnRpZmFjdHM6IHJlbmRlckFydGlmYWN0cyh0aGlzKSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGVudmlyb25tZW50ID0geyBidWlsZEltYWdlOiBjb2RlYnVpbGQuTGludXhCdWlsZEltYWdlLlNUQU5EQVJEXzRfMCwgLi4udGhpcy5wcm9wcy5lbnZpcm9ubWVudCB9O1xuICAgICAgICBjb25zdCBlbnZpcm9ubWVudFZhcmlhYmxlcyA9IHtcbiAgICAgICAgICAgIC4uLmNvcHlFbnZpcm9ubWVudFZhcmlhYmxlcyguLi50aGlzLnByb3BzLmNvcHlFbnZpcm9ubWVudFZhcmlhYmxlcyB8fCBbXSksXG4gICAgICAgICAgICAuLi50aGlzLnByb3BzLmVudmlyb25tZW50VmFyaWFibGVzLFxuICAgICAgICB9O1xuICAgICAgICAvLyBBIGhhc2ggb3ZlciB0aGUgdmFsdWVzIHRoYXQgbWFrZSB0aGUgQ29kZUJ1aWxkIFByb2plY3QgdW5pcXVlIChhbmQgbmVjZXNzYXJ5XG4gICAgICAgIC8vIHRvIHJlc3RhcnQgdGhlIHBpcGVsaW5lIGlmIG9uZSBvZiB0aGVtIGNoYW5nZXMpLiBwcm9qZWN0TmFtZSBpcyBub3QgbmVjZXNzYXJ5IHRvIGluY2x1ZGVcbiAgICAgICAgLy8gaGVyZSBiZWNhdXNlIHRoZSBwaXBlbGluZSB3aWxsIGRlZmluaXRlbHkgcmVzdGFydCBpZiBwcm9qZWN0TmFtZSBjaGFuZ2VzLlxuICAgICAgICAvLyAoUmVzb2x2ZSB0b2tlbnMpXG4gICAgICAgIGNvbnN0IHByb2plY3RDb25maWdIYXNoID0gaGFzaChTdGFjay5vZihzY29wZSkucmVzb2x2ZSh7XG4gICAgICAgICAgICBlbnZpcm9ubWVudCxcbiAgICAgICAgICAgIGJ1aWxkU3BlY1N0cmluZzogYnVpbGRTcGVjLnRvQnVpbGRTcGVjKCksXG4gICAgICAgICAgICBlbnZpcm9ubWVudFZhcmlhYmxlcyxcbiAgICAgICAgfSkpO1xuICAgICAgICBjb25zdCBwcm9qZWN0ID0gbmV3IGNvZGVidWlsZC5QaXBlbGluZVByb2plY3Qoc2NvcGUsICdDZGtCdWlsZFByb2plY3QnLCB7XG4gICAgICAgICAgICBwcm9qZWN0TmFtZTogdGhpcy5wcm9wcy5wcm9qZWN0TmFtZSxcbiAgICAgICAgICAgIGVudmlyb25tZW50LFxuICAgICAgICAgICAgdnBjOiB0aGlzLnByb3BzLnZwYyxcbiAgICAgICAgICAgIHN1Ym5ldFNlbGVjdGlvbjogdGhpcy5wcm9wcy5zdWJuZXRTZWxlY3Rpb24sXG4gICAgICAgICAgICBidWlsZFNwZWMsXG4gICAgICAgICAgICBlbnZpcm9ubWVudFZhcmlhYmxlcyxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmICh0aGlzLnByb3BzLnJvbGVQb2xpY3lTdGF0ZW1lbnRzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRoaXMucHJvcHMucm9sZVBvbGljeVN0YXRlbWVudHMuZm9yRWFjaChwb2xpY3lTdGF0ZW1lbnQgPT4ge1xuICAgICAgICAgICAgICAgIHByb2plY3QuYWRkVG9Sb2xlUG9saWN5KHBvbGljeVN0YXRlbWVudCk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLl9wcm9qZWN0ID0gcHJvamVjdDtcbiAgICAgICAgdGhpcy5fYWN0aW9uID0gbmV3IGNvZGVwaXBlbGluZV9hY3Rpb25zLkNvZGVCdWlsZEFjdGlvbih7XG4gICAgICAgICAgICBhY3Rpb25OYW1lOiB0aGlzLmFjdGlvblByb3BlcnRpZXMuYWN0aW9uTmFtZSxcbiAgICAgICAgICAgIGlucHV0OiB0aGlzLnByb3BzLnNvdXJjZUFydGlmYWN0LFxuICAgICAgICAgICAgb3V0cHV0czogW3RoaXMucHJvcHMuY2xvdWRBc3NlbWJseUFydGlmYWN0LCAuLi4odGhpcy5wcm9wcy5hZGRpdGlvbmFsQXJ0aWZhY3RzID8/IFtdKS5tYXAoYSA9PiBhLmFydGlmYWN0KV0sXG4gICAgICAgICAgICAvLyBJbmNsdXNpb24gb2YgdGhlIGhhc2ggaGVyZSB3aWxsIGxlYWQgdG8gdGhlIHBpcGVsaW5lIHN0cnVjdHVyZSBmb3IgYW55IGNoYW5nZXNcbiAgICAgICAgICAgIC8vIG1hZGUgdGhlIGNvbmZpZyBvZiB0aGUgdW5kZXJseWluZyBDb2RlQnVpbGQgUHJvamVjdC5cbiAgICAgICAgICAgIC8vIEhlbmNlLCB0aGUgcGlwZWxpbmUgd2lsbCBiZSByZXN0YXJ0ZWQuIFRoaXMgaXMgbmVjZXNzYXJ5IGlmIHRoZSB1c2Vyc1xuICAgICAgICAgICAgLy8gYWRkcyAoZm9yIGV4YW1wbGUpIGJ1aWxkIG9yIHRlc3QgY29tbWFuZHMgdG8gdGhlIGJ1aWxkc3BlYy5cbiAgICAgICAgICAgIGVudmlyb25tZW50VmFyaWFibGVzOiB7XG4gICAgICAgICAgICAgICAgX1BST0pFQ1RfQ09ORklHX0hBU0g6IHsgdmFsdWU6IHByb2plY3RDb25maWdIYXNoIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgcHJvamVjdCxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuX2FjdGlvblByb3BlcnRpZXMgPSB0aGlzLl9hY3Rpb24uYWN0aW9uUHJvcGVydGllcztcbiAgICAgICAgcmV0dXJuIHRoaXMuX2FjdGlvbi5iaW5kKHNjb3BlLCBzdGFnZSwgb3B0aW9ucyk7XG4gICAgICAgIGZ1bmN0aW9uIHJlbmRlckFydGlmYWN0cyhzZWxmOiBTaW1wbGVTeW50aEFjdGlvbikge1xuICAgICAgICAgICAgLy8gc2F2ZSB0aGUgZ2VuZXJhdGVkIGZpbGVzIGluIHRoZSBvdXRwdXQgYXJ0aWZhY3RcbiAgICAgICAgICAgIC8vIFRoaXMgcGFydCBvZiB0aGUgYnVpbGRzcGVjIGhhcyB0byBsb29rIGNvbXBsZXRlbHkgZGlmZmVyZW50IGRlcGVuZGluZyBvbiB3aGV0aGVyIHdlJ3JlXG4gICAgICAgICAgICAvLyB1c2luZyBzZWNvbmRhcnkgYXJ0aWZhY3RzIG9yIG5vdC5cbiAgICAgICAgICAgIGNvbnN0IGNsb3VkQXNtQXJ0aWZhY3RTcGVjID0ge1xuICAgICAgICAgICAgICAgICdiYXNlLWRpcmVjdG9yeSc6IHBhdGguam9pbihzZWxmLnByb3BzLnN1YmRpcmVjdG9yeSA/PyAnLicsIGNsb3VkQXNzZW1ibHlCdWlsZFNwZWNEaXIoc2NvcGUpKSxcbiAgICAgICAgICAgICAgICAnZmlsZXMnOiAnKiovKicsXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgaWYgKHNlbGYucHJvcHMuYWRkaXRpb25hbEFydGlmYWN0cykge1xuICAgICAgICAgICAgICAgIGNvbnN0IHNlY29uZGFyeTogUmVjb3JkPHN0cmluZywgYW55PiA9IHt9O1xuICAgICAgICAgICAgICAgIGlmICghc2VsZi5wcm9wcy5jbG91ZEFzc2VtYmx5QXJ0aWZhY3QuYXJ0aWZhY3ROYW1lKSB7XG4gICAgICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignV2hlbiB1c2luZyBhZGRpdGlvbmFsIG91dHB1dCBhcnRpZmFjdHMsIHlvdSBtdXN0IGFsc28gbmFtZSB0aGUgQ2xvdWRBc3NlbWJseSBhcnRpZmFjdCcpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBzZWNvbmRhcnlbc2VsZi5wcm9wcy5jbG91ZEFzc2VtYmx5QXJ0aWZhY3QuYXJ0aWZhY3ROYW1lXSA9IGNsb3VkQXNtQXJ0aWZhY3RTcGVjO1xuICAgICAgICAgICAgICAgIHNlbGYucHJvcHMuYWRkaXRpb25hbEFydGlmYWN0cy5mb3JFYWNoKChhcnQpID0+IHtcbiAgICAgICAgICAgICAgICAgICAgaWYgKCFhcnQuYXJ0aWZhY3QuYXJ0aWZhY3ROYW1lKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBtdXN0IGdpdmUgdGhlIG91dHB1dCBhcnRpZmFjdCBhIG5hbWUnKTtcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICBzZWNvbmRhcnlbYXJ0LmFydGlmYWN0LmFydGlmYWN0TmFtZV0gPSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAnYmFzZS1kaXJlY3RvcnknOiBwYXRoLmpvaW4oc2VsZi5wcm9wcy5zdWJkaXJlY3RvcnkgPz8gJy4nLCBhcnQuZGlyZWN0b3J5KSxcbiAgICAgICAgICAgICAgICAgICAgICAgICdmaWxlcyc6ICcqKi8qJyxcbiAgICAgICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICByZXR1cm4geyAnc2Vjb25kYXJ5LWFydGlmYWN0cyc6IHNlY29uZGFyeSB9O1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcmV0dXJuIGNsb3VkQXNtQXJ0aWZhY3RTcGVjO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSBDb2RlQnVpbGQgUHJvamVjdCdzIHByaW5jaXBhbFxuICAgICAqL1xuICAgIHB1YmxpYyBnZXQgZ3JhbnRQcmluY2lwYWwoKTogaWFtLklQcmluY2lwYWwge1xuICAgICAgICByZXR1cm4gdGhpcy5wcm9qZWN0LmdyYW50UHJpbmNpcGFsO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBFeGlzdHMgdG8gaW1wbGVtZW50IElBY3Rpb25cbiAgICAgKi9cbiAgICBwdWJsaWMgb25TdGF0ZUNoYW5nZShuYW1lOiBzdHJpbmcsIHRhcmdldD86IGV2ZW50cy5JUnVsZVRhcmdldCwgb3B0aW9ucz86IGV2ZW50cy5SdWxlUHJvcHMpOiBldmVudHMuUnVsZSB7XG4gICAgICAgIGlmICghdGhpcy5fYWN0aW9uKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ05lZWQgYmluZCgpIGZpcnN0Jyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX2FjdGlvbi5vblN0YXRlQ2hhbmdlKG5hbWUsIHRhcmdldCwgb3B0aW9ucyk7XG4gICAgfVxufVxuLyoqXG4gKiBPcHRpb25zIGZvciBhIGNvbnZlbnRpb24tYmFzZWQgc3ludGggdXNpbmcgTlBNXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3RhbmRhcmROcG1TeW50aE9wdGlvbnMgZXh0ZW5kcyBTaW1wbGVTeW50aE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBpbnN0YWxsIGNvbW1hbmRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICducG0gY2knXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFsbENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGJ1aWxkIGNvbW1hbmRcbiAgICAgKlxuICAgICAqIEJ5IGRlZmF1bHQsIHdlIGFzc3VtZSBOUE0gcHJvamVjdHMgYXJlIGVpdGhlciB3cml0dGVuIGluIEphdmFTY3JpcHQgb3IgYXJlXG4gICAgICogdXNpbmcgYHRzLW5vZGVgLCBzbyBkb24ndCBuZWVkIGEgYnVpbGQgY29tbWFuZC5cbiAgICAgKlxuICAgICAqIE90aGVyd2lzZSwgcHV0IHRoZSBidWlsZCBjb21tYW5kIGhlcmUsIGZvciBleGFtcGxlIGBucG0gcnVuIGJ1aWxkYC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gYnVpbGQgcmVxdWlyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBidWlsZENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHN5bnRoIGNvbW1hbmRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICducHggY2RrIHN5bnRoJ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHN5bnRoQ29tbWFuZD86IHN0cmluZztcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgYSBjb252ZW50aW9uLWJhc2VkIHN5bnRoIHVzaW5nIFlhcm5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdGFuZGFyZFlhcm5TeW50aE9wdGlvbnMgZXh0ZW5kcyBTaW1wbGVTeW50aE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBpbnN0YWxsIGNvbW1hbmRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICd5YXJuIGluc3RhbGwgLS1mcm96ZW4tbG9ja2ZpbGUnXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFsbENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGJ1aWxkIGNvbW1hbmRcbiAgICAgKlxuICAgICAqIEJ5IGRlZmF1bHQsIHdlIGFzc3VtZSBOUE0gcHJvamVjdHMgYXJlIGVpdGhlciB3cml0dGVuIGluIEphdmFTY3JpcHQgb3IgYXJlXG4gICAgICogdXNpbmcgYHRzLW5vZGVgLCBzbyBkb24ndCBuZWVkIGEgYnVpbGQgY29tbWFuZC5cbiAgICAgKlxuICAgICAqIE90aGVyd2lzZSwgcHV0IHRoZSBidWlsZCBjb21tYW5kIGhlcmUsIGZvciBleGFtcGxlIGBucG0gcnVuIGJ1aWxkYC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gYnVpbGQgcmVxdWlyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBidWlsZENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHN5bnRoIGNvbW1hbmRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICducHggY2RrIHN5bnRoJ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHN5bnRoQ29tbWFuZD86IHN0cmluZztcbn1cbmZ1bmN0aW9uIGhhc2g8QT4ob2JqOiBBKSB7XG4gICAgY29uc3QgZCA9IGNyeXB0by5jcmVhdGVIYXNoKCdzaGEyNTYnKTtcbiAgICBkLnVwZGF0ZShKU09OLnN0cmluZ2lmeShvYmopKTtcbiAgICByZXR1cm4gZC5kaWdlc3QoJ2hleCcpO1xufVxuIl19