"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                crossAccountKeys: props.crossAccountKeys,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        this._pipeline.addStage({
            stageName: 'UpdatePipeline',
            actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                    cloudAssemblyInput: this._cloudAssemblyArtifact,
                    pipelineStackName: pipelineStack.stackName,
                    cdkCliVersion: props.cdkCliVersion,
                    projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                })],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
        });
        core_1.Aspects.of(this).add({ visit: () => this._assets.removeAssetsStageIfEmpty() });
    }
    /**
     * The underlying CodePipeline object
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * Access one of the pipeline's stages by stage name
     *
     * You can use this to add more Actions to a stage.
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this.assetRoles = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
        this.pipeline = this.props.pipeline;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
                vpc: this.props.vpc,
                subnetSelection: this.props.subnetSelection,
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGlwZWxpbmUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwaXBlbGluZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsdURBQXVELENBQUMsNERBQTREO0FBRXBILHFDQUFxQyxDQUFDLG1EQUFtRDtBQUN6RixxQ0FBeUcsQ0FBQyxnREFBZ0Q7QUFDMUosdUNBQXVHO0FBQ3ZHLHVFQUF5RTtBQUN6RSxtQ0FBeUY7QUF1RnpGOzs7Ozs7Ozs7OztHQVdHO0FBQ0gsTUFBYSxXQUFZLFNBQVEsZ0JBQVM7SUFNdEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUM3RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBSkosWUFBTyxHQUFlLEVBQUUsQ0FBQztRQUN6QixxQkFBZ0IsR0FBMEMsRUFBRSxDQUFDO1FBSTFFLElBQUksQ0FBQyxVQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDNUIsTUFBTSxJQUFJLEtBQUssQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1NBQy9EO1FBQ0QsSUFBSSxDQUFDLHNCQUFzQixHQUFHLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztRQUMxRCxNQUFNLGFBQWEsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3JDLElBQUksS0FBSyxDQUFDLFlBQVksRUFBRTtZQUNwQixJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7Z0JBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMseUZBQXlGLENBQUMsQ0FBQzthQUM5RztZQUNELElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtnQkFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RkFBNkYsQ0FBQyxDQUFDO2FBQ2xIO1lBQ0QsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1NBQ3ZDO2FBQ0k7WUFDRCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO2dCQUN6RCxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7Z0JBQ2hDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7Z0JBQ3hDLHdCQUF3QixFQUFFLElBQUk7YUFDakMsQ0FBQyxDQUFDO1NBQ047UUFDRCxJQUFJLEtBQUssQ0FBQyxZQUFZLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQzFDLGtGQUFrRjtZQUNsRix5RkFBeUY7WUFDekYsc0ZBQXNGO1lBQ3RGLGdEQUFnRDtZQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLG9IQUFvSCxDQUFDLENBQUM7U0FDekk7UUFDRCxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLEVBQUU7WUFDdEYsTUFBTSxJQUFJLEtBQUssQ0FBQywwRkFBMEYsQ0FBQyxDQUFDO1NBQy9HO1FBQ0QsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3BCLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDO2dCQUNwQixTQUFTLEVBQUUsUUFBUTtnQkFDbkIsT0FBTyxFQUFFLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQzthQUNoQyxDQUFDLENBQUM7U0FDTjtRQUNELElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRTtZQUNuQixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztnQkFDcEIsU0FBUyxFQUFFLE9BQU87Z0JBQ2xCLE9BQU8sRUFBRSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUM7YUFDL0IsQ0FBQyxDQUFDO1NBQ047UUFDRCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztZQUNwQixTQUFTLEVBQUUsZ0JBQWdCO1lBQzNCLE9BQU8sRUFBRSxDQUFDLElBQUksOEJBQW9CLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO29CQUNuRCxrQkFBa0IsRUFBRSxJQUFJLENBQUMsc0JBQXNCO29CQUMvQyxpQkFBaUIsRUFBRSxhQUFhLENBQUMsU0FBUztvQkFDMUMsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO29CQUNsQyxXQUFXLEVBQUUsV0FBVyxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQUUsYUFBYSxDQUFDO2lCQUM5RCxDQUFDLENBQUM7U0FDVixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksZUFBZSxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUU7WUFDL0Msa0JBQWtCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQjtZQUMvQyxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDbEMsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTO1lBQ3hCLFdBQVcsRUFBRSxXQUFXLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRSxVQUFVLENBQUM7WUFDeEQsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO1lBQ2QsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1NBQ3pDLENBQUMsQ0FBQztRQUNILGNBQU8sQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsd0JBQXdCLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0gsSUFBVyxZQUFZO1FBQ25CLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQztJQUMxQixDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLEtBQUssQ0FBQyxTQUFpQjtRQUMxQixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFDRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksbUJBQW1CLENBQUMsUUFBZSxFQUFFLFVBQTJCLEVBQUU7UUFDckUsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDaEQsS0FBSyxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDeEMsT0FBTyxLQUFLLENBQUM7SUFDakIsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLFFBQVEsQ0FBQyxTQUFpQjtRQUM3QixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztZQUMxQyxTQUFTO1NBQ1osQ0FBQyxDQUFDO1FBQ0gsTUFBTSxLQUFLLEdBQUcsSUFBSSxnQkFBUSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDeEMscUJBQXFCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQjtZQUNsRCxhQUFhO1lBQ2IsU0FBUztZQUNULElBQUksRUFBRTtnQkFDRixZQUFZLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQztnQkFDbkUsbUJBQW1CLEVBQUUsQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUM7YUFDekU7U0FDSixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN6QixPQUFPLEtBQUssQ0FBQztJQUNqQixDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLFdBQVcsQ0FBQyxTQUFvQjtRQUNuQyxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQzFDLHNFQUFzRTtZQUN0RSxzQ0FBc0M7WUFDdEMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsWUFBWSxLQUFLLENBQUMsVUFBVSxVQUFVLENBQUMsQ0FBQztTQUMvRztRQUNELE9BQU8sSUFBSSxtQkFBVyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNoSCxDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDTyxRQUFRO1FBQ2QsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUNoQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQztRQUN4QyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixFQUFFLENBQUMsQ0FBQztRQUM3QyxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFDRDs7T0FFRztJQUNILElBQVksWUFBWTtRQUNwQixPQUFPLE9BQU8sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUNPLENBQUMsbUJBQW1CO1FBQ3hCLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUM7UUFDdkMsS0FBSyxNQUFNLFdBQVcsSUFBSSxZQUFZLEVBQUU7WUFDcEMsc0ZBQXNGO1lBQ3RGLEtBQUssTUFBTSxLQUFLLElBQUksV0FBVyxDQUFDLDBCQUEwQixFQUFFO2dCQUN4RCxNQUFNLFNBQVMsR0FBRyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsS0FBSyxLQUFLLENBQUMsQ0FBQztnQkFDdEUsSUFBSSxTQUFTLEtBQUssU0FBUyxFQUFFO29CQUN6QixrQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsVUFBVSxXQUFXLENBQUMsU0FBUyxxQkFBcUI7d0JBQ2hGLElBQUksS0FBSyw4REFBOEQsQ0FBQyxDQUFDO2lCQUNoRjtxQkFDSSxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUMsZUFBZSxHQUFHLFdBQVcsQ0FBQyxlQUFlLENBQUMsRUFBRTtvQkFDakUsTUFBTSxVQUFVLFdBQVcsQ0FBQyxTQUFTLHFCQUFxQjt3QkFDdEQsSUFBSSxTQUFTLENBQUMsU0FBUywrQ0FBK0MsQ0FBQztpQkFDOUU7YUFDSjtTQUNKO0lBQ0wsQ0FBQztJQUNPLENBQUMsd0JBQXdCO1FBQzdCLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQ2xFLEtBQUssTUFBTSxVQUFVLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsRUFBRTtZQUN6RCxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsRUFBRTtnQkFDbkMsTUFBTSxvQ0FBb0MsVUFBVSx3RUFBd0UsQ0FBQzthQUNoSTtTQUNKO0lBQ0wsQ0FBQztDQUNKO0FBdkxELGtDQXVMQztBQUNELFNBQVMsY0FBYyxDQUFDLENBQXVCO0lBQzNDLE9BQU8sQ0FBQyxZQUFZLDhCQUFvQixDQUFDO0FBQzdDLENBQUM7QUFDRCxTQUFTLE9BQU8sQ0FBTyxFQUFPLEVBQUUsQ0FBZ0I7SUFDNUMsT0FBTyxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDcEQsQ0FBQztBQVNEOztHQUVHO0FBQ0gsTUFBTSxlQUFnQixTQUFRLGdCQUFTO0lBUW5DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQW1CLEtBQTJCO1FBQ2xGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFEc0MsVUFBSyxHQUFMLEtBQUssQ0FBc0I7UUFQckUsZUFBVSxHQUF3QyxFQUFFLENBQUM7UUFDckQsZUFBVSxHQUE4QixFQUFFLENBQUM7UUFJcEQsa0JBQWEsR0FBRyxDQUFDLENBQUM7UUFDbEIsb0JBQWUsR0FBRyxDQUFDLENBQUM7UUFHeEIsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLHVDQUFpQixDQUFDLDJCQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN2RSxrRkFBa0Y7UUFDbEYsbUJBQW1CO1FBQ25CLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDbkUsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQztJQUN4QyxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0kscUJBQXFCLENBQUMsT0FBK0I7UUFDeEQsaUZBQWlGO1FBQ2pGLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUNoRix5RkFBeUY7UUFDekYsd0ZBQXdGO1FBQ3hGLHFFQUFxRTtRQUNyRSxJQUFJLFlBQVksQ0FBQyxVQUFVLENBQUMsS0FBSyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRTtZQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixPQUFPLENBQUMsaUJBQWlCLHFEQUFxRCxJQUFJLENBQUMsV0FBVyw4R0FBOEcsQ0FBQyxDQUFDO1NBQ3hQO1FBQ0QsK0hBQStIO1FBQy9ILElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUNyQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQzdDO1FBQ0QsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNULHVHQUF1RztZQUN2RyxtR0FBbUc7WUFDbkcsMENBQTBDO1lBQzFDLEVBQUU7WUFDRiw0RkFBNEY7WUFDNUYsaUNBQWlDO1lBQ2pDLE1BQU0sRUFBRSxHQUFHLE9BQU8sQ0FBQyxTQUFTLEtBQUssbUJBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFlBQVksSUFBSSxDQUFDLGFBQWEsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLGNBQWMsSUFBSSxDQUFDLGVBQWUsRUFBRSxFQUFFLENBQUM7WUFDOUgsZ0ZBQWdGO1lBQ2hGLCtGQUErRjtZQUMvRixzRkFBc0Y7WUFDdEYsTUFBTSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxHQUFHLElBQUksNkJBQW1CLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtnQkFDMUUsVUFBVSxFQUFFLEVBQUU7Z0JBQ2Qsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0I7Z0JBQ2pELGFBQWEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWE7Z0JBQ3ZDLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUztnQkFDNUIsSUFBSSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztnQkFDeEMsR0FBRyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRztnQkFDbkIsZUFBZSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZTthQUM5QyxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUNoQztRQUNELE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFDRDs7T0FFRztJQUNJLHdCQUF3QjtRQUMzQixJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDM0MsNkNBQTZDO1lBQzdDLGlFQUFpRTtZQUNqRSxNQUFNLE1BQU0sR0FBMkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFnQixDQUFDLE9BQU8sQ0FBQztZQUMzRSxNQUFNLEVBQUUsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUN0QyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUMsRUFBRTtnQkFDVCxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQzthQUN4QjtTQUNKO0lBQ0wsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0ssaUJBQWlCLENBQUMsU0FBb0I7UUFDMUMsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQzVCLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUNyQztRQUNELE1BQU0sVUFBVSxHQUFHLFNBQVMsS0FBSyxtQkFBUyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFDNUUsTUFBTSxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLFVBQVUsTUFBTSxFQUFFO1lBQ3RELFFBQVEsRUFBRSxtQkFBWSxDQUFDLGtCQUFrQjtZQUN6QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQy9JLENBQUMsQ0FBQztRQUNILHNCQUFzQjtRQUN0QixNQUFNLFdBQVcsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUN6QyxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxXQUFXO1lBQ3JCLEdBQUcsRUFBRSxHQUFHO1lBQ1IsWUFBWSxFQUFFLGtCQUFrQjtTQUNuQyxDQUFDLENBQUM7UUFDSCxTQUFTLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUMxQyxTQUFTLEVBQUUsQ0FBQyxXQUFXLENBQUM7WUFDeEIsT0FBTyxFQUFFLENBQUMscUJBQXFCLEVBQUUsc0JBQXNCLEVBQUUsbUJBQW1CLENBQUM7U0FDaEYsQ0FBQyxDQUFDLENBQUM7UUFDSiwwQkFBMEI7UUFDMUIsTUFBTSxZQUFZLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDMUMsT0FBTyxFQUFFLFdBQVc7WUFDcEIsUUFBUSxFQUFFLGNBQWM7WUFDeEIsWUFBWSxFQUFFLEdBQUc7U0FDcEIsQ0FBQyxDQUFDO1FBQ0gsU0FBUyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDMUMsT0FBTyxFQUFFO2dCQUNMLDZCQUE2QjtnQkFDN0Isd0JBQXdCO2dCQUN4Qix3QkFBd0I7Z0JBQ3hCLDZCQUE2QjthQUNoQztZQUNELFNBQVMsRUFBRSxDQUFDLFlBQVksQ0FBQztTQUM1QixDQUFDLENBQUMsQ0FBQztRQUNKLHVCQUF1QjtRQUN2QixTQUFTLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUMxQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFO2dCQUNMLDBCQUEwQjtnQkFDMUIsc0JBQXNCO2dCQUN0QixxQkFBcUI7YUFDeEI7U0FDSixDQUFDLENBQUMsQ0FBQztRQUNKLHlCQUF5QjtRQUN6QixNQUFNLFdBQVcsR0FBRyxTQUFTLEtBQUssbUJBQVMsQ0FBQyxZQUFZO1lBQ3BELENBQUMsQ0FBQyw2Q0FBNkM7WUFDL0MsQ0FBQyxDQUFDLDRDQUE0QyxDQUFDO1FBQ25ELFNBQVMsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzFDLE9BQU8sRUFBRSxDQUFDLGdCQUFnQixDQUFDO1lBQzNCLFNBQVMsRUFBRSxDQUFDLFdBQVcsQ0FBQztTQUMzQixDQUFDLENBQUMsQ0FBQztRQUNKLGtCQUFrQjtRQUNsQixJQUFJLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDbEQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztRQUM5RCxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDdEMsQ0FBQztDQUNKO0FBQ0QsU0FBUyxXQUFXLENBQUMsQ0FBcUIsRUFBRSxNQUFjO0lBQ3RELElBQUksQ0FBQyxLQUFLLFNBQVMsRUFBRTtRQUNqQixPQUFPLFNBQVMsQ0FBQztLQUNwQjtJQUNELE9BQU8sR0FBRyxDQUFDLEdBQUcsTUFBTSxFQUFFLENBQUM7QUFDM0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSBcIi4uLy4uL2F3cy1jb2RlcGlwZWxpbmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnXG5pbXBvcnQgKiBhcyBlYzIgZnJvbSBcIi4uLy4uL2F3cy1lYzJcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInXG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgeyBBbm5vdGF0aW9ucywgQXBwLCBDZm5PdXRwdXQsIENvbnN0cnVjdCwgUGh5c2ljYWxOYW1lLCBTdGFjaywgU3RhZ2UsIEFzcGVjdHMgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IEFzc2V0VHlwZSwgRGVwbG95Q2RrU3RhY2tBY3Rpb24sIFB1Ymxpc2hBc3NldHNBY3Rpb24sIFVwZGF0ZVBpcGVsaW5lQWN0aW9uIH0gZnJvbSAnLi9hY3Rpb25zJztcbmltcG9ydCB7IGFwcE9mLCBhc3NlbWJseUJ1aWxkZXJPZiB9IGZyb20gJy4vcHJpdmF0ZS9jb25zdHJ1Y3QtaW50ZXJuYWxzJztcbmltcG9ydCB7IEFkZFN0YWdlT3B0aW9ucywgQXNzZXRQdWJsaXNoaW5nQ29tbWFuZCwgQ2RrU3RhZ2UsIFN0YWNrT3V0cHV0IH0gZnJvbSAnLi9zdGFnZSc7XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgQ2RrUGlwZWxpbmVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDZGtQaXBlbGluZVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgQ29kZVBpcGVsaW5lIGFjdGlvbiB1c2VkIHRvIHJldHJpZXZlIHRoZSBDREsgYXBwJ3Mgc291cmNlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFJlcXVpcmVkIHVubGVzcyBgY29kZVBpcGVsaW5lYCBpcyBnaXZlblxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNvdXJjZUFjdGlvbj86IGNvZGVwaXBlbGluZS5JQWN0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBDb2RlUGlwZWxpbmUgYWN0aW9uIGJ1aWxkIGFuZCBzeW50aGVzaXMgc3RlcCBvZiB0aGUgQ0RLIGFwcFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBSZXF1aXJlZCB1bmxlc3MgYGNvZGVQaXBlbGluZWAgb3IgYHNvdXJjZUFjdGlvbmAgaXMgZ2l2ZW5cbiAgICAgKi9cbiAgICByZWFkb25seSBzeW50aEFjdGlvbj86IGNvZGVwaXBlbGluZS5JQWN0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBhcnRpZmFjdCB5b3UgaGF2ZSBkZWZpbmVkIHRvIGJlIHRoZSBhcnRpZmFjdCB0byBob2xkIHRoZSBjbG91ZEFzc2VtYmx5QXJ0aWZhY3QgZm9yIHRoZSBzeW50aCBhY3Rpb25cbiAgICAgKi9cbiAgICByZWFkb25seSBjbG91ZEFzc2VtYmx5QXJ0aWZhY3Q6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICAvKipcbiAgICAgKiBFeGlzdGluZyBDb2RlUGlwZWxpbmUgdG8gYWRkIGRlcGxveW1lbnQgc3RhZ2VzIHRvXG4gICAgICpcbiAgICAgKiBVc2UgdGhpcyBpZiB5b3Ugd2FudCBtb3JlIGNvbnRyb2wgb3ZlciB0aGUgQ29kZVBpcGVsaW5lIHRoYXQgZ2V0cyBjcmVhdGVkLlxuICAgICAqIFlvdSBjYW4gY2hvb3NlIHRvIG5vdCBwYXNzIHRoaXMgdmFsdWUsIGluIHdoaWNoIGNhc2UgYSBuZXcgQ29kZVBpcGVsaW5lIGlzXG4gICAgICogY3JlYXRlZCB3aXRoIGRlZmF1bHQgc2V0dGluZ3MuXG4gICAgICpcbiAgICAgKiBJZiB5b3UgcGFzcyBhbiBleGlzdGluZyBDb2RlUGlwZWxpbmUsIGl0IHNob3VsZCBzaG91bGQgaGF2ZSBiZWVuIGNyZWF0ZWRcbiAgICAgKiB3aXRoIGByZXN0YXJ0RXhlY3V0aW9uT25VcGRhdGU6IHRydWVgLlxuICAgICAqXG4gICAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV1cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQSBuZXcgQ29kZVBpcGVsaW5lIGlzIGF1dG9tYXRpY2FsbHkgZ2VuZXJhdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29kZVBpcGVsaW5lPzogY29kZXBpcGVsaW5lLlBpcGVsaW5lO1xuICAgIC8qKlxuICAgICAqIE5hbWUgb2YgdGhlIHBpcGVsaW5lXG4gICAgICpcbiAgICAgKiBDYW4gb25seSBiZSBzZXQgaWYgYGNvZGVQaXBlbGluZWAgaXMgbm90IHNldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQSBuYW1lIGlzIGF1dG9tYXRpY2FsbHkgZ2VuZXJhdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGlwZWxpbmVOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIENyZWF0ZSBLTVMga2V5cyBmb3IgY3Jvc3MtYWNjb3VudCBkZXBsb3ltZW50c1xuICAgICAqXG4gICAgICogVGhpcyBjb250cm9scyB3aGV0aGVyIHRoZSBwaXBlbGluZSBpcyBlbmFibGVkIGZvciBjcm9zcy1hY2NvdW50IGRlcGxveW1lbnRzLlxuICAgICAqXG4gICAgICogQ2FuIG9ubHkgYmUgc2V0IGlmIGBjb2RlUGlwZWxpbmVgIGlzIG5vdCBzZXQuXG4gICAgICpcbiAgICAgKiBCeSBkZWZhdWx0IGNyb3NzLWFjY291bnQgZGVwbG95bWVudHMgYXJlIGVuYWJsZWQsIGJ1dCB0aGlzIGZlYXR1cmUgcmVxdWlyZXNcbiAgICAgKiB0aGF0IEtNUyBDdXN0b21lciBNYXN0ZXIgS2V5cyBhcmUgY3JlYXRlZCB3aGljaCBoYXZlIGEgY29zdCBvZiAkMS9tb250aC5cbiAgICAgKlxuICAgICAqIElmIHlvdSBkbyBub3QgbmVlZCBjcm9zcy1hY2NvdW50IGRlcGxveW1lbnRzLCB5b3UgY2FuIHNldCB0aGlzIHRvIGBmYWxzZWAgdG9cbiAgICAgKiBub3QgY3JlYXRlIHRob3NlIGtleXMgYW5kIHNhdmUgb24gdGhhdCBjb3N0ICh0aGUgYXJ0aWZhY3QgYnVja2V0IHdpbGwgYmVcbiAgICAgKiBlbmNyeXB0ZWQgd2l0aCBhbiBBV1MtbWFuYWdlZCBrZXkpLiBIb3dldmVyLCBjcm9zcy1hY2NvdW50IGRlcGxveW1lbnRzIHdpbGxcbiAgICAgKiBubyBsb25nZXIgYmUgcG9zc2libGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCB0cnVlXG4gICAgICovXG4gICAgcmVhZG9ubHkgY3Jvc3NBY2NvdW50S2V5cz86IGJvb2xlYW47XG4gICAgLy8gQGRlcHJlY2F0ZWQodjIpOiBzd2l0Y2ggdG8gZGVmYXVsdCBmYWxzZVxuICAgIC8qKlxuICAgICAqIENESyBDTEkgdmVyc2lvbiB0byB1c2UgaW4gcGlwZWxpbmVcbiAgICAgKlxuICAgICAqIFNvbWUgQWN0aW9ucyBpbiB0aGUgcGlwZWxpbmUgd2lsbCBkb3dubG9hZCBhbmQgcnVuIGEgdmVyc2lvbiBvZiB0aGUgQ0RLXG4gICAgICogQ0xJLiBTcGVjaWZ5IHRoZSB2ZXJzaW9uIGhlcmUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIExhdGVzdCB2ZXJzaW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgY2RrQ2xpVmVyc2lvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgVlBDIHdoZXJlIHRvIGV4ZWN1dGUgdGhlIENka1BpcGVsaW5lIGFjdGlvbnMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIFZQQ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICAgIC8qKlxuICAgICAqIFdoaWNoIHN1Ym5ldHMgdG8gdXNlLlxuICAgICAqXG4gICAgICogT25seSB1c2VkIGlmICd2cGMnIGlzIHN1cHBsaWVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBbGwgcHJpdmF0ZSBzdWJuZXRzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN1Ym5ldFNlbGVjdGlvbj86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG59XG4vKipcbiAqIEEgUGlwZWxpbmUgdG8gZGVwbG95IENESyBhcHBzXG4gKlxuICogRGVmaW5lcyBhbiBBV1MgQ29kZVBpcGVsaW5lLWJhc2VkIFBpcGVsaW5lIHRvIGRlcGxveSBDREsgYXBwbGljYXRpb25zLlxuICpcbiAqIEF1dG9tYXRpY2FsbHkgbWFuYWdlcyB0aGUgZm9sbG93aW5nOlxuICpcbiAqIC0gU3RhY2sgZGVwZW5kZW5jeSBvcmRlci5cbiAqIC0gQXNzZXQgcHVibGlzaGluZy5cbiAqIC0gS2VlcGluZyB0aGUgcGlwZWxpbmUgdXAtdG8tZGF0ZSBhcyB0aGUgQ0RLIGFwcHMgY2hhbmdlLlxuICogLSBVc2luZyBzdGFjayBvdXRwdXRzIGxhdGVyIG9uIGluIHRoZSBwaXBlbGluZS5cbiAqL1xuZXhwb3J0IGNsYXNzIENka1BpcGVsaW5lIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IF9waXBlbGluZTogY29kZXBpcGVsaW5lLlBpcGVsaW5lO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgX2Fzc2V0czogQXNzZXRQdWJsaXNoaW5nO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgX3N0YWdlczogQ2RrU3RhZ2VbXSA9IFtdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgX291dHB1dEFydGlmYWN0czogUmVjb3JkPHN0cmluZywgY29kZXBpcGVsaW5lLkFydGlmYWN0PiA9IHt9O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgX2Nsb3VkQXNzZW1ibHlBcnRpZmFjdDogY29kZXBpcGVsaW5lLkFydGlmYWN0O1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZGtQaXBlbGluZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIGlmICghQXBwLmlzQXBwKHRoaXMubm9kZS5yb290KSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDZGtQaXBlbGluZSBtdXN0IGJlIGNyZWF0ZWQgdW5kZXIgYW4gQXBwJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5fY2xvdWRBc3NlbWJseUFydGlmYWN0ID0gcHJvcHMuY2xvdWRBc3NlbWJseUFydGlmYWN0O1xuICAgICAgICBjb25zdCBwaXBlbGluZVN0YWNrID0gU3RhY2sub2YodGhpcyk7XG4gICAgICAgIGlmIChwcm9wcy5jb2RlUGlwZWxpbmUpIHtcbiAgICAgICAgICAgIGlmIChwcm9wcy5waXBlbGluZU5hbWUpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBzZXQgXFwncGlwZWxpbmVOYW1lXFwnIGlmIGFuIGV4aXN0aW5nIENvZGVQaXBlbGluZSBpcyBnaXZlbiB1c2luZyBcXCdjb2RlUGlwZWxpbmVcXCcnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmIChwcm9wcy5jcm9zc0FjY291bnRLZXlzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBzZXQgXFwnY3Jvc3NBY2NvdW50S2V5c1xcJyBpZiBhbiBleGlzdGluZyBDb2RlUGlwZWxpbmUgaXMgZ2l2ZW4gdXNpbmcgXFwnY29kZVBpcGVsaW5lXFwnJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB0aGlzLl9waXBlbGluZSA9IHByb3BzLmNvZGVQaXBlbGluZTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMuX3BpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZSh0aGlzLCAnUGlwZWxpbmUnLCB7XG4gICAgICAgICAgICAgICAgcGlwZWxpbmVOYW1lOiBwcm9wcy5waXBlbGluZU5hbWUsXG4gICAgICAgICAgICAgICAgY3Jvc3NBY2NvdW50S2V5czogcHJvcHMuY3Jvc3NBY2NvdW50S2V5cyxcbiAgICAgICAgICAgICAgICByZXN0YXJ0RXhlY3V0aW9uT25VcGRhdGU6IHRydWUsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuc291cmNlQWN0aW9uICYmICFwcm9wcy5zeW50aEFjdGlvbikge1xuICAgICAgICAgICAgLy8gQmVjYXVzZSBvZiBvcmRlcmluZyBsaW1pdGF0aW9ucywgeW91IGNhbjogYnJpbmcgeW91ciBvd24gU291cmNlLCBicmluZyB5b3VyIG93blxuICAgICAgICAgICAgLy8gQm90aCwgb3IgYnJpbmcgeW91ciBvd24gTm90aGluZy4gWW91IGNhbm5vdCBicmluZyB5b3VyIG93biBCdWlsZCAod2hpY2ggYmVjYXVzZSBvZiB0aGVcbiAgICAgICAgICAgIC8vIGN1cnJlbnQgQ29kZVBpcGVsaW5lIEFQSSBtdXN0IGdvIEJFRk9SRSB3aGF0IHdlJ3JlIGFkZGluZykgYW5kIHRoZW4gaGF2aW5nIHVzIGFkZCBhXG4gICAgICAgICAgICAvLyBTb3VyY2UgYWZ0ZXIgaXQuIFRoYXQgZG9lc24ndCBtYWtlIGFueSBzZW5zZS5cbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignV2hlbiBwYXNzaW5nIGEgXFwnc291cmNlQWN0aW9uXFwnIHlvdSBtdXN0IGFsc28gcGFzcyBhIFxcJ3N5bnRoQWN0aW9uXFwnIChvciBhIFxcJ2NvZGVQaXBlbGluZVxcJyB0aGF0IGFscmVhZHkgaGFzIGJvdGgpJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKCFwcm9wcy5zb3VyY2VBY3Rpb24gJiYgKCFwcm9wcy5jb2RlUGlwZWxpbmUgfHwgcHJvcHMuY29kZVBpcGVsaW5lLnN0YWdlcy5sZW5ndGggPCAxKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBwYXNzIGEgXFwnc291cmNlQWN0aW9uXFwnIChvciBhIFxcJ2NvZGVQaXBlbGluZVxcJyB0aGF0IGFscmVhZHkgaGFzIGEgU291cmNlIHN0YWdlKScpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5zb3VyY2VBY3Rpb24pIHtcbiAgICAgICAgICAgIHRoaXMuX3BpcGVsaW5lLmFkZFN0YWdlKHtcbiAgICAgICAgICAgICAgICBzdGFnZU5hbWU6ICdTb3VyY2UnLFxuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFtwcm9wcy5zb3VyY2VBY3Rpb25dLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLnN5bnRoQWN0aW9uKSB7XG4gICAgICAgICAgICB0aGlzLl9waXBlbGluZS5hZGRTdGFnZSh7XG4gICAgICAgICAgICAgICAgc3RhZ2VOYW1lOiAnQnVpbGQnLFxuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFtwcm9wcy5zeW50aEFjdGlvbl0sXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLl9waXBlbGluZS5hZGRTdGFnZSh7XG4gICAgICAgICAgICBzdGFnZU5hbWU6ICdVcGRhdGVQaXBlbGluZScsXG4gICAgICAgICAgICBhY3Rpb25zOiBbbmV3IFVwZGF0ZVBpcGVsaW5lQWN0aW9uKHRoaXMsICdVcGRhdGVQaXBlbGluZScsIHtcbiAgICAgICAgICAgICAgICAgICAgY2xvdWRBc3NlbWJseUlucHV0OiB0aGlzLl9jbG91ZEFzc2VtYmx5QXJ0aWZhY3QsXG4gICAgICAgICAgICAgICAgICAgIHBpcGVsaW5lU3RhY2tOYW1lOiBwaXBlbGluZVN0YWNrLnN0YWNrTmFtZSxcbiAgICAgICAgICAgICAgICAgICAgY2RrQ2xpVmVyc2lvbjogcHJvcHMuY2RrQ2xpVmVyc2lvbixcbiAgICAgICAgICAgICAgICAgICAgcHJvamVjdE5hbWU6IG1heWJlU3VmZml4KHByb3BzLnBpcGVsaW5lTmFtZSwgJy1zZWxmdXBkYXRlJyksXG4gICAgICAgICAgICAgICAgfSldLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5fYXNzZXRzID0gbmV3IEFzc2V0UHVibGlzaGluZyh0aGlzLCAnQXNzZXRzJywge1xuICAgICAgICAgICAgY2xvdWRBc3NlbWJseUlucHV0OiB0aGlzLl9jbG91ZEFzc2VtYmx5QXJ0aWZhY3QsXG4gICAgICAgICAgICBjZGtDbGlWZXJzaW9uOiBwcm9wcy5jZGtDbGlWZXJzaW9uLFxuICAgICAgICAgICAgcGlwZWxpbmU6IHRoaXMuX3BpcGVsaW5lLFxuICAgICAgICAgICAgcHJvamVjdE5hbWU6IG1heWJlU3VmZml4KHByb3BzLnBpcGVsaW5lTmFtZSwgJy1wdWJsaXNoJyksXG4gICAgICAgICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgICAgICAgIHN1Ym5ldFNlbGVjdGlvbjogcHJvcHMuc3VibmV0U2VsZWN0aW9uLFxuICAgICAgICB9KTtcbiAgICAgICAgQXNwZWN0cy5vZih0aGlzKS5hZGQoeyB2aXNpdDogKCkgPT4gdGhpcy5fYXNzZXRzLnJlbW92ZUFzc2V0c1N0YWdlSWZFbXB0eSgpIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgdW5kZXJseWluZyBDb2RlUGlwZWxpbmUgb2JqZWN0XG4gICAgICpcbiAgICAgKiBZb3UgY2FuIHVzZSB0aGlzIHRvIGFkZCBtb3JlIFN0YWdlcyB0byB0aGUgcGlwZWxpbmUsIG9yIEFjdGlvbnNcbiAgICAgKiB0byBTdGFnZXMuXG4gICAgICovXG4gICAgcHVibGljIGdldCBjb2RlUGlwZWxpbmUoKTogY29kZXBpcGVsaW5lLlBpcGVsaW5lIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuX3BpcGVsaW5lO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBY2Nlc3Mgb25lIG9mIHRoZSBwaXBlbGluZSdzIHN0YWdlcyBieSBzdGFnZSBuYW1lXG4gICAgICpcbiAgICAgKiBZb3UgY2FuIHVzZSB0aGlzIHRvIGFkZCBtb3JlIEFjdGlvbnMgdG8gYSBzdGFnZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhZ2Uoc3RhZ2VOYW1lOiBzdHJpbmcpOiBjb2RlcGlwZWxpbmUuSVN0YWdlIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuX3BpcGVsaW5lLnN0YWdlKHN0YWdlTmFtZSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBwaXBlbGluZSBzdGFnZSB0aGF0IHdpbGwgZGVwbG95IHRoZSBnaXZlbiBhcHBsaWNhdGlvbiBzdGFnZVxuICAgICAqXG4gICAgICogVGhlIGFwcGxpY2F0aW9uIGNvbnN0cnVjdCBzaG91bGQgc3ViY2xhc3MgYFN0YWdlYCBhbmQgY2FuIGNvbnRhaW4gYW55XG4gICAgICogbnVtYmVyIG9mIGBTdGFja3NgIGluc2lkZSBpdCB0aGF0IG1heSBoYXZlIGRlcGVuZGVuY3kgcmVsYXRpb25zaGlwc1xuICAgICAqIG9uIG9uZSBhbm90aGVyLlxuICAgICAqXG4gICAgICogQWxsIHN0YWNrcyBpbiB0aGUgYXBwbGljYXRpb24gd2lsbCBiZSBkZXBsb3llZCBpbiB0aGUgYXBwcm9wcmlhdGUgb3JkZXIsXG4gICAgICogYW5kIGFsbCBhc3NldHMgZm91bmQgaW4gdGhlIGFwcGxpY2F0aW9uIHdpbGwgYmUgYWRkZWQgdG8gdGhlIGFzc2V0XG4gICAgICogcHVibGlzaGluZyBzdGFnZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQXBwbGljYXRpb25TdGFnZShhcHBTdGFnZTogU3RhZ2UsIG9wdGlvbnM6IEFkZFN0YWdlT3B0aW9ucyA9IHt9KTogQ2RrU3RhZ2Uge1xuICAgICAgICBjb25zdCBzdGFnZSA9IHRoaXMuYWRkU3RhZ2UoYXBwU3RhZ2Uuc3RhZ2VOYW1lKTtcbiAgICAgICAgc3RhZ2UuYWRkQXBwbGljYXRpb24oYXBwU3RhZ2UsIG9wdGlvbnMpO1xuICAgICAgICByZXR1cm4gc3RhZ2U7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIG5ldywgZW1wdHkgc3RhZ2UgdG8gdGhlIHBpcGVsaW5lXG4gICAgICpcbiAgICAgKiBQcmVmZXIgdG8gdXNlIGBhZGRBcHBsaWNhdGlvblN0YWdlYCBpZiB5b3UgYXJlIGludGVuZGVkIHRvIGRlcGxveSBhIENES1xuICAgICAqIGFwcGxpY2F0aW9uLCBidXQgeW91IGNhbiB1c2UgdGhpcyBtZXRob2QgaWYgeW91IHdhbnQgdG8gYWRkIG90aGVyIGtpbmRzIG9mXG4gICAgICogQWN0aW9ucyB0byBhIHBpcGVsaW5lLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRTdGFnZShzdGFnZU5hbWU6IHN0cmluZykge1xuICAgICAgICBjb25zdCBwaXBlbGluZVN0YWdlID0gdGhpcy5fcGlwZWxpbmUuYWRkU3RhZ2Uoe1xuICAgICAgICAgICAgc3RhZ2VOYW1lLFxuICAgICAgICB9KTtcbiAgICAgICAgY29uc3Qgc3RhZ2UgPSBuZXcgQ2RrU3RhZ2UodGhpcywgc3RhZ2VOYW1lLCB7XG4gICAgICAgICAgICBjbG91ZEFzc2VtYmx5QXJ0aWZhY3Q6IHRoaXMuX2Nsb3VkQXNzZW1ibHlBcnRpZmFjdCxcbiAgICAgICAgICAgIHBpcGVsaW5lU3RhZ2UsXG4gICAgICAgICAgICBzdGFnZU5hbWUsXG4gICAgICAgICAgICBob3N0OiB7XG4gICAgICAgICAgICAgICAgcHVibGlzaEFzc2V0OiB0aGlzLl9hc3NldHMuYWRkUHVibGlzaEFzc2V0QWN0aW9uLmJpbmQodGhpcy5fYXNzZXRzKSxcbiAgICAgICAgICAgICAgICBzdGFja091dHB1dEFydGlmYWN0OiAoYXJ0aWZhY3RJZCkgPT4gdGhpcy5fb3V0cHV0QXJ0aWZhY3RzW2FydGlmYWN0SWRdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuX3N0YWdlcy5wdXNoKHN0YWdlKTtcbiAgICAgICAgcmV0dXJuIHN0YWdlO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBHZXQgdGhlIFN0YWNrT3V0cHV0IG9iamVjdCB0aGF0IGhvbGRzIHRoaXMgQ2ZuT3V0cHV0J3MgdmFsdWUgaW4gdGhpcyBwaXBlbGluZVxuICAgICAqXG4gICAgICogYFN0YWNrT3V0cHV0YCBjYW4gYmUgdXNlZCBpbiB2YWxpZGF0aW9uIGFjdGlvbnMgbGF0ZXIgaW4gdGhlIHBpcGVsaW5lLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGFja091dHB1dChjZm5PdXRwdXQ6IENmbk91dHB1dCk6IFN0YWNrT3V0cHV0IHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihjZm5PdXRwdXQpO1xuICAgICAgICBpZiAoIXRoaXMuX291dHB1dEFydGlmYWN0c1tzdGFjay5hcnRpZmFjdElkXSkge1xuICAgICAgICAgICAgLy8gV2Ugc2hvdWxkIGhhdmUgc3RvcmVkIHRoZSBBcnRpZmFjdFBhdGggaW4gdGhlIG1hcCwgYnV0IGl0cyBBcnRpZmFjdFxuICAgICAgICAgICAgLy8gcHJvcGVydHkgaXNuJ3QgcHVibGljbHkgcmVhZGFibGUuLi5cbiAgICAgICAgICAgIHRoaXMuX291dHB1dEFydGlmYWN0c1tzdGFjay5hcnRpZmFjdElkXSA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoYEFydGlmYWN0XyR7c3RhY2suYXJ0aWZhY3RJZH1fT3V0cHV0c2ApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgU3RhY2tPdXRwdXQodGhpcy5fb3V0cHV0QXJ0aWZhY3RzW3N0YWNrLmFydGlmYWN0SWRdLmF0UGF0aCgnb3V0cHV0cy5qc29uJyksIGNmbk91dHB1dC5sb2dpY2FsSWQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBWYWxpZGF0ZSB0aGF0IHdlIGRvbid0IGhhdmUgYW55IHN0YWNrcyB2aW9sYXRpbmcgZGVwZW5kZW5jeSBvcmRlciBpbiB0aGUgcGlwZWxpbmVcbiAgICAgKlxuICAgICAqIE91ciBvd24gY29udmVuaWVuY2UgbWV0aG9kcyB3aWxsIG5ldmVyIGdlbmVyYXRlIGEgcGlwZWxpbmUgdGhhdCBkb2VzIHRoYXQgKGFsdGhvdWdoXG4gICAgICogdGhpcyBpcyBhIG5pY2UgdmVyaWZpY2F0aW9uKSwgYnV0IGEgdXNlciBjYW4gYWxzbyBhZGQgdGhlIHN0YWNrcyBieSBoYW5kLlxuICAgICAqL1xuICAgIHByb3RlY3RlZCB2YWxpZGF0ZSgpOiBzdHJpbmdbXSB7XG4gICAgICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG4gICAgICAgIHJldC5wdXNoKC4uLnRoaXMudmFsaWRhdGVEZXBsb3lPcmRlcigpKTtcbiAgICAgICAgcmV0LnB1c2goLi4udGhpcy52YWxpZGF0ZVJlcXVlc3RlZE91dHB1dHMoKSk7XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybiBhbGwgU3RhY2tEZXBsb3lBY3Rpb25zIGluIGFuIG9yZGVyZWQgbGlzdFxuICAgICAqL1xuICAgIHByaXZhdGUgZ2V0IHN0YWNrQWN0aW9ucygpOiBEZXBsb3lDZGtTdGFja0FjdGlvbltdIHtcbiAgICAgICAgcmV0dXJuIGZsYXRNYXAodGhpcy5fcGlwZWxpbmUuc3RhZ2VzLCBzID0+IHMuYWN0aW9ucy5maWx0ZXIoaXNEZXBsb3lBY3Rpb24pKTtcbiAgICB9XG4gICAgcHJpdmF0ZSAqdmFsaWRhdGVEZXBsb3lPcmRlcigpOiBJdGVyYWJsZUl0ZXJhdG9yPHN0cmluZz4ge1xuICAgICAgICBjb25zdCBzdGFja0FjdGlvbnMgPSB0aGlzLnN0YWNrQWN0aW9ucztcbiAgICAgICAgZm9yIChjb25zdCBzdGFja0FjdGlvbiBvZiBzdGFja0FjdGlvbnMpIHtcbiAgICAgICAgICAgIC8vIEZvciBldmVyeSBkZXBlbmRlbmN5LCBpdCBtdXN0IGJlIGV4ZWN1dGVkIGluIGFuIGFjdGlvbiBiZWZvcmUgdGhpcyBvbmUgaXMgcHJlcGFyZWQuXG4gICAgICAgICAgICBmb3IgKGNvbnN0IGRlcElkIG9mIHN0YWNrQWN0aW9uLmRlcGVuZGVuY3lTdGFja0FydGlmYWN0SWRzKSB7XG4gICAgICAgICAgICAgICAgY29uc3QgZGVwQWN0aW9uID0gc3RhY2tBY3Rpb25zLmZpbmQocyA9PiBzLnN0YWNrQXJ0aWZhY3RJZCA9PT0gZGVwSWQpO1xuICAgICAgICAgICAgICAgIGlmIChkZXBBY3Rpb24gPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgICAgICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKGBTdGFjayAnJHtzdGFja0FjdGlvbi5zdGFja05hbWV9JyBkZXBlbmRzIG9uIHN0YWNrIGAgK1xuICAgICAgICAgICAgICAgICAgICAgICAgYCcke2RlcElkfScsIGJ1dCB0aGF0IGRlcGVuZGVuY3kgaXMgbm90IGRlcGxveWVkIHRocm91Z2ggdGhlIHBpcGVsaW5lIWApO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBlbHNlIGlmICghKGRlcEFjdGlvbi5leGVjdXRlUnVuT3JkZXIgPCBzdGFja0FjdGlvbi5wcmVwYXJlUnVuT3JkZXIpKSB7XG4gICAgICAgICAgICAgICAgICAgIHlpZWxkIGBTdGFjayAnJHtzdGFja0FjdGlvbi5zdGFja05hbWV9JyBkZXBlbmRzIG9uIHN0YWNrIGAgK1xuICAgICAgICAgICAgICAgICAgICAgICAgYCcke2RlcEFjdGlvbi5zdGFja05hbWV9JywgYnV0IGlzIGRlcGxveWVkIGJlZm9yZSBpdCBpbiB0aGUgcGlwZWxpbmUhYDtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSAqdmFsaWRhdGVSZXF1ZXN0ZWRPdXRwdXRzKCk6IEl0ZXJhYmxlSXRlcmF0b3I8c3RyaW5nPiB7XG4gICAgICAgIGNvbnN0IGFydGlmYWN0SWRzID0gdGhpcy5zdGFja0FjdGlvbnMubWFwKHMgPT4gcy5zdGFja0FydGlmYWN0SWQpO1xuICAgICAgICBmb3IgKGNvbnN0IGFydGlmYWN0SWQgb2YgT2JqZWN0LmtleXModGhpcy5fb3V0cHV0QXJ0aWZhY3RzKSkge1xuICAgICAgICAgICAgaWYgKCFhcnRpZmFjdElkcy5pbmNsdWRlcyhhcnRpZmFjdElkKSkge1xuICAgICAgICAgICAgICAgIHlpZWxkIGBUcnlpbmcgdG8gdXNlIG91dHB1dHMgZm9yIFN0YWNrICcke2FydGlmYWN0SWR9JywgYnV0IFN0YWNrIGlzIG5vdCBkZXBsb3llZCBpbiB0aGlzIHBpcGVsaW5lLiBBZGQgaXQgdG8gdGhlIHBpcGVsaW5lLmA7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICB9XG59XG5mdW5jdGlvbiBpc0RlcGxveUFjdGlvbihhOiBjb2RlcGlwZWxpbmUuSUFjdGlvbik6IGEgaXMgRGVwbG95Q2RrU3RhY2tBY3Rpb24ge1xuICAgIHJldHVybiBhIGluc3RhbmNlb2YgRGVwbG95Q2RrU3RhY2tBY3Rpb247XG59XG5mdW5jdGlvbiBmbGF0TWFwPEEsIEI+KHhzOiBBW10sIGY6ICh4OiBBKSA9PiBCW10pOiBCW10ge1xuICAgIHJldHVybiBBcnJheS5wcm90b3R5cGUuY29uY2F0KFtdLCAuLi54cy5tYXAoZikpO1xufVxuaW50ZXJmYWNlIEFzc2V0UHVibGlzaGluZ1Byb3BzIHtcbiAgICByZWFkb25seSBjbG91ZEFzc2VtYmx5SW5wdXQ6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICByZWFkb25seSBwaXBlbGluZTogY29kZXBpcGVsaW5lLlBpcGVsaW5lO1xuICAgIHJlYWRvbmx5IGNka0NsaVZlcnNpb24/OiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgcHJvamVjdE5hbWU/OiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG4gICAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbn1cbi8qKlxuICogQWRkIGFwcHJvcHJpYXRlIHB1Ymxpc2hpbmcgYWN0aW9ucyB0byB0aGUgYXNzZXQgcHVibGlzaGluZyBzdGFnZVxuICovXG5jbGFzcyBBc3NldFB1Ymxpc2hpbmcgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcHVibGlzaGVyczogUmVjb3JkPHN0cmluZywgUHVibGlzaEFzc2V0c0FjdGlvbj4gPSB7fTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGFzc2V0Um9sZXM6IFJlY29yZDxzdHJpbmcsIGlhbS5JUm9sZT4gPSB7fTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IG15Q3hBc21Sb290OiBzdHJpbmc7XG4gICAgcHJpdmF0ZSByZWFkb25seSBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHBpcGVsaW5lOiBjb2RlcGlwZWxpbmUuUGlwZWxpbmU7XG4gICAgcHJpdmF0ZSBfZmlsZUFzc2V0Q3RyID0gMTtcbiAgICBwcml2YXRlIF9kb2NrZXJBc3NldEN0ciA9IDE7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBwcm9wczogQXNzZXRQdWJsaXNoaW5nUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy5teUN4QXNtUm9vdCA9IHBhdGgucmVzb2x2ZShhc3NlbWJseUJ1aWxkZXJPZihhcHBPZih0aGlzKSkub3V0ZGlyKTtcbiAgICAgICAgLy8gV2UgTVVTVCBhZGQgdGhlIFN0YWdlIGltbWVkaWF0ZWx5IGhlcmUsIG90aGVyd2lzZSBpdCB3aWxsIGJlIGluIHRoZSB3cm9uZyBwbGFjZVxuICAgICAgICAvLyBpbiB0aGUgcGlwZWxpbmUhXG4gICAgICAgIHRoaXMuc3RhZ2UgPSB0aGlzLnByb3BzLnBpcGVsaW5lLmFkZFN0YWdlKHsgc3RhZ2VOYW1lOiAnQXNzZXRzJyB9KTtcbiAgICAgICAgdGhpcy5waXBlbGluZSA9IHRoaXMucHJvcHMucGlwZWxpbmU7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIE1ha2Ugc3VyZSB0aGVyZSBpcyBhbiBhY3Rpb24gaW4gdGhlIHN0YWdlIHRvIHB1Ymxpc2ggdGhlIGdpdmVuIGFzc2V0XG4gICAgICpcbiAgICAgKiBBc3NldHMgYXJlIGdyb3VwZWQgYnkgYXNzZXQgSUQgKHdoaWNoIHJlcHJlc2VudCBpbmRpdmlkdWFsIGFzc2V0cykgc28gYWxsIGFzc2V0c1xuICAgICAqIGFyZSBwdWJsaXNoZWQgaW4gcGFyYWxsZWwuIEZvciBlYWNoIGFzc2V0cywgYWxsIGRlc3RpbmF0aW9ucyBhcmUgcHVibGlzaGVkIHNlcXVlbnRpYWxseVxuICAgICAqIHNvIHRoYXQgd2UgY2FuIHJldXNlIGV4cGVuc2l2ZSBvcGVyYXRpb25zIGJldHdlZW4gdGhlbSAobW9zdGx5OiBidWlsZGluZyBhIERvY2tlciBpbWFnZSkuXG4gICAgICovXG4gICAgcHVibGljIGFkZFB1Ymxpc2hBc3NldEFjdGlvbihjb21tYW5kOiBBc3NldFB1Ymxpc2hpbmdDb21tYW5kKSB7XG4gICAgICAgIC8vIEZJWE1FOiB0aGlzIGlzIHNpbGx5LCB3ZSBuZWVkIHRoZSByZWxhdGl2ZSBwYXRoIGhlcmUgYnV0IG5vIGVhc3kgd2F5IHRvIGdldCBpdFxuICAgICAgICBjb25zdCByZWxhdGl2ZVBhdGggPSBwYXRoLnJlbGF0aXZlKHRoaXMubXlDeEFzbVJvb3QsIGNvbW1hbmQuYXNzZXRNYW5pZmVzdFBhdGgpO1xuICAgICAgICAvLyBUaGUgcGF0aCBjYW5ub3QgYmUgb3V0c2lkZSB0aGUgYXNtIHJvb3QuIEkgZG9uJ3QgcmVhbGx5IHVuZGVyc3RhbmQgaG93IHRoaXMgY291bGQgZXZlclxuICAgICAgICAvLyBjb21lIHRvIHBhc3MsIGJ1dCBhcHBhcmVudGx5IGl0IGhhcyAoc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvOTc2NikuXG4gICAgICAgIC8vIEFkZCBhIHNhbml0eSBjaGVjayBoZXJlIHNvIHdlIGNhbiBjYXRjaCBpdCBtb3JlIHF1aWNrbHkgbmV4dCB0aW1lLlxuICAgICAgICBpZiAocmVsYXRpdmVQYXRoLnN0YXJ0c1dpdGgoYC4uJHtwYXRoLnNlcH1gKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgYXNzZXQgbWFuaWZlc3QgKCR7Y29tbWFuZC5hc3NldE1hbmlmZXN0UGF0aH0pIGNhbm5vdCBiZSBvdXRzaWRlIHRoZSBDbG91ZCBBc3NlbWJseSBkaXJlY3RvcnkgKCR7dGhpcy5teUN4QXNtUm9vdH0pLiBQbGVhc2UgcmVwb3J0IHRoaXMgZXJyb3IgYXQgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3VlcyB0byBoZWxwIHVzIGRlYnVnIHdoeSB0aGlzIGlzIGhhcHBlbmluZy5gKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBMYXRlLWJpbmRpbmcgaGVyZSAocmF0aGVyIHRoYW4gaW4gdGhlIGNvbnN0cnVjdG9yKSB0byBwcmV2ZW50IGNyZWF0aW5nIHRoZSByb2xlIGluIGNhc2VzIHdoZXJlIG5vIGFzc2V0IGFjdGlvbnMgYXJlIGNyZWF0ZWQuXG4gICAgICAgIGlmICghdGhpcy5hc3NldFJvbGVzW2NvbW1hbmQuYXNzZXRUeXBlXSkge1xuICAgICAgICAgICAgdGhpcy5nZW5lcmF0ZUFzc2V0Um9sZShjb21tYW5kLmFzc2V0VHlwZSk7XG4gICAgICAgIH1cbiAgICAgICAgbGV0IGFjdGlvbiA9IHRoaXMucHVibGlzaGVyc1tjb21tYW5kLmFzc2V0SWRdO1xuICAgICAgICBpZiAoIWFjdGlvbikge1xuICAgICAgICAgICAgLy8gVGhlIGFzc2V0IElEIHdvdWxkIGJlIGEgbG9naWNhbCBjYW5kaWRhdGUgZm9yIHRoZSBjb25zdHJ1Y3QgcGF0aCBhbmQgcHJvamVjdCBuYW1lcywgYnV0IGlmIHRoZSBhc3NldFxuICAgICAgICAgICAgLy8gY2hhbmdlcyBpdCBsZWFkcyB0byByZWNyZWF0aW9uIG9mIGEgbnVtYmVyIG9mIFJvbGUvUG9saWN5L1Byb2plY3QgcmVzb3VyY2VzIHdoaWNoIGlzIHNsb3dlciB0aGFuXG4gICAgICAgICAgICAvLyBuZWNlc3NhcnkuIE51bWJlciBzZXF1ZW50aWFsbHkgaW5zdGVhZC5cbiAgICAgICAgICAgIC8vXG4gICAgICAgICAgICAvLyBGSVhNRTogVGhlIHVsdGltYXRlIGJlc3Qgc29sdXRpb24gaXMgcHJvYmFibHkgdG8gZ2VuZXJhdGUgYSBzaW5nbGUgUHJvamVjdCBwZXIgYXNzZXQgdHlwZVxuICAgICAgICAgICAgLy8gYW5kIHJldXNlIHRoYXQgZm9yIGFsbCBhc3NldHMuXG4gICAgICAgICAgICBjb25zdCBpZCA9IGNvbW1hbmQuYXNzZXRUeXBlID09PSBBc3NldFR5cGUuRklMRSA/IGBGaWxlQXNzZXQke3RoaXMuX2ZpbGVBc3NldEN0cisrfWAgOiBgRG9ja2VyQXNzZXQke3RoaXMuX2RvY2tlckFzc2V0Q3RyKyt9YDtcbiAgICAgICAgICAgIC8vIE5PVEU6IEl0J3MgaW1wb3J0YW50IHRoYXQgYXNzZXQgY2hhbmdlcyBkb24ndCBmb3JjZSBhIHBpcGVsaW5lIHNlbGYtbXV0YXRpb24uXG4gICAgICAgICAgICAvLyBUaGlzIGNhbiBjYXVzZSBhbiBpbmZpbml0ZSBsb29wIG9mIHVwZGF0ZXMgKHNlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvaXNzdWVzLzkwODApLlxuICAgICAgICAgICAgLy8gRm9yIHRoYXQgcmVhc29uLCB3ZSB1c2UgdGhlIGlkIGFzIHRoZSBhY3Rpb25OYW1lIGJlbG93LCByYXRoZXIgdGhhbiB0aGUgYXNzZXQgaGFzaC5cbiAgICAgICAgICAgIGFjdGlvbiA9IHRoaXMucHVibGlzaGVyc1tjb21tYW5kLmFzc2V0SWRdID0gbmV3IFB1Ymxpc2hBc3NldHNBY3Rpb24odGhpcywgaWQsIHtcbiAgICAgICAgICAgICAgICBhY3Rpb25OYW1lOiBpZCxcbiAgICAgICAgICAgICAgICBjbG91ZEFzc2VtYmx5SW5wdXQ6IHRoaXMucHJvcHMuY2xvdWRBc3NlbWJseUlucHV0LFxuICAgICAgICAgICAgICAgIGNka0NsaVZlcnNpb246IHRoaXMucHJvcHMuY2RrQ2xpVmVyc2lvbixcbiAgICAgICAgICAgICAgICBhc3NldFR5cGU6IGNvbW1hbmQuYXNzZXRUeXBlLFxuICAgICAgICAgICAgICAgIHJvbGU6IHRoaXMuYXNzZXRSb2xlc1tjb21tYW5kLmFzc2V0VHlwZV0sXG4gICAgICAgICAgICAgICAgdnBjOiB0aGlzLnByb3BzLnZwYyxcbiAgICAgICAgICAgICAgICBzdWJuZXRTZWxlY3Rpb246IHRoaXMucHJvcHMuc3VibmV0U2VsZWN0aW9uLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB0aGlzLnN0YWdlLmFkZEFjdGlvbihhY3Rpb24pO1xuICAgICAgICB9XG4gICAgICAgIGFjdGlvbi5hZGRQdWJsaXNoQ29tbWFuZChyZWxhdGl2ZVBhdGgsIGNvbW1hbmQuYXNzZXRTZWxlY3Rvcik7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlbW92ZSB0aGUgQXNzZXRzIHN0YWdlIGlmIGl0IHR1cm5zIG91dCB3ZSBkaWRuJ3QgYWRkIGFueSBBc3NldHMgdG8gcHVibGlzaFxuICAgICAqL1xuICAgIHB1YmxpYyByZW1vdmVBc3NldHNTdGFnZUlmRW1wdHkoKSB7XG4gICAgICAgIGlmIChPYmplY3Qua2V5cyh0aGlzLnB1Ymxpc2hlcnMpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgLy8gSGFja3MgdG8gZ2V0IGFjY2VzcyB0byBpbm5hcmRzIG9mIFBpcGVsaW5lXG4gICAgICAgICAgICAvLyBNb2RpZnkgJ3N0YWdlcycgYXJyYXkgaW4tcGxhY2UgdG8gcmVtb3ZlIEFzc2V0cyBzdGFnZSBpZiBlbXB0eVxuICAgICAgICAgICAgY29uc3Qgc3RhZ2VzOiBjb2RlcGlwZWxpbmUuSVN0YWdlW10gPSAodGhpcy5wcm9wcy5waXBlbGluZSBhcyBhbnkpLl9zdGFnZXM7XG4gICAgICAgICAgICBjb25zdCBpeCA9IHN0YWdlcy5pbmRleE9mKHRoaXMuc3RhZ2UpO1xuICAgICAgICAgICAgaWYgKGl4ID4gLTEpIHtcbiAgICAgICAgICAgICAgICBzdGFnZXMuc3BsaWNlKGl4LCAxKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGlzIHJvbGUgaXMgdXNlZCBieSBib3RoIHRoZSBDb2RlUGlwZWxpbmUgYnVpbGQgYWN0aW9uIGFuZCByZWxhdGVkIENvZGVCdWlsZCBwcm9qZWN0LiBDb25zb2xpZGF0aW5nIHRoZXNlIHR3b1xuICAgICAqIHJvbGVzIGludG8gb25lLCBhbmQgcmUtdXNpbmcgYWNyb3NzIGFsbCBhc3NldHMsIHNhdmVzIHNpZ25pZmljYW50IHNpemUgb2YgdGhlIGZpbmFsIHN5bnRoZXNpemVkIG91dHB1dC5cbiAgICAgKiBNb2RlbGVkIGFmdGVyIHRoZSBDb2RlUGlwZWxpbmUgcm9sZSBhbmQgJ0NvZGVQaXBlbGluZUFjdGlvblJvbGUnIHJvbGVzLlxuICAgICAqIEdlbmVyYXRlcyBvbmUgcm9sZSBwZXIgYXNzZXQgdHlwZSB0byBzZXBhcmF0ZSBmaWxlIGFuZCBEb2NrZXIvaW1hZ2UtYmFzZWQgcGVybWlzc2lvbnMuXG4gICAgICovXG4gICAgcHJpdmF0ZSBnZW5lcmF0ZUFzc2V0Um9sZShhc3NldFR5cGU6IEFzc2V0VHlwZSkge1xuICAgICAgICBpZiAodGhpcy5hc3NldFJvbGVzW2Fzc2V0VHlwZV0pIHtcbiAgICAgICAgICAgIHJldHVybiB0aGlzLmFzc2V0Um9sZXNbYXNzZXRUeXBlXTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCByb2xlUHJlZml4ID0gYXNzZXRUeXBlID09PSBBc3NldFR5cGUuRE9DS0VSX0lNQUdFID8gJ0RvY2tlcicgOiAnRmlsZSc7XG4gICAgICAgIGNvbnN0IGFzc2V0Um9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCBgJHtyb2xlUHJlZml4fVJvbGVgLCB7XG4gICAgICAgICAgICByb2xlTmFtZTogUGh5c2ljYWxOYW1lLkdFTkVSQVRFX0lGX05FRURFRCxcbiAgICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5Db21wb3NpdGVQcmluY2lwYWwobmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdjb2RlYnVpbGQuYW1hem9uYXdzLmNvbScpLCBuZXcgaWFtLkFjY291bnRQcmluY2lwYWwoU3RhY2sub2YodGhpcykuYWNjb3VudCkpLFxuICAgICAgICB9KTtcbiAgICAgICAgLy8gTG9nZ2luZyBwZXJtaXNzaW9uc1xuICAgICAgICBjb25zdCBsb2dHcm91cEFybiA9IFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgICBzZXJ2aWNlOiAnbG9ncycsXG4gICAgICAgICAgICByZXNvdXJjZTogJ2xvZy1ncm91cCcsXG4gICAgICAgICAgICBzZXA6ICc6JyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogJy9hd3MvY29kZWJ1aWxkLyonLFxuICAgICAgICB9KTtcbiAgICAgICAgYXNzZXRSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIHJlc291cmNlczogW2xvZ0dyb3VwQXJuXSxcbiAgICAgICAgICAgIGFjdGlvbnM6IFsnbG9nczpDcmVhdGVMb2dHcm91cCcsICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsICdsb2dzOlB1dExvZ0V2ZW50cyddLFxuICAgICAgICB9KSk7XG4gICAgICAgIC8vIENvZGVCdWlsZCByZXBvcnQgZ3JvdXBzXG4gICAgICAgIGNvbnN0IGNvZGVCdWlsZEFybiA9IFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgICBzZXJ2aWNlOiAnY29kZWJ1aWxkJyxcbiAgICAgICAgICAgIHJlc291cmNlOiAncmVwb3J0LWdyb3VwJyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogJyonLFxuICAgICAgICB9KTtcbiAgICAgICAgYXNzZXRSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAnY29kZWJ1aWxkOkNyZWF0ZVJlcG9ydEdyb3VwJyxcbiAgICAgICAgICAgICAgICAnY29kZWJ1aWxkOkNyZWF0ZVJlcG9ydCcsXG4gICAgICAgICAgICAgICAgJ2NvZGVidWlsZDpVcGRhdGVSZXBvcnQnLFxuICAgICAgICAgICAgICAgICdjb2RlYnVpbGQ6QmF0Y2hQdXRUZXN0Q2FzZXMnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHJlc291cmNlczogW2NvZGVCdWlsZEFybl0sXG4gICAgICAgIH0pKTtcbiAgICAgICAgLy8gQ29kZUJ1aWxkIHN0YXJ0L3N0b3BcbiAgICAgICAgYXNzZXRSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgJ2NvZGVidWlsZDpCYXRjaEdldEJ1aWxkcycsXG4gICAgICAgICAgICAgICAgJ2NvZGVidWlsZDpTdGFydEJ1aWxkJyxcbiAgICAgICAgICAgICAgICAnY29kZWJ1aWxkOlN0b3BCdWlsZCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICB9KSk7XG4gICAgICAgIC8vIFB1Ymxpc2hpbmcgcm9sZSBhY2Nlc3NcbiAgICAgICAgY29uc3Qgcm9sZVBhdHRlcm4gPSBhc3NldFR5cGUgPT09IEFzc2V0VHlwZS5ET0NLRVJfSU1BR0VcbiAgICAgICAgICAgID8gJ2FybjoqOmlhbTo6Kjpyb2xlLyotaW1hZ2UtcHVibGlzaGluZy1yb2xlLSonXG4gICAgICAgICAgICA6ICdhcm46KjppYW06Oio6cm9sZS8qLWZpbGUtcHVibGlzaGluZy1yb2xlLSonO1xuICAgICAgICBhc3NldFJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydzdHM6QXNzdW1lUm9sZSddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbcm9sZVBhdHRlcm5dLFxuICAgICAgICB9KSk7XG4gICAgICAgIC8vIEFydGlmYWN0IGFjY2Vzc1xuICAgICAgICB0aGlzLnBpcGVsaW5lLmFydGlmYWN0QnVja2V0LmdyYW50UmVhZChhc3NldFJvbGUpO1xuICAgICAgICB0aGlzLmFzc2V0Um9sZXNbYXNzZXRUeXBlXSA9IGFzc2V0Um9sZS53aXRob3V0UG9saWN5VXBkYXRlcygpO1xuICAgICAgICByZXR1cm4gdGhpcy5hc3NldFJvbGVzW2Fzc2V0VHlwZV07XG4gICAgfVxufVxuZnVuY3Rpb24gbWF5YmVTdWZmaXgoeDogc3RyaW5nIHwgdW5kZWZpbmVkLCBzdWZmaXg6IHN0cmluZyk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKHggPT09IHVuZGVmaW5lZCkge1xuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICByZXR1cm4gYCR7eH0ke3N1ZmZpeH1gO1xufVxuIl19