import * as codepipeline from "../../../aws-codepipeline";
import * as events from "../../../aws-events";
import * as iam from "../../../aws-iam";
import { Construct } from "../../../core";
import * as cxapi from "../../../cx-api";
/**
 * Customization options for a DeployCdkStackAction
 */
export interface DeployCdkStackActionOptions {
    /**
     * Base name of the action
     *
     * @default stackName
     */
    readonly baseActionName?: string;
    /**
     * The CodePipeline artifact that holds the Cloud Assembly.
     */
    readonly cloudAssemblyInput: codepipeline.Artifact;
    /**
     * Run order for the Prepare action
     *
     * @default 1
     */
    readonly prepareRunOrder?: number;
    /**
     * Run order for the Execute action
     *
     * @default - prepareRunOrder + 1
     */
    readonly executeRunOrder?: number;
    /**
     * Artifact to write Stack Outputs to
     *
     * @default - No outputs
     */
    readonly output?: codepipeline.Artifact;
    /**
     * Filename in output to write Stack outputs to
     *
     * @default - Required when 'output' is set
     */
    readonly outputFileName?: string;
    /**
     * Name of the change set to create and deploy
     *
     * @default 'PipelineChange'
     */
    readonly changeSetName?: string;
}
/**
 * Properties for a DeployCdkStackAction
 */
export interface DeployCdkStackActionProps extends DeployCdkStackActionOptions {
    /**
     * Relative path of template in the input artifact
     */
    readonly templatePath: string;
    /**
     * Role for the action to assume
     *
     * This controls the account to deploy into
     */
    readonly actionRole: iam.IRole;
    /**
     * The name of the stack that should be created/updated
     */
    readonly stackName: string;
    /**
     * Role to execute CloudFormation under
     *
     * @default - Execute CloudFormation using the action role
     */
    readonly cloudFormationExecutionRole?: iam.IRole;
    /**
     * Region to deploy into
     *
     * @default - Same region as pipeline
     */
    readonly region?: string;
    /**
     * Artifact ID for the stack deployed here
     *
     * Used for pipeline order checking.
     *
     * @default - Order will not be checked
     */
    readonly stackArtifactId?: string;
    /**
     * Artifact ID for the stacks this stack depends on
     *
     * Used for pipeline order checking.
     *
     * @default - No dependencies
     */
    readonly dependencyStackArtifactIds?: string[];
    /**
     * Template configuration path relative to the input artifact
     *
     * @default - No template configuration
     */
    readonly templateConfigurationPath?: string;
}
/**
 * Options for the 'fromStackArtifact' operation
 */
export interface CdkStackActionFromArtifactOptions extends DeployCdkStackActionOptions {
    /**
     * The name of the stack that should be created/updated
     *
     * @default - Same as stack artifact
     */
    readonly stackName?: string;
}
/**
 * Action to deploy a CDK Stack
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 */
export declare class DeployCdkStackAction implements codepipeline.IAction {
    /**
     * Construct a DeployCdkStackAction from a Stack artifact
     */
    static fromStackArtifact(scope: Construct, artifact: cxapi.CloudFormationStackArtifact, options: CdkStackActionFromArtifactOptions): DeployCdkStackAction;
    /**
     * The runorder for the prepare action
     */
    readonly prepareRunOrder: number;
    /**
     * The runorder for the execute action
     */
    readonly executeRunOrder: number;
    /**
     * Name of the deployed stack
     */
    readonly stackName: string;
    /**
     * Artifact id of the artifact this action was based on
     */
    readonly stackArtifactId?: string;
    /**
     * Artifact ids of the artifact this stack artifact depends on
     */
    readonly dependencyStackArtifactIds: string[];
    private readonly prepareChangeSetAction;
    private readonly executeChangeSetAction;
    constructor(props: DeployCdkStackActionProps);
    /**
     * Exists to implement IAction
     */
    bind(scope: Construct, stage: codepipeline.IStage, options: codepipeline.ActionBindOptions): codepipeline.ActionConfig;
    /**
     * Exists to implement IAction
     */
    onStateChange(name: string, target?: events.IRuleTarget, options?: events.RuleProps): events.Rule;
    /**
     * Exists to implement IAction
     */
    get actionProperties(): codepipeline.ActionProperties;
}
/**
 * Options for CdkDeployAction.fromStackArtifact
 */
export interface FromStackArtifactOptions {
    /**
     * The CodePipeline artifact that holds the Cloud Assembly.
     */
    readonly cloudAssemblyInput: codepipeline.Artifact;
    /**
     * Run order for the 2 actions that will be created
     *
     * @default 1
     */
    readonly prepareRunOrder?: number;
    /**
     * Run order for the Execute action
     *
     * @default - prepareRunOrder + 1
     */
    readonly executeRunOrder?: number;
    /**
     * Artifact to write Stack Outputs to
     *
     * @default - No outputs
     */
    readonly output?: codepipeline.Artifact;
    /**
     * Filename in output to write Stack outputs to
     *
     * @default - Required when 'output' is set
     */
    readonly outputFileName?: string;
}
