"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationStackArtifact = void 0;
const fs = require("fs");
const path = require("path");
const cxschema = require("../../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cloud_artifact_1 = require("../cloud-artifact");
const environment_1 = require("../environment");
class CloudFormationStackArtifact extends cloud_artifact_1.CloudArtifact {
    constructor(assembly, artifactId, artifact) {
        var _a, _b;
        super(assembly, artifactId, artifact);
        const properties = (this.manifest.properties || {});
        if (!properties.templateFile) {
            throw new Error('Invalid CloudFormation stack artifact. Missing "templateFile" property in cloud assembly manifest');
        }
        if (!artifact.environment) {
            throw new Error('Invalid CloudFormation stack artifact. Missing environment');
        }
        this.environment = environment_1.EnvironmentUtils.parse(artifact.environment);
        this.templateFile = properties.templateFile;
        this.parameters = (_a = properties.parameters) !== null && _a !== void 0 ? _a : {};
        // We get the tags from 'properties' if available (cloud assembly format >= 6.0.0), otherwise
        // from the stack metadata
        this.tags = (_b = properties.tags) !== null && _b !== void 0 ? _b : this.tagsFromMetadata();
        this.assumeRoleArn = properties.assumeRoleArn;
        this.cloudFormationExecutionRoleArn = properties.cloudFormationExecutionRoleArn;
        this.stackTemplateAssetObjectUrl = properties.stackTemplateAssetObjectUrl;
        this.requiresBootstrapStackVersion = properties.requiresBootstrapStackVersion;
        this.terminationProtection = properties.terminationProtection;
        this.stackName = properties.stackName || artifactId;
        this.assets = this.findMetadataByType(cxschema.ArtifactMetadataEntryType.ASSET).map(e => e.data);
        this.displayName = this.stackName === artifactId
            ? this.stackName
            : `${artifactId} (${this.stackName})`;
        this.name = this.stackName; // backwards compat
        this.originalName = this.stackName;
    }
    /**
     * Full path to the template file
     */
    get templateFullPath() {
        return path.join(this.assembly.directory, this.templateFile);
    }
    /**
     * The CloudFormation template for this stack.
     */
    get template() {
        if (this._template === undefined) {
            this._template = JSON.parse(fs.readFileSync(this.templateFullPath, 'utf-8'));
        }
        return this._template;
    }
    tagsFromMetadata() {
        var _a;
        const ret = {};
        for (const metadataEntry of this.findMetadataByType(cxschema.ArtifactMetadataEntryType.STACK_TAGS)) {
            for (const tag of ((_a = metadataEntry.data) !== null && _a !== void 0 ? _a : [])) {
                ret[tag.key] = tag.value;
            }
        }
        return ret;
    }
}
exports.CloudFormationStackArtifact = CloudFormationStackArtifact;
//# sourceMappingURL=data:application/json;base64,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