"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Provider = void 0;
const path = require("path");
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const consts = require("./runtime/consts");
const util_1 = require("./util");
const waiter_state_machine_1 = require("./waiter-state-machine");
const RUNTIME_HANDLER_PATH = path.join(__dirname, 'runtime');
const FRAMEWORK_HANDLER_TIMEOUT = core_1.Duration.minutes(15); // keep it simple for now
/**
 * Defines an AWS CloudFormation custom resource provider.
 */
class Provider extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.isCompleteHandler && (props.queryInterval || props.totalTimeout)) {
            throw new Error('"queryInterval" and "totalTimeout" can only be configured if "isCompleteHandler" is specified. '
                + 'Otherwise, they have no meaning');
        }
        this.onEventHandler = props.onEventHandler;
        this.isCompleteHandler = props.isCompleteHandler;
        this.logRetention = props.logRetention;
        const onEventFunction = this.createFunction(consts.FRAMEWORK_ON_EVENT_HANDLER_NAME);
        if (this.isCompleteHandler) {
            const isCompleteFunction = this.createFunction(consts.FRAMEWORK_IS_COMPLETE_HANDLER_NAME);
            const timeoutFunction = this.createFunction(consts.FRAMEWORK_ON_TIMEOUT_HANDLER_NAME);
            const retry = util_1.calculateRetryPolicy(props);
            const waiterStateMachine = new waiter_state_machine_1.WaiterStateMachine(this, 'waiter-state-machine', {
                isCompleteHandler: isCompleteFunction,
                timeoutHandler: timeoutFunction,
                backoffRate: retry.backoffRate,
                interval: retry.interval,
                maxAttempts: retry.maxAttempts,
            });
            // the on-event entrypoint is going to start the execution of the waiter
            onEventFunction.addEnvironment(consts.WAITER_STATE_MACHINE_ARN_ENV, waiterStateMachine.stateMachineArn);
            waiterStateMachine.grantStartExecution(onEventFunction);
        }
        this.entrypoint = onEventFunction;
        this.serviceToken = this.entrypoint.functionArn;
    }
    /**
     * Called by `CustomResource` which uses this provider.
     * @deprecated use `provider.serviceToken` instead
     */
    bind(_) {
        return {
            serviceToken: this.entrypoint.functionArn,
        };
    }
    createFunction(entrypoint) {
        const fn = new lambda.Function(this, `framework-${entrypoint}`, {
            code: lambda.Code.fromAsset(RUNTIME_HANDLER_PATH),
            description: `AWS CDK resource provider framework - ${entrypoint} (${this.node.path})`.slice(0, 256),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: `framework.${entrypoint}`,
            timeout: FRAMEWORK_HANDLER_TIMEOUT,
            logRetention: this.logRetention,
        });
        fn.addEnvironment(consts.USER_ON_EVENT_FUNCTION_ARN_ENV, this.onEventHandler.functionArn);
        this.onEventHandler.grantInvoke(fn);
        if (this.isCompleteHandler) {
            fn.addEnvironment(consts.USER_IS_COMPLETE_FUNCTION_ARN_ENV, this.isCompleteHandler.functionArn);
            this.isCompleteHandler.grantInvoke(fn);
        }
        return fn;
    }
}
exports.Provider = Provider;
//# sourceMappingURL=data:application/json;base64,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