"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCustomResource = exports.AwsCustomResourcePolicy = exports.PhysicalResourceId = exports.PhysicalResourceIdReference = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const runtime_1 = require("./runtime");
// don't use "require" since the typescript compiler emits errors since this
// file is not listed in tsconfig.json.
const metadata = JSON.parse(fs.readFileSync(path.join(__dirname, 'sdk-api-metadata.json'), 'utf-8'));
const awsSdkMetadata = metadata;
/**
 * Reference to the physical resource id that can be passed to the AWS operation as a parameter.
 */
class PhysicalResourceIdReference {
    /**
     * toJSON serialization to replace `PhysicalResourceIdReference` with a magic string.
     */
    toJSON() {
        return runtime_1.PHYSICAL_RESOURCE_ID_REFERENCE;
    }
}
exports.PhysicalResourceIdReference = PhysicalResourceIdReference;
/**
 * Physical ID of the custom resource.
 */
class PhysicalResourceId {
    /**
     * @param responsePath Path to a response data element to be used as the physical id.
     * @param id Literal string to be used as the physical id.
     */
    constructor(responsePath, id) {
        this.responsePath = responsePath;
        this.id = id;
    }
    /**
     * Extract the physical resource id from the path (dot notation) to the data in the API call response.
     */
    static fromResponse(responsePath) {
        return new PhysicalResourceId(responsePath, undefined);
    }
    /**
     * Explicit physical resource id.
     */
    static of(id) {
        return new PhysicalResourceId(undefined, id);
    }
}
exports.PhysicalResourceId = PhysicalResourceId;
/**
 * The IAM Policy that will be applied to the different calls.
 */
class AwsCustomResourcePolicy {
    /**
     * @param statements statements for explicit policy.
     * @param resources resources for auto-generated from SDK calls.
     */
    constructor(statements, resources) {
        this.statements = statements;
        this.resources = resources;
    }
    /**
     * Explicit IAM Policy Statements.
     *
     * @param statements the statements to propagate to the SDK calls.
     */
    static fromStatements(statements) {
        return new AwsCustomResourcePolicy(statements, undefined);
    }
    /**
     * Generate IAM Policy Statements from the configured SDK calls.
     *
     * Each SDK call with be translated to an IAM Policy Statement in the form of: `call.service:call.action` (e.g `s3:PutObject`).
     *
     * @param options options for the policy generation
     */
    static fromSdkCalls(options) {
        return new AwsCustomResourcePolicy([], options.resources);
    }
}
exports.AwsCustomResourcePolicy = AwsCustomResourcePolicy;
/**
 * Use this constant to configure access to any resource.
 */
AwsCustomResourcePolicy.ANY_RESOURCE = ['*'];
/**
 * Defines a custom resource that is materialized using specific AWS API calls.
 *
 * Use this to bridge any gap that might exist in the CloudFormation Coverage.
 * You can specify exactly which calls are invoked for the 'CREATE', 'UPDATE' and 'DELETE' life cycle events.
 *
 */
class AwsCustomResource extends cdk.Construct {
    // 'props' cannot be optional, even though all its properties are optional.
    // this is because at least one sdk call must be provided.
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (!props.onCreate && !props.onUpdate && !props.onDelete) {
            throw new Error('At least `onCreate`, `onUpdate` or `onDelete` must be specified.');
        }
        for (const call of [props.onCreate, props.onUpdate]) {
            if (call && !call.physicalResourceId) {
                throw new Error('`physicalResourceId` must be specified for onCreate and onUpdate calls.');
            }
        }
        for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
            if ((_a = call === null || call === void 0 ? void 0 : call.physicalResourceId) === null || _a === void 0 ? void 0 : _a.responsePath) {
                AwsCustomResource.breakIgnoreErrorsCircuit([call], 'PhysicalResourceId.fromResponse');
            }
        }
        if ((_b = props.onCreate) === null || _b === void 0 ? void 0 : _b.parameters) {
            const flattenedOnCreateParams = runtime_1.flatten(JSON.parse(JSON.stringify(props.onCreate.parameters)));
            for (const param in flattenedOnCreateParams) {
                if (flattenedOnCreateParams[param] === runtime_1.PHYSICAL_RESOURCE_ID_REFERENCE) {
                    throw new Error('`PhysicalResourceIdReference` must not be specified in `onCreate` parameters.');
                }
            }
        }
        this.props = props;
        const provider = new lambda.SingletonFunction(this, 'Provider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: '679f53fa-c002-430c-b0da-5b7982bd2287',
            lambdaPurpose: 'AWS',
            timeout: props.timeout || cdk.Duration.minutes(2),
            role: props.role,
            logRetention: props.logRetention,
            functionName: props.functionName,
        });
        this.grantPrincipal = provider.grantPrincipal;
        // Create the policy statements for the custom resource function role, or use the user-provided ones
        const statements = [];
        if (props.policy.statements.length !== 0) {
            // Use custom statements provided by the user
            for (const statement of props.policy.statements) {
                statements.push(statement);
            }
        }
        else {
            // Derive statements from AWS SDK calls
            for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
                if (call) {
                    const statement = new iam.PolicyStatement({
                        actions: [awsSdkToIamAction(call.service, call.action)],
                        resources: props.policy.resources,
                    });
                    statements.push(statement);
                }
            }
        }
        const policy = new iam.Policy(this, 'CustomResourcePolicy', {
            statements: statements,
        });
        if (provider.role !== undefined) {
            policy.attachToRole(provider.role);
        }
        const create = props.onCreate || props.onUpdate;
        this.customResource = new cdk.CustomResource(this, 'Resource', {
            resourceType: props.resourceType || 'Custom::AWS',
            serviceToken: provider.functionArn,
            pascalCaseProperties: true,
            properties: {
                create: create && encodeBooleans(create),
                update: props.onUpdate && encodeBooleans(props.onUpdate),
                delete: props.onDelete && encodeBooleans(props.onDelete),
                installLatestAwsSdk: (_c = props.installLatestAwsSdk) !== null && _c !== void 0 ? _c : true,
            },
        });
        // If the policy was deleted first, then the function might lose permissions to delete the custom resource
        // This is here so that the policy doesn't get removed before onDelete is called
        this.customResource.node.addDependency(policy);
    }
    static breakIgnoreErrorsCircuit(sdkCalls, caller) {
        for (const call of sdkCalls) {
            if (call === null || call === void 0 ? void 0 : call.ignoreErrorCodesMatching) {
                throw new Error(`\`${caller}\`` + ' cannot be called along with `ignoreErrorCodesMatching`.');
            }
        }
    }
    /**
     * Returns response data for the AWS SDK call.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getDataString` for string attributes.
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseFieldReference(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getData');
        return this.customResource.getAtt(dataPath);
    }
    /**
     * Returns response data for the AWS SDK call as string.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseField(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getDataString');
        return this.customResource.getAttString(dataPath);
    }
}
exports.AwsCustomResource = AwsCustomResource;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 * Example: CloudWatchLogs with putRetentionPolicy => logs:PutRetentionPolicy
 *
 * TODO: is this mapping correct for all services?
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = (awsSdkMetadata[srv] && awsSdkMetadata[srv].prefix) || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
/**
 * Encodes booleans as special strings
 */
function encodeBooleans(object) {
    return JSON.parse(JSON.stringify(object), (_k, v) => {
        switch (v) {
            case true:
                return 'TRUE:BOOLEAN';
            case false:
                return 'FALSE:BOOLEAN';
            default:
                return v;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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