"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.withResolved = exports.isResolvableObject = exports.Tokenization = exports.Token = exports.TokenComparison = void 0;
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * An enum-like class that represents the result of comparing two Tokens.
 * The return type of {@link Token.compareStrings}.
 */
class TokenComparison {
    constructor() {
    }
}
exports.TokenComparison = TokenComparison;
/**
 * This means we're certain the two components are NOT
 * Tokens, and identical.
 */
TokenComparison.SAME = new TokenComparison();
/**
 * This means we're certain the two components are NOT
 * Tokens, and different.
 */
TokenComparison.DIFFERENT = new TokenComparison();
/** This means exactly one of the components is a Token. */
TokenComparison.ONE_UNRESOLVED = new TokenComparison();
/** This means both components are Tokens. */
TokenComparison.BOTH_UNRESOLVED = new TokenComparison();
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === 'string') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === 'number') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every(x => typeof x === 'string')) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    /** Compare two strings that might contain Tokens with each other. */
    static compareStrings(possibleToken1, possibleToken2) {
        const firstIsUnresolved = Token.isUnresolved(possibleToken1);
        const secondIsUnresolved = Token.isUnresolved(possibleToken2);
        if (firstIsUnresolved && secondIsUnresolved) {
            return TokenComparison.BOTH_UNRESOLVED;
        }
        if (firstIsUnresolved || secondIsUnresolved) {
            return TokenComparison.ONE_UNRESOLVED;
        }
        return possibleToken1 === possibleToken2 ? TokenComparison.SAME : TokenComparison.DIFFERENT;
    }
    constructor() {
    }
}
exports.Token = Token;
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: (options.preparing !== undefined ? options.preparing : false),
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return isResolvableObject(obj);
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.stringValue({
                produce: context => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== 'number' ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== 'number' ? x : `${x}`;
        }
    }
    constructor() {
    }
}
exports.Tokenization = Tokenization;
function isResolvableObject(x) {
    return typeof (x) === 'object' && x !== null && typeof x.resolve === 'function';
}
exports.isResolvableObject = isResolvableObject;
function withResolved(...args) {
    if (args.length < 2) {
        return;
    }
    const argArray = args.slice(0, args.length - 1);
    if (argArray.some(Token.isUnresolved)) {
        return;
    }
    args[args.length - 1].apply(arguments, argArray);
}
exports.withResolved = withResolved;
//# sourceMappingURL=data:application/json;base64,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