"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs_1 = require("constructs");
const construct_compat_1 = require("./construct-compat");
const synthesis_1 = require("./private/synthesis");
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        this.parentStage = Stage.of(this);
        this.region = (_b = (_a = props.env) === null || _a === void 0 ? void 0 : _a.region) !== null && _b !== void 0 ? _b : (_c = this.parentStage) === null || _c === void 0 ? void 0 : _c.region;
        this.account = (_e = (_d = props.env) === null || _d === void 0 ? void 0 : _d.account) !== null && _e !== void 0 ? _e : (_f = this.parentStage) === null || _f === void 0 ? void 0 : _f.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_g = this.parentStage) === null || _g === void 0 ? void 0 : _g.stageName, id].filter(x => x).join('-');
    }
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return constructs_1.Node.of(construct).scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && x instanceof Stage;
    }
    /**
     * The cloud assembly output directory.
     */
    get outdir() {
        return this._assemblyBuilder.outdir;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        if (!this.assembly || options.force) {
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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