"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
// v2 - leave this as a separate section so it reduces merge conflicts when compat is removed
// eslint-disable-next-line import/order
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        this.env = {
            account: (_a = props.account) !== null && _a !== void 0 ? _a : this.stack.account,
            region: (_b = props.region) !== null && _b !== void 0 ? _b : this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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