"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnReference = exports.ReferenceRendering = void 0;
const reference_1 = require("../reference");
const CFN_REFERENCE_SYMBOL = Symbol.for('@aws-cdk/core.CfnReference');
/**
 * An enum that allows controlling how will the created reference
 * be rendered in the resulting CloudFormation template.
 */
var ReferenceRendering;
(function (ReferenceRendering) {
    /**
     * Used for rendering a reference inside Fn::Sub expressions,
     * which mean these must resolve to "${Sth}" instead of { Ref: "Sth" }.
     */
    ReferenceRendering[ReferenceRendering["FN_SUB"] = 0] = "FN_SUB";
    /**
     * Used for rendering Fn::GetAtt with its arguments in string form
     * (as opposed to the more common arguments in array form, which we render by default).
     */
    ReferenceRendering[ReferenceRendering["GET_ATT_STRING"] = 1] = "GET_ATT_STRING";
})(ReferenceRendering = exports.ReferenceRendering || (exports.ReferenceRendering = {}));
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class CfnReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        // prepend scope path to display name
        super(value, target, displayName);
        this.replacementTokens = new Map();
        this.targetStack = stack_1.Stack.of(target);
        Object.defineProperty(this, CFN_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x) {
        return CFN_REFERENCE_SYMBOL in x;
    }
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.stringValue({ produce: () => new CfnReference(...) })
     *
     */
    static for(target, attribute, refRender) {
        return CfnReference.singletonReference(target, attribute, refRender, () => {
            const cfnIntrinsic = refRender === ReferenceRendering.FN_SUB
                ? ('${' + target.logicalId + (attribute === 'Ref' ? '' : `.${attribute}`) + '}')
                : (attribute === 'Ref'
                    ? { Ref: target.logicalId }
                    : {
                        'Fn::GetAtt': refRender === ReferenceRendering.GET_ATT_STRING
                            ? `${target.logicalId}.${attribute}`
                            : [target.logicalId, attribute],
                    });
            return new CfnReference(cfnIntrinsic, attribute, target);
        });
    }
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return CfnReference.singletonReference(scope, `Pseudo:${pseudoName}`, undefined, () => {
            const cfnIntrinsic = { Ref: pseudoName };
            return new CfnReference(cfnIntrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table.
     * Passing fnSub = true allows cloudformation-include to correctly handle Fn::Sub.
     */
    static singletonReference(target, attribKey, refRender, fresh) {
        let attribs = CfnReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            CfnReference.referenceTable.set(target, attribs);
        }
        let cacheKey = attribKey;
        switch (refRender) {
            case ReferenceRendering.FN_SUB:
                cacheKey += 'Fn::Sub';
                break;
            case ReferenceRendering.GET_ATT_STRING:
                cacheKey += 'Fn::GetAtt::String';
                break;
        }
        let ref = attribs.get(cacheKey);
        if (!ref) {
            ref = fresh();
            attribs.set(cacheKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const consumingStack = stack_1.Stack.of(context.scope);
        const token = this.replacementTokens.get(consumingStack);
        // if (!token && this.isCrossStackReference(consumingStack) && !context.preparing) {
        // eslint-disable-next-line max-len
        //   throw new Error(`Cross-stack reference (${context.scope.node.path} -> ${this.target.node.path}) has not been assigned a value--call prepare() first`);
        // }
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    hasValueForStack(stack) {
        if (stack === this.targetStack) {
            return true;
        }
        return this.replacementTokens.has(stack);
    }
    assignValueForStack(stack, value) {
        if (stack === this.targetStack) {
            throw new Error('cannot assign a value for the same stack');
        }
        if (this.hasValueForStack(stack)) {
            throw new Error('Cannot assign a reference value twice to the same stack. Use hasValueForStack to check first');
        }
        this.replacementTokens.set(stack, value);
    }
    /**
     * Implementation of toString() that will use the display name
     */
    toString() {
        return token_1.Token.asString(this, {
            displayHint: `${this.target.node.id}.${this.displayName}`,
        });
    }
}
exports.CfnReference = CfnReference;
/**
 * Static table where we keep singleton CfnReference instances
 */
CfnReference.referenceTable = new Map();
const stack_1 = require("../stack");
const token_1 = require("../token");
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLXJlZmVyZW5jZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNmbi1yZWZlcmVuY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNENBQXlDO0FBQ3pDLE1BQU0sb0JBQW9CLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0FBQ3RFOzs7R0FHRztBQUNILElBQVksa0JBV1g7QUFYRCxXQUFZLGtCQUFrQjtJQUMxQjs7O09BR0c7SUFDSCwrREFBTSxDQUFBO0lBQ047OztPQUdHO0lBQ0gsK0VBQWMsQ0FBQTtBQUNsQixDQUFDLEVBWFcsa0JBQWtCLEdBQWxCLDBCQUFrQixLQUFsQiwwQkFBa0IsUUFXN0I7QUFDRDs7Ozs7Ozs7Ozs7OztHQWFHO0FBQ0gsTUFBYSxZQUFhLFNBQVEscUJBQVM7SUE0RXZDLFlBQXNCLEtBQVUsRUFBRSxXQUFtQixFQUFFLE1BQWtCO1FBQ3JFLHFDQUFxQztRQUNyQyxLQUFLLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNsQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxHQUFHLEVBQXNCLENBQUM7UUFDdkQsSUFBSSxDQUFDLFdBQVcsR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3BDLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7SUFDdkUsQ0FBQztJQWpGRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxjQUFjLENBQUMsQ0FBYztRQUN2QyxPQUFPLG9CQUFvQixJQUFJLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7T0FVRztJQUNJLE1BQU0sQ0FBQyxHQUFHLENBQUMsTUFBa0IsRUFBRSxTQUFpQixFQUFFLFNBQThCO1FBQ25GLE9BQU8sWUFBWSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLEdBQUcsRUFBRTtZQUN0RSxNQUFNLFlBQVksR0FBRyxTQUFTLEtBQUssa0JBQWtCLENBQUMsTUFBTTtnQkFDeEQsQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLE1BQU0sQ0FBQyxTQUFTLEdBQUcsQ0FBQyxTQUFTLEtBQUssS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksU0FBUyxFQUFFLENBQUMsR0FBRyxHQUFHLENBQUM7Z0JBQ2hGLENBQUMsQ0FBQyxDQUFDLFNBQVMsS0FBSyxLQUFLO29CQUNsQixDQUFDLENBQUMsRUFBRSxHQUFHLEVBQUUsTUFBTSxDQUFDLFNBQVMsRUFBRTtvQkFDM0IsQ0FBQyxDQUFDO3dCQUNFLFlBQVksRUFBRSxTQUFTLEtBQUssa0JBQWtCLENBQUMsY0FBYzs0QkFDekQsQ0FBQyxDQUFDLEdBQUcsTUFBTSxDQUFDLFNBQVMsSUFBSSxTQUFTLEVBQUU7NEJBQ3BDLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO3FCQUN0QyxDQUFDLENBQUM7WUFDWCxPQUFPLElBQUksWUFBWSxDQUFDLFlBQVksRUFBRSxTQUFTLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDN0QsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLFVBQWtCLEVBQUUsS0FBZ0I7UUFDeEQsT0FBTyxZQUFZLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLFVBQVUsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLEdBQUcsRUFBRTtZQUNsRixNQUFNLFlBQVksR0FBRyxFQUFFLEdBQUcsRUFBRSxVQUFVLEVBQUUsQ0FBQztZQUN6QyxPQUFPLElBQUksWUFBWSxDQUFDLFlBQVksRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDN0QsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBS0Q7OztPQUdHO0lBQ0ssTUFBTSxDQUFDLGtCQUFrQixDQUFDLE1BQWlCLEVBQUUsU0FBaUIsRUFBRSxTQUF5QyxFQUFFLEtBQXlCO1FBQ3hJLElBQUksT0FBTyxHQUFHLFlBQVksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDVixPQUFPLEdBQUcsSUFBSSxHQUFHLEVBQUUsQ0FBQztZQUNwQixZQUFZLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7U0FDcEQ7UUFDRCxJQUFJLFFBQVEsR0FBRyxTQUFTLENBQUM7UUFDekIsUUFBUSxTQUFTLEVBQUU7WUFDZixLQUFLLGtCQUFrQixDQUFDLE1BQU07Z0JBQzFCLFFBQVEsSUFBSSxTQUFTLENBQUM7Z0JBQ3RCLE1BQU07WUFDVixLQUFLLGtCQUFrQixDQUFDLGNBQWM7Z0JBQ2xDLFFBQVEsSUFBSSxvQkFBb0IsQ0FBQztnQkFDakMsTUFBTTtTQUNiO1FBQ0QsSUFBSSxHQUFHLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNoQyxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ04sR0FBRyxHQUFHLEtBQUssRUFBRSxDQUFDO1lBQ2QsT0FBTyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDOUI7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFhTSxPQUFPLENBQUMsT0FBd0I7UUFDbkMsNkZBQTZGO1FBQzdGLDRCQUE0QjtRQUM1QixNQUFNLGNBQWMsR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3pELG9GQUFvRjtRQUNwRixtQ0FBbUM7UUFDbkMsMkpBQTJKO1FBQzNKLElBQUk7UUFDSixJQUFJLEtBQUssRUFBRTtZQUNQLE9BQU8sS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUNqQzthQUNJO1lBQ0QsT0FBTyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ2pDO0lBQ0wsQ0FBQztJQUNNLGdCQUFnQixDQUFDLEtBQVk7UUFDaEMsSUFBSSxLQUFLLEtBQUssSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUM1QixPQUFPLElBQUksQ0FBQztTQUNmO1FBQ0QsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFDTSxtQkFBbUIsQ0FBQyxLQUFZLEVBQUUsS0FBa0I7UUFDdkQsSUFBSSxLQUFLLEtBQUssSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7U0FDL0Q7UUFDRCxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUM5QixNQUFNLElBQUksS0FBSyxDQUFDLDhGQUE4RixDQUFDLENBQUM7U0FDbkg7UUFDRCxJQUFJLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxRQUFRO1FBQ1gsT0FBTyxhQUFLLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRTtZQUN4QixXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtTQUM1RCxDQUFDLENBQUM7SUFDUCxDQUFDOztBQXpITCxvQ0EwSEM7QUFqRkc7O0dBRUc7QUFDWSwyQkFBYyxHQUFHLElBQUksR0FBRyxFQUF3QyxDQUFDO0FBa0ZwRixvQ0FBaUM7QUFDakMsb0NBQWlDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgUmVmZXJlbmNlIH0gZnJvbSAnLi4vcmVmZXJlbmNlJztcbmNvbnN0IENGTl9SRUZFUkVOQ0VfU1lNQk9MID0gU3ltYm9sLmZvcignQGF3cy1jZGsvY29yZS5DZm5SZWZlcmVuY2UnKTtcbi8qKlxuICogQW4gZW51bSB0aGF0IGFsbG93cyBjb250cm9sbGluZyBob3cgd2lsbCB0aGUgY3JlYXRlZCByZWZlcmVuY2VcbiAqIGJlIHJlbmRlcmVkIGluIHRoZSByZXN1bHRpbmcgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUuXG4gKi9cbmV4cG9ydCBlbnVtIFJlZmVyZW5jZVJlbmRlcmluZyB7XG4gICAgLyoqXG4gICAgICogVXNlZCBmb3IgcmVuZGVyaW5nIGEgcmVmZXJlbmNlIGluc2lkZSBGbjo6U3ViIGV4cHJlc3Npb25zLFxuICAgICAqIHdoaWNoIG1lYW4gdGhlc2UgbXVzdCByZXNvbHZlIHRvIFwiJHtTdGh9XCIgaW5zdGVhZCBvZiB7IFJlZjogXCJTdGhcIiB9LlxuICAgICAqL1xuICAgIEZOX1NVQixcbiAgICAvKipcbiAgICAgKiBVc2VkIGZvciByZW5kZXJpbmcgRm46OkdldEF0dCB3aXRoIGl0cyBhcmd1bWVudHMgaW4gc3RyaW5nIGZvcm1cbiAgICAgKiAoYXMgb3Bwb3NlZCB0byB0aGUgbW9yZSBjb21tb24gYXJndW1lbnRzIGluIGFycmF5IGZvcm0sIHdoaWNoIHdlIHJlbmRlciBieSBkZWZhdWx0KS5cbiAgICAgKi9cbiAgICBHRVRfQVRUX1NUUklOR1xufVxuLyoqXG4gKiBBIFRva2VuIHRoYXQgcmVwcmVzZW50cyBhIENsb3VkRm9ybWF0aW9uIHJlZmVyZW5jZSB0byBhbm90aGVyIHJlc291cmNlXG4gKlxuICogSWYgdGhlc2UgcmVmZXJlbmNlcyBhcmUgdXNlZCBpbiBhIGRpZmZlcmVudCBzdGFjayBmcm9tIHdoZXJlIHRoZXkgYXJlXG4gKiBkZWZpbmVkLCBhcHByb3ByaWF0ZSBDbG91ZEZvcm1hdGlvbiBgRXhwb3J0YHMgYW5kIGBGbjo6SW1wb3J0VmFsdWVgcyB3aWxsIGJlXG4gKiBzeW50aGVzaXplZCBhdXRvbWF0aWNhbGx5IGluc3RlYWQgb2YgdGhlIHJlZ3VsYXIgQ2xvdWRGb3JtYXRpb24gcmVmZXJlbmNlcy5cbiAqXG4gKiBBZGRpdGlvbmFsbHksIHRoZSBkZXBlbmRlbmN5IGJldHdlZW4gdGhlIHN0YWNrcyB3aWxsIGJlIHJlY29yZGVkLCBhbmQgdGhlIHRvb2xraXRcbiAqIHdpbGwgbWFrZSBzdXJlIHRvIGRlcGxveSBwcm9kdWNpbmcgc3RhY2sgYmVmb3JlIHRoZSBjb25zdW1pbmcgc3RhY2suXG4gKlxuICogVGhpcyBtYWdpYyBoYXBwZW5zIGluIHRoZSBwcmVwYXJlKCkgcGhhc2UsIHdoZXJlIGNvbnN1bWluZyBzdGFja3Mgd2lsbCBjYWxsXG4gKiBgY29uc3VtZUZyb21TdGFja2Agb24gdGhlc2UgVG9rZW5zIGFuZCBpZiB0aGV5IGhhcHBlbiB0byBiZSBleHBvcnRlZCBieSBhIGRpZmZlcmVudFxuICogU3RhY2ssIHdlJ2xsIHJlZ2lzdGVyIHRoZSBkZXBlbmRlbmN5LlxuICovXG5leHBvcnQgY2xhc3MgQ2ZuUmVmZXJlbmNlIGV4dGVuZHMgUmVmZXJlbmNlIHtcbiAgICAvKipcbiAgICAgKiBDaGVjayB3aGV0aGVyIHRoaXMgaXMgYWN0dWFsbHkgYSBSZWZlcmVuY2VcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGlzQ2ZuUmVmZXJlbmNlKHg6IElSZXNvbHZhYmxlKTogeCBpcyBDZm5SZWZlcmVuY2Uge1xuICAgICAgICByZXR1cm4gQ0ZOX1JFRkVSRU5DRV9TWU1CT0wgaW4geDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBDZm5SZWZlcmVuY2UgZm9yIHRoZSBpbmRpY2F0ZWQgdGFyZ2V0XG4gICAgICpcbiAgICAgKiBXaWxsIG1ha2Ugc3VyZSB0aGF0IG11bHRpcGxlIGludm9jYXRpb25zIGZvciB0aGUgc2FtZSB0YXJnZXQgYW5kIGludHJpbnNpY1xuICAgICAqIHJldHVybiB0aGUgc2FtZSBDZm5SZWZlcmVuY2UuIEJlY2F1c2UgQ2ZuUmVmZXJlbmNlcyBhY2N1bXVsYXRlIHN0YXRlIGluXG4gICAgICogdGhlIHByZXBhcmUoKSBwaGFzZSAoZm9yIHRoZSBwdXJwb3NlIG9mIGNyb3NzLXN0YWNrIHJlZmVyZW5jZXMpLCBpdCdzXG4gICAgICogaW1wb3J0YW50IHRoYXQgdGhlIHN0YXRlIGlzbid0IGxvc3QgaWYgaXQncyBsYXppbHkgY3JlYXRlZCwgbGlrZSBzbzpcbiAgICAgKlxuICAgICAqICAgICBMYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT4gbmV3IENmblJlZmVyZW5jZSguLi4pIH0pXG4gICAgICpcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZvcih0YXJnZXQ6IENmbkVsZW1lbnQsIGF0dHJpYnV0ZTogc3RyaW5nLCByZWZSZW5kZXI/OiBSZWZlcmVuY2VSZW5kZXJpbmcpIHtcbiAgICAgICAgcmV0dXJuIENmblJlZmVyZW5jZS5zaW5nbGV0b25SZWZlcmVuY2UodGFyZ2V0LCBhdHRyaWJ1dGUsIHJlZlJlbmRlciwgKCkgPT4ge1xuICAgICAgICAgICAgY29uc3QgY2ZuSW50cmluc2ljID0gcmVmUmVuZGVyID09PSBSZWZlcmVuY2VSZW5kZXJpbmcuRk5fU1VCXG4gICAgICAgICAgICAgICAgPyAoJyR7JyArIHRhcmdldC5sb2dpY2FsSWQgKyAoYXR0cmlidXRlID09PSAnUmVmJyA/ICcnIDogYC4ke2F0dHJpYnV0ZX1gKSArICd9JylcbiAgICAgICAgICAgICAgICA6IChhdHRyaWJ1dGUgPT09ICdSZWYnXG4gICAgICAgICAgICAgICAgICAgID8geyBSZWY6IHRhcmdldC5sb2dpY2FsSWQgfVxuICAgICAgICAgICAgICAgICAgICA6IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogcmVmUmVuZGVyID09PSBSZWZlcmVuY2VSZW5kZXJpbmcuR0VUX0FUVF9TVFJJTkdcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICA/IGAke3RhcmdldC5sb2dpY2FsSWR9LiR7YXR0cmlidXRlfWBcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICA6IFt0YXJnZXQubG9naWNhbElkLCBhdHRyaWJ1dGVdLFxuICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIHJldHVybiBuZXcgQ2ZuUmVmZXJlbmNlKGNmbkludHJpbnNpYywgYXR0cmlidXRlLCB0YXJnZXQpO1xuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIGEgQ2ZuUmVmZXJlbmNlIHRoYXQgcmVmZXJlbmNlcyBhIHBzZXVkbyByZWZlcmVuY2RcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZvclBzZXVkbyhwc2V1ZG9OYW1lOiBzdHJpbmcsIHNjb3BlOiBDb25zdHJ1Y3QpIHtcbiAgICAgICAgcmV0dXJuIENmblJlZmVyZW5jZS5zaW5nbGV0b25SZWZlcmVuY2Uoc2NvcGUsIGBQc2V1ZG86JHtwc2V1ZG9OYW1lfWAsIHVuZGVmaW5lZCwgKCkgPT4ge1xuICAgICAgICAgICAgY29uc3QgY2ZuSW50cmluc2ljID0geyBSZWY6IHBzZXVkb05hbWUgfTtcbiAgICAgICAgICAgIHJldHVybiBuZXcgQ2ZuUmVmZXJlbmNlKGNmbkludHJpbnNpYywgcHNldWRvTmFtZSwgc2NvcGUpO1xuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogU3RhdGljIHRhYmxlIHdoZXJlIHdlIGtlZXAgc2luZ2xldG9uIENmblJlZmVyZW5jZSBpbnN0YW5jZXNcbiAgICAgKi9cbiAgICBwcml2YXRlIHN0YXRpYyByZWZlcmVuY2VUYWJsZSA9IG5ldyBNYXA8Q29uc3RydWN0LCBNYXA8c3RyaW5nLCBDZm5SZWZlcmVuY2U+PigpO1xuICAgIC8qKlxuICAgICAqIEdldCBvciBjcmVhdGUgdGhlIHRhYmxlLlxuICAgICAqIFBhc3NpbmcgZm5TdWIgPSB0cnVlIGFsbG93cyBjbG91ZGZvcm1hdGlvbi1pbmNsdWRlIHRvIGNvcnJlY3RseSBoYW5kbGUgRm46OlN1Yi5cbiAgICAgKi9cbiAgICBwcml2YXRlIHN0YXRpYyBzaW5nbGV0b25SZWZlcmVuY2UodGFyZ2V0OiBDb25zdHJ1Y3QsIGF0dHJpYktleTogc3RyaW5nLCByZWZSZW5kZXI6IFJlZmVyZW5jZVJlbmRlcmluZyB8IHVuZGVmaW5lZCwgZnJlc2g6ICgpID0+IENmblJlZmVyZW5jZSkge1xuICAgICAgICBsZXQgYXR0cmlicyA9IENmblJlZmVyZW5jZS5yZWZlcmVuY2VUYWJsZS5nZXQodGFyZ2V0KTtcbiAgICAgICAgaWYgKCFhdHRyaWJzKSB7XG4gICAgICAgICAgICBhdHRyaWJzID0gbmV3IE1hcCgpO1xuICAgICAgICAgICAgQ2ZuUmVmZXJlbmNlLnJlZmVyZW5jZVRhYmxlLnNldCh0YXJnZXQsIGF0dHJpYnMpO1xuICAgICAgICB9XG4gICAgICAgIGxldCBjYWNoZUtleSA9IGF0dHJpYktleTtcbiAgICAgICAgc3dpdGNoIChyZWZSZW5kZXIpIHtcbiAgICAgICAgICAgIGNhc2UgUmVmZXJlbmNlUmVuZGVyaW5nLkZOX1NVQjpcbiAgICAgICAgICAgICAgICBjYWNoZUtleSArPSAnRm46OlN1Yic7XG4gICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICBjYXNlIFJlZmVyZW5jZVJlbmRlcmluZy5HRVRfQVRUX1NUUklORzpcbiAgICAgICAgICAgICAgICBjYWNoZUtleSArPSAnRm46OkdldEF0dDo6U3RyaW5nJztcbiAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgfVxuICAgICAgICBsZXQgcmVmID0gYXR0cmlicy5nZXQoY2FjaGVLZXkpO1xuICAgICAgICBpZiAoIXJlZikge1xuICAgICAgICAgICAgcmVmID0gZnJlc2goKTtcbiAgICAgICAgICAgIGF0dHJpYnMuc2V0KGNhY2hlS2V5LCByZWYpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZWY7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSBUb2tlbnMgdGhhdCBzaG91bGQgYmUgcmV0dXJuZWQgZm9yIGVhY2ggY29uc3VtaW5nIHN0YWNrIChhcyBkZWNpZGVkIGJ5IHRoZSBwcm9kdWNpbmcgU3RhY2spXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSByZXBsYWNlbWVudFRva2VuczogTWFwPFN0YWNrLCBJUmVzb2x2YWJsZT47XG4gICAgcHJpdmF0ZSByZWFkb25seSB0YXJnZXRTdGFjazogU3RhY2s7XG4gICAgcHJvdGVjdGVkIGNvbnN0cnVjdG9yKHZhbHVlOiBhbnksIGRpc3BsYXlOYW1lOiBzdHJpbmcsIHRhcmdldDogSUNvbnN0cnVjdCkge1xuICAgICAgICAvLyBwcmVwZW5kIHNjb3BlIHBhdGggdG8gZGlzcGxheSBuYW1lXG4gICAgICAgIHN1cGVyKHZhbHVlLCB0YXJnZXQsIGRpc3BsYXlOYW1lKTtcbiAgICAgICAgdGhpcy5yZXBsYWNlbWVudFRva2VucyA9IG5ldyBNYXA8U3RhY2ssIElSZXNvbHZhYmxlPigpO1xuICAgICAgICB0aGlzLnRhcmdldFN0YWNrID0gU3RhY2sub2YodGFyZ2V0KTtcbiAgICAgICAgT2JqZWN0LmRlZmluZVByb3BlcnR5KHRoaXMsIENGTl9SRUZFUkVOQ0VfU1lNQk9MLCB7IHZhbHVlOiB0cnVlIH0pO1xuICAgIH1cbiAgICBwdWJsaWMgcmVzb2x2ZShjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpOiBhbnkge1xuICAgICAgICAvLyBJZiB3ZSBoYXZlIGEgc3BlY2lhbCB0b2tlbiBmb3IgdGhpcyBjb25zdW1pbmcgc3RhY2ssIHJlc29sdmUgdGhhdC4gT3RoZXJ3aXNlIHJlc29sdmUgYXMgaWZcbiAgICAgICAgLy8gd2UgYXJlIGluIHRoZSBzYW1lIHN0YWNrLlxuICAgICAgICBjb25zdCBjb25zdW1pbmdTdGFjayA9IFN0YWNrLm9mKGNvbnRleHQuc2NvcGUpO1xuICAgICAgICBjb25zdCB0b2tlbiA9IHRoaXMucmVwbGFjZW1lbnRUb2tlbnMuZ2V0KGNvbnN1bWluZ1N0YWNrKTtcbiAgICAgICAgLy8gaWYgKCF0b2tlbiAmJiB0aGlzLmlzQ3Jvc3NTdGFja1JlZmVyZW5jZShjb25zdW1pbmdTdGFjaykgJiYgIWNvbnRleHQucHJlcGFyaW5nKSB7XG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgICAgIC8vICAgdGhyb3cgbmV3IEVycm9yKGBDcm9zcy1zdGFjayByZWZlcmVuY2UgKCR7Y29udGV4dC5zY29wZS5ub2RlLnBhdGh9IC0+ICR7dGhpcy50YXJnZXQubm9kZS5wYXRofSkgaGFzIG5vdCBiZWVuIGFzc2lnbmVkIGEgdmFsdWUtLWNhbGwgcHJlcGFyZSgpIGZpcnN0YCk7XG4gICAgICAgIC8vIH1cbiAgICAgICAgaWYgKHRva2VuKSB7XG4gICAgICAgICAgICByZXR1cm4gdG9rZW4ucmVzb2x2ZShjb250ZXh0KTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHJldHVybiBzdXBlci5yZXNvbHZlKGNvbnRleHQpO1xuICAgICAgICB9XG4gICAgfVxuICAgIHB1YmxpYyBoYXNWYWx1ZUZvclN0YWNrKHN0YWNrOiBTdGFjaykge1xuICAgICAgICBpZiAoc3RhY2sgPT09IHRoaXMudGFyZ2V0U3RhY2spIHtcbiAgICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLnJlcGxhY2VtZW50VG9rZW5zLmhhcyhzdGFjayk7XG4gICAgfVxuICAgIHB1YmxpYyBhc3NpZ25WYWx1ZUZvclN0YWNrKHN0YWNrOiBTdGFjaywgdmFsdWU6IElSZXNvbHZhYmxlKSB7XG4gICAgICAgIGlmIChzdGFjayA9PT0gdGhpcy50YXJnZXRTdGFjaykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdjYW5ub3QgYXNzaWduIGEgdmFsdWUgZm9yIHRoZSBzYW1lIHN0YWNrJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHRoaXMuaGFzVmFsdWVGb3JTdGFjayhzdGFjaykpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGFzc2lnbiBhIHJlZmVyZW5jZSB2YWx1ZSB0d2ljZSB0byB0aGUgc2FtZSBzdGFjay4gVXNlIGhhc1ZhbHVlRm9yU3RhY2sgdG8gY2hlY2sgZmlyc3QnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnJlcGxhY2VtZW50VG9rZW5zLnNldChzdGFjaywgdmFsdWUpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBJbXBsZW1lbnRhdGlvbiBvZiB0b1N0cmluZygpIHRoYXQgd2lsbCB1c2UgdGhlIGRpc3BsYXkgbmFtZVxuICAgICAqL1xuICAgIHB1YmxpYyB0b1N0cmluZygpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gVG9rZW4uYXNTdHJpbmcodGhpcywge1xuICAgICAgICAgICAgZGlzcGxheUhpbnQ6IGAke3RoaXMudGFyZ2V0Lm5vZGUuaWR9LiR7dGhpcy5kaXNwbGF5TmFtZX1gLFxuICAgICAgICB9KTtcbiAgICB9XG59XG5pbXBvcnQgeyBDZm5FbGVtZW50IH0gZnJvbSAnLi4vY2ZuLWVsZW1lbnQnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJQ29uc3RydWN0IH0gZnJvbSAnLi4vY29uc3RydWN0LWNvbXBhdCc7XG5pbXBvcnQgeyBJUmVzb2x2YWJsZSwgSVJlc29sdmVDb250ZXh0IH0gZnJvbSAnLi4vcmVzb2x2YWJsZSc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJy4uL3N0YWNrJztcbmltcG9ydCB7IFRva2VuIH0gZnJvbSAnLi4vdG9rZW4nO1xuIl19