"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expiration = void 0;
/**
 * Represents a date of expiration.
 *
 * The amount can be specified either as a Date object, timestamp, Duration or string.
 */
class Expiration {
    constructor(date) {
        this.date = date;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { return new Expiration(d); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { return Expiration.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { return Expiration.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Expire at specified date, represented as a string
     *
     * @param s the string that represents date to expire at
     */
    static fromString(s) { return new Expiration(new Date(s)); }
    /**
     * Exipration Value in a formatted Unix Epoch Time in seconds
     */
    toEpoch() {
        return Math.round(this.date.getTime() / 1000);
    }
    /**
     * Check if Exipiration expires before input
     * @param t the duration to check against
     */
    isBefore(t) {
        return this.date < new Date(Date.now() + t.toMilliseconds());
    }
    /**
     * Check if Exipiration expires after input
     * @param t the duration to check against
     */
    isAfter(t) {
        return this.date > new Date(Date.now() + t.toMilliseconds());
    }
}
exports.Expiration = Expiration;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXhwaXJhdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImV4cGlyYXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0E7Ozs7R0FJRztBQUNILE1BQWEsVUFBVTtJQTBCbkIsWUFBb0IsSUFBVTtRQUMxQixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztJQUNyQixDQUFDO0lBM0JEOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBTyxJQUFJLE9BQU8sSUFBSSxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzNEOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBUyxJQUFJLE9BQU8sVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMvRTs7O09BR0c7SUFDSSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQVcsSUFBSSxPQUFPLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3pHOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLENBQVMsSUFBSSxPQUFPLElBQUksVUFBVSxDQUFDLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBUTNFOztPQUVHO0lBQ0ksT0FBTztRQUNWLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFDRDs7O09BR0c7SUFDSSxRQUFRLENBQUMsQ0FBVztRQUN2QixPQUFPLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFDRDs7O09BR0c7SUFDSSxPQUFPLENBQUMsQ0FBVztRQUN0QixPQUFPLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7Q0FDSjtBQWpERCxnQ0FpREMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBEdXJhdGlvbiB9IGZyb20gJy4vZHVyYXRpb24nO1xuLyoqXG4gKiBSZXByZXNlbnRzIGEgZGF0ZSBvZiBleHBpcmF0aW9uLlxuICpcbiAqIFRoZSBhbW91bnQgY2FuIGJlIHNwZWNpZmllZCBlaXRoZXIgYXMgYSBEYXRlIG9iamVjdCwgdGltZXN0YW1wLCBEdXJhdGlvbiBvciBzdHJpbmcuXG4gKi9cbmV4cG9ydCBjbGFzcyBFeHBpcmF0aW9uIHtcbiAgICAvKipcbiAgICAgKiBFeHBpcmUgYXQgdGhlIHNwZWNpZmllZCBkYXRlXG4gICAgICogQHBhcmFtIGQgZGF0ZSB0byBleHBpcmUgYXRcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGF0RGF0ZShkOiBEYXRlKSB7IHJldHVybiBuZXcgRXhwaXJhdGlvbihkKTsgfVxuICAgIC8qKlxuICAgICAqIEV4cGlyZSBhdCB0aGUgc3BlY2lmaWVkIHRpbWVzdGFtcFxuICAgICAqIEBwYXJhbSB0IHRpbWVzdGFtcCBpbiB1bml4IG1pbGxpc2Vjb25kc1xuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgYXRUaW1lc3RhbXAodDogbnVtYmVyKSB7IHJldHVybiBFeHBpcmF0aW9uLmF0RGF0ZShuZXcgRGF0ZSh0KSk7IH1cbiAgICAvKipcbiAgICAgKiBFeHBpcmUgb25jZSB0aGUgc3BlY2lmaWVkIGR1cmF0aW9uIGhhcyBwYXNzZWQgc2luY2UgZGVwbG95bWVudCB0aW1lXG4gICAgICogQHBhcmFtIHQgdGhlIGR1cmF0aW9uIHRvIHdhaXQgYmVmb3JlIGV4cGlyaW5nXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBhZnRlcih0OiBEdXJhdGlvbikgeyByZXR1cm4gRXhwaXJhdGlvbi5hdERhdGUobmV3IERhdGUoRGF0ZS5ub3coKSArIHQudG9NaWxsaXNlY29uZHMoKSkpOyB9XG4gICAgLyoqXG4gICAgICogRXhwaXJlIGF0IHNwZWNpZmllZCBkYXRlLCByZXByZXNlbnRlZCBhcyBhIHN0cmluZ1xuICAgICAqXG4gICAgICogQHBhcmFtIHMgdGhlIHN0cmluZyB0aGF0IHJlcHJlc2VudHMgZGF0ZSB0byBleHBpcmUgYXRcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcoczogc3RyaW5nKSB7IHJldHVybiBuZXcgRXhwaXJhdGlvbihuZXcgRGF0ZShzKSk7IH1cbiAgICAvKipcbiAgICAgKiBFeHBpcmF0aW9uIHZhbHVlIGFzIGEgRGF0ZSBvYmplY3RcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZGF0ZTogRGF0ZTtcbiAgICBwcml2YXRlIGNvbnN0cnVjdG9yKGRhdGU6IERhdGUpIHtcbiAgICAgICAgdGhpcy5kYXRlID0gZGF0ZTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogRXhpcHJhdGlvbiBWYWx1ZSBpbiBhIGZvcm1hdHRlZCBVbml4IEVwb2NoIFRpbWUgaW4gc2Vjb25kc1xuICAgICAqL1xuICAgIHB1YmxpYyB0b0Vwb2NoKCk6IG51bWJlciB7XG4gICAgICAgIHJldHVybiBNYXRoLnJvdW5kKHRoaXMuZGF0ZS5nZXRUaW1lKCkgLyAxMDAwKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ2hlY2sgaWYgRXhpcGlyYXRpb24gZXhwaXJlcyBiZWZvcmUgaW5wdXRcbiAgICAgKiBAcGFyYW0gdCB0aGUgZHVyYXRpb24gdG8gY2hlY2sgYWdhaW5zdFxuICAgICAqL1xuICAgIHB1YmxpYyBpc0JlZm9yZSh0OiBEdXJhdGlvbik6IGJvb2xlYW4ge1xuICAgICAgICByZXR1cm4gdGhpcy5kYXRlIDwgbmV3IERhdGUoRGF0ZS5ub3coKSArIHQudG9NaWxsaXNlY29uZHMoKSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIENoZWNrIGlmIEV4aXBpcmF0aW9uIGV4cGlyZXMgYWZ0ZXIgaW5wdXRcbiAgICAgKiBAcGFyYW0gdCB0aGUgZHVyYXRpb24gdG8gY2hlY2sgYWdhaW5zdFxuICAgICAqL1xuICAgIHB1YmxpYyBpc0FmdGVyKHQ6IER1cmF0aW9uKTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiB0aGlzLmRhdGUgPiBuZXcgRGF0ZShEYXRlLm5vdygpICsgdC50b01pbGxpc2Vjb25kcygpKTtcbiAgICB9XG59XG4iXX0=