"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParameter = void 0;
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._type = props.type || 'String';
        this._default = props.default;
        this._allowedPattern = props.allowedPattern;
        this._allowedValues = props.allowedValues;
        this._constraintDescription = props.constraintDescription;
        this._description = props.description;
        this._maxLength = props.maxLength;
        this._maxValue = props.maxValue;
        this._minLength = props.minLength;
        this._minValue = props.minValue;
        this._noEcho = props.noEcho;
    }
    /**
     * The data type for the parameter (DataType).
     *
     * @default String
     */
    get type() {
        return this._type;
    }
    set type(type) {
        this._type = type;
    }
    /**
     * A value of the appropriate type for the template to use if no value is specified
     * when a stack is created. If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     */
    get default() {
        return this._default;
    }
    set default(value) {
        this._default = value;
    }
    /**
     * A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     */
    get allowedPattern() {
        return this._allowedPattern;
    }
    set allowedPattern(pattern) {
        this._allowedPattern = pattern;
    }
    /**
     * An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     */
    get allowedValues() {
        return this._allowedValues;
    }
    set allowedValues(values) {
        this._allowedValues = values;
    }
    /**
     * A string that explains a constraint when the constraint is violated.
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     */
    get constraintDescription() {
        return this._constraintDescription;
    }
    set constraintDescription(desc) {
        this._constraintDescription = desc;
    }
    /**
     * A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     */
    get description() {
        return this._description;
    }
    set description(desc) {
        this._description = desc;
    }
    /**
     * An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get maxLength() {
        return this._maxLength;
    }
    set maxLength(len) {
        this._maxLength = len;
    }
    /**
     * An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get minLength() {
        return this._minLength;
    }
    set minLength(len) {
        this._minLength = len;
    }
    /**
     * A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get maxValue() {
        return this._maxValue;
    }
    set maxValue(len) {
        this._maxValue = len;
    }
    /**
     * A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get minValue() {
        return this._minValue;
    }
    set minValue(len) {
        this._minValue = len;
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this._noEcho;
    }
    set noEcho(echo) {
        this._noEcho = echo;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type) && !isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string or number type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.default,
                    AllowedPattern: this.allowedPattern,
                    AllowedValues: this.allowedValues,
                    ConstraintDescription: this.constraintDescription,
                    Description: this.description,
                    MaxLength: this.maxLength,
                    MaxValue: this.maxValue,
                    MinLength: this.minLength,
                    MinValue: this.minValue,
                    NoEcho: this._noEcho,
                },
            },
        };
    }
    resolve(_context) {
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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