"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const tree_metadata_1 = require("./private/tree-metadata");
const stage_1 = require("./stage");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends stage_1.Stage {
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        var _a, _b;
        super(undefined, '', {
            outdir: (_a = props.outdir) !== null && _a !== void 0 ? _a : process.env[cxapi.OUTDIR_ENV],
        });
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        const analyticsReporting = (_b = props.analyticsReporting) !== null && _b !== void 0 ? _b : props.runtimeInfo;
        if (analyticsReporting !== undefined) {
            this.node.setContext(cxapi.ANALYTICS_REPORTING_ENABLED_CONTEXT, analyticsReporting);
        }
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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