"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wait = exports.WaitTime = void 0;
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Represents the Wait state which delays a state machine from continuing for a specified time
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-wait-state.html
 */
class WaitTime {
    constructor(json) {
        this.json = json;
    }
    /**
     * Wait a fixed amount of time.
     */
    static duration(duration) { return new WaitTime({ Seconds: duration.toSeconds() }); }
    /**
     * Wait until the given ISO8601 timestamp
     *
     * @example 2016-03-14T01:59:00Z
     */
    static timestamp(timestamp) { return new WaitTime({ Timestamp: timestamp }); }
    /**
     * Wait for a number of seconds stored in the state object.
     *
     * @example $.waitSeconds
     */
    static secondsPath(path) { return new WaitTime({ SecondsPath: path }); }
    /**
     * Wait until a timestamp found in the state object.
     *
     * @example $.waitTimestamp
     */
    static timestampPath(path) { return new WaitTime({ TimestampPath: path }); }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.WaitTime = WaitTime;
/**
 * Define a Wait state in the state machine
 *
 * A Wait state can be used to delay execution of the state machine for a while.
 */
class Wait extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.time = props.time;
        this.endStates = [this];
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.WAIT,
            Comment: this.comment,
            ...this.time._json,
            ...this.renderNextEnd(),
        };
    }
}
exports.Wait = Wait;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2FpdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIndhaXQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBRUEsb0NBQWlDO0FBRWpDLHFEQUFpRDtBQUNqRCxtQ0FBZ0M7QUFDaEM7OztHQUdHO0FBQ0gsTUFBYSxRQUFRO0lBdUJqQixZQUFxQyxJQUFTO1FBQVQsU0FBSSxHQUFKLElBQUksQ0FBSztJQUFJLENBQUM7SUF0Qm5EOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFzQixJQUFJLE9BQU8sSUFBSSxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsUUFBUSxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDMUc7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsU0FBaUIsSUFBSSxPQUFPLElBQUksUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzdGOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUFDLElBQVksSUFBSSxPQUFPLElBQUksUUFBUSxDQUFDLEVBQUUsV0FBVyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3ZGOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQVksSUFBSSxPQUFPLElBQUksUUFBUSxDQUFDLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRTNGOztPQUVHO0lBQ0gsSUFBVyxLQUFLO1FBQ1osT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDO0lBQ3JCLENBQUM7Q0FDSjtBQTlCRCw0QkE4QkM7QUFnQkQ7Ozs7R0FJRztBQUNILE1BQWEsSUFBSyxTQUFRLGFBQUs7SUFHM0IsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFnQjtRQUN0RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QixJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDdkIsSUFBSSxDQUFDLFNBQVMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFDRDs7T0FFRztJQUNJLElBQUksQ0FBQyxJQUFnQjtRQUN4QixLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNoQyxPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFDRDs7T0FFRztJQUNJLFdBQVc7UUFDZCxPQUFPO1lBQ0gsSUFBSSxFQUFFLHNCQUFTLENBQUMsSUFBSTtZQUNwQixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUs7WUFDbEIsR0FBRyxJQUFJLENBQUMsYUFBYSxFQUFFO1NBQzFCLENBQUM7SUFDTixDQUFDO0NBQ0o7QUExQkQsb0JBMEJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDaGFpbiB9IGZyb20gJy4uL2NoYWluJztcbmltcG9ydCB7IElDaGFpbmFibGUsIElOZXh0YWJsZSB9IGZyb20gJy4uL3R5cGVzJztcbmltcG9ydCB7IFN0YXRlVHlwZSB9IGZyb20gJy4vcHJpdmF0ZS9zdGF0ZS10eXBlJztcbmltcG9ydCB7IFN0YXRlIH0gZnJvbSAnLi9zdGF0ZSc7XG4vKipcbiAqIFJlcHJlc2VudHMgdGhlIFdhaXQgc3RhdGUgd2hpY2ggZGVsYXlzIGEgc3RhdGUgbWFjaGluZSBmcm9tIGNvbnRpbnVpbmcgZm9yIGEgc3BlY2lmaWVkIHRpbWVcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9hbWF6b24tc3RhdGVzLWxhbmd1YWdlLXdhaXQtc3RhdGUuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgV2FpdFRpbWUge1xuICAgIC8qKlxuICAgICAqIFdhaXQgYSBmaXhlZCBhbW91bnQgb2YgdGltZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGR1cmF0aW9uKGR1cmF0aW9uOiBjZGsuRHVyYXRpb24pIHsgcmV0dXJuIG5ldyBXYWl0VGltZSh7IFNlY29uZHM6IGR1cmF0aW9uLnRvU2Vjb25kcygpIH0pOyB9XG4gICAgLyoqXG4gICAgICogV2FpdCB1bnRpbCB0aGUgZ2l2ZW4gSVNPODYwMSB0aW1lc3RhbXBcbiAgICAgKlxuICAgICAqIEBleGFtcGxlIDIwMTYtMDMtMTRUMDE6NTk6MDBaXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyB0aW1lc3RhbXAodGltZXN0YW1wOiBzdHJpbmcpIHsgcmV0dXJuIG5ldyBXYWl0VGltZSh7IFRpbWVzdGFtcDogdGltZXN0YW1wIH0pOyB9XG4gICAgLyoqXG4gICAgICogV2FpdCBmb3IgYSBudW1iZXIgb2Ygc2Vjb25kcyBzdG9yZWQgaW4gdGhlIHN0YXRlIG9iamVjdC5cbiAgICAgKlxuICAgICAqIEBleGFtcGxlICQud2FpdFNlY29uZHNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHNlY29uZHNQYXRoKHBhdGg6IHN0cmluZykgeyByZXR1cm4gbmV3IFdhaXRUaW1lKHsgU2Vjb25kc1BhdGg6IHBhdGggfSk7IH1cbiAgICAvKipcbiAgICAgKiBXYWl0IHVudGlsIGEgdGltZXN0YW1wIGZvdW5kIGluIHRoZSBzdGF0ZSBvYmplY3QuXG4gICAgICpcbiAgICAgKiBAZXhhbXBsZSAkLndhaXRUaW1lc3RhbXBcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHRpbWVzdGFtcFBhdGgocGF0aDogc3RyaW5nKSB7IHJldHVybiBuZXcgV2FpdFRpbWUoeyBUaW1lc3RhbXBQYXRoOiBwYXRoIH0pOyB9XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGpzb246IGFueSkgeyB9XG4gICAgLyoqXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHVibGljIGdldCBfanNvbigpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuanNvbjtcbiAgICB9XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIGEgV2FpdCBzdGF0ZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFdhaXRQcm9wcyB7XG4gICAgLyoqXG4gICAgICogQW4gb3B0aW9uYWwgZGVzY3JpcHRpb24gZm9yIHRoaXMgc3RhdGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGNvbW1lbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFdhaXQgZHVyYXRpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGltZTogV2FpdFRpbWU7XG59XG4vKipcbiAqIERlZmluZSBhIFdhaXQgc3RhdGUgaW4gdGhlIHN0YXRlIG1hY2hpbmVcbiAqXG4gKiBBIFdhaXQgc3RhdGUgY2FuIGJlIHVzZWQgdG8gZGVsYXkgZXhlY3V0aW9uIG9mIHRoZSBzdGF0ZSBtYWNoaW5lIGZvciBhIHdoaWxlLlxuICovXG5leHBvcnQgY2xhc3MgV2FpdCBleHRlbmRzIFN0YXRlIGltcGxlbWVudHMgSU5leHRhYmxlIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgZW5kU3RhdGVzOiBJTmV4dGFibGVbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHRpbWU6IFdhaXRUaW1lO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBXYWl0UHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gICAgICAgIHRoaXMudGltZSA9IHByb3BzLnRpbWU7XG4gICAgICAgIHRoaXMuZW5kU3RhdGVzID0gW3RoaXNdO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDb250aW51ZSBub3JtYWwgZXhlY3V0aW9uIHdpdGggdGhlIGdpdmVuIHN0YXRlXG4gICAgICovXG4gICAgcHVibGljIG5leHQobmV4dDogSUNoYWluYWJsZSk6IENoYWluIHtcbiAgICAgICAgc3VwZXIubWFrZU5leHQobmV4dC5zdGFydFN0YXRlKTtcbiAgICAgICAgcmV0dXJuIENoYWluLnNlcXVlbmNlKHRoaXMsIG5leHQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm4gdGhlIEFtYXpvbiBTdGF0ZXMgTGFuZ3VhZ2Ugb2JqZWN0IGZvciB0aGlzIHN0YXRlXG4gICAgICovXG4gICAgcHVibGljIHRvU3RhdGVKc29uKCk6IG9iamVjdCB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBUeXBlOiBTdGF0ZVR5cGUuV0FJVCxcbiAgICAgICAgICAgIENvbW1lbnQ6IHRoaXMuY29tbWVudCxcbiAgICAgICAgICAgIC4uLnRoaXMudGltZS5fanNvbixcbiAgICAgICAgICAgIC4uLnRoaXMucmVuZGVyTmV4dEVuZCgpLFxuICAgICAgICB9O1xuICAgIH1cbn1cbiJdfQ==