"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const fields_1 = require("../fields");
const types_1 = require("../types");
/**
 * Base class for all other state classes
 */
class State extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...constructs_1.Node.of(el).children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        validateErrors(props.errors);
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        validateErrors(props.errors);
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry, (a, b) => compareErrors(a.errors, b.errors)),
            Catch: renderList(this.catches, renderCatch, (a, b) => compareErrors(a.props.errors, b.props.errors)),
        };
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Compares a list of Errors to move Errors.ALL last in a sort function
 */
function compareErrors(a, b) {
    if (a === null || a === void 0 ? void 0 : a.includes(types_1.Errors.ALL)) {
        return 1;
    }
    if (b === null || b === void 0 ? void 0 : b.includes(types_1.Errors.ALL)) {
        return -1;
    }
    return 0;
}
/**
 * Validates an errors list
 */
function validateErrors(errors) {
    if ((errors === null || errors === void 0 ? void 0 : errors.includes(types_1.Errors.ALL)) && errors.length > 1) {
        throw new Error(`${types_1.Errors.ALL} must appear alone in an error list`);
    }
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, mapFn, sortFn) {
    if (xs.length === 0) {
        return undefined;
    }
    let list = xs;
    if (sortFn) {
        list = xs.sort(sortFn);
    }
    return list.map(mapFn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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