"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.AcceleratorType = exports.AcceleratorClass = exports.Mode = exports.ContainerDefinition = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const aws_ecr_assets_1 = require("../../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 * @experimental
 */
class ContainerDefinition {
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task
     */
    bind(task) {
        var _a, _b, _c;
        return {
            parameters: {
                ContainerHostname: this.options.containerHostName,
                Image: (_a = this.options.image) === null || _a === void 0 ? void 0 : _a.bind(task).imageUri,
                Mode: this.options.mode,
                ModelDataUrl: (_b = this.options.modelS3Location) === null || _b === void 0 ? void 0 : _b.bind(task, { forReading: true }).uri,
                ModelPackageName: this.options.modelPackageName,
                Environment: (_c = this.options.environmentVariables) === null || _c === void 0 ? void 0 : _c.value,
            },
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
/**
 * Specifies how many models the container hosts
 *
 * @experimental
 */
var Mode;
(function (Mode) {
    /**
     * Container hosts a single model
     */
    Mode["SINGLE_MODEL"] = "SingleModel";
    /**
     * Container hosts multiple models
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
     */
    Mode["MULTI_MODEL"] = "MultiModel";
})(Mode = exports.Mode || (exports.Mode = {}));
/**
 * The generation of Elastic Inference (EI) instance
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @experimental
 */
class AcceleratorClass {
    /**
     * @param version - Elastic Inference accelerator generation
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom AcceleratorType
     * @param version - Elastic Inference accelerator generation
    */
    static of(version) { return new AcceleratorClass(version); }
}
exports.AcceleratorClass = AcceleratorClass;
/**
 * Elastic Inference accelerator 1st generation
 */
AcceleratorClass.EIA1 = AcceleratorClass.of('eia1');
/**
 * Elastic Inference accelerator 2nd generation
 */
AcceleratorClass.EIA2 = AcceleratorClass.of('eia2');
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @experimental
 */
class AcceleratorType {
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * AcceleratorType
     *
     * This class takes a combination of a class and size.
     */
    static of(acceleratorClass, instanceSize) {
        return new AcceleratorType(`ml.${acceleratorClass}.${instanceSize}`);
    }
    /**
     * Return the accelerator type as a dotted string
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
}
exports.AcceleratorType = AcceleratorType;
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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