"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunGlueJobTask = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Glue job as a Task
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 *
 * @deprecated use `GlueStartJobRun`
 */
class RunGlueJobTask {
    constructor(glueJobName, props = {}) {
        this.glueJobName = glueJobName;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Glue.`);
        }
    }
    bind(task) {
        var _a;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        let iamActions;
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.FIRE_AND_FORGET) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun',
            ];
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('glue', 'startJobRun', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [
                        core_1.Stack.of(task).formatArn({
                            service: 'glue',
                            resource: 'job',
                            resourceName: this.glueJobName,
                        }),
                    ],
                    actions: iamActions,
                })],
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.glueJobName },
            parameters: {
                JobName: this.glueJobName,
                Arguments: this.props.arguments,
                Timeout: (_a = this.props.timeout) === null || _a === void 0 ? void 0 : _a.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty,
            },
        };
    }
}
exports.RunGlueJobTask = RunGlueJobTask;
//# sourceMappingURL=data:application/json;base64,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