"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IpInstance = void 0;
const instance_1 = require("./instance");
const service_1 = require("./service");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
/**
 * Instance that is accessible using an IP address.
 *
 * @resource AWS::ServiceDiscovery::Instance
 */
class IpInstance extends instance_1.InstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const dnsRecordType = props.service.dnsRecordType;
        if (dnsRecordType === service_1.DnsRecordType.CNAME) {
            throw new Error('Service must support `A`, `AAAA` or `SRV` records to register this instance type.');
        }
        if (dnsRecordType === service_1.DnsRecordType.SRV) {
            if (!props.port) {
                throw new Error('A `port` must be specified for a service using a `SRV` record.');
            }
            if (!props.ipv4 && !props.ipv6) {
                throw new Error('At least `ipv4` or `ipv6` must be specified for a service using a `SRV` record.');
            }
        }
        if (!props.ipv4 && (dnsRecordType === service_1.DnsRecordType.A || dnsRecordType === service_1.DnsRecordType.A_AAAA)) {
            throw new Error('An `ipv4` must be specified for a service using a `A` record.');
        }
        if (!props.ipv6 &&
            (dnsRecordType === service_1.DnsRecordType.AAAA || dnsRecordType === service_1.DnsRecordType.A_AAAA)) {
            throw new Error('An `ipv6` must be specified for a service using a `AAAA` record.');
        }
        const port = props.port || 80;
        const resource = new servicediscovery_generated_1.CfnInstance(this, 'Resource', {
            instanceAttributes: {
                AWS_INSTANCE_IPV4: props.ipv4,
                AWS_INSTANCE_IPV6: props.ipv6,
                AWS_INSTANCE_PORT: port.toString(),
                ...props.customAttributes,
            },
            instanceId: props.instanceId || this.uniqueInstanceId(),
            serviceId: props.service.serviceId,
        });
        this.service = props.service;
        this.instanceId = resource.ref;
        this.ipv4 = props.ipv4 || '';
        this.ipv6 = props.ipv6 || '';
        this.port = port;
    }
}
exports.IpInstance = IpInstance;
//# sourceMappingURL=data:application/json;base64,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