"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies. Typically, this is just the secret ARN.
     * However, secrets imported by name require a different format.
     */
    get arnForPolicies() { return this.secretArn; }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        this.secretName = this.physicalName;
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principal);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Imports a secret by secret name; the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     */
    static fromSecretName(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    sep: ':',
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
                this.secretName = parseSecretName(scope, attrs.secretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBProxy
     */
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        this.secretName = props.secret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).parseArn(secretArn, ':').resourceName;
    if (resourceName) {
        // Can't operate on the token to remove the SecretsManager suffix, so just return the full secret name
        if (core_1.Token.isUnresolved(resourceName)) {
            return resourceName;
        }
        // Secret resource names are in the format `${secretName}-${SecretsManager suffix}`
        // If there is no hyphen, assume no suffix was provided, and return the whole name.
        return resourceName.substr(0, resourceName.lastIndexOf('-')) || resourceName;
    }
    throw new Error('invalid ARN format; no secret name provided');
}
//# sourceMappingURL=data:application/json;base64,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