"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretRotation = exports.SecretRotationApplication = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const serverless = require("../../aws-sam"); // Automatically re-written from '@aws-cdk/aws-sam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        // Max length of 64 chars, get the last 64 chars
        const uniqueId = this.node.uniqueId;
        const rotationFunctionName = uniqueId.substring(Math.max(uniqueId.length - 64, 0), uniqueId.length);
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.excludeCharacters !== undefined) {
            parameters.excludeCharacters = props.excludeCharacters;
        }
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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