"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const handlerCodeBundle = path.join(__dirname, '..', 'lambda', 'bundle.zip');
const handlerSourceDirectory = path.join(__dirname, '..', 'lambda', 'src');
class BucketDeployment extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.distributionPaths && !props.distribution) {
            throw new Error('Distribution must be specified if distribution paths are specified');
        }
        const assetHash = calcSourceHash(handlerSourceDirectory);
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit),
            code: lambda.Code.fromAsset(handlerCodeBundle, { assetHash }),
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_a = props.prune) !== null && _a !== void 0 ? _a : true,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution ? props.distribution.distributionId : undefined,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.BucketDeployment = BucketDeployment;
/**
 * We need a custom source hash calculation since the bundle.zip file
 * contains python dependencies installed during build and results in a
 * non-deterministic behavior.
 *
 * So we just take the `src/` directory of our custom resoruce code.
 */
function calcSourceHash(srcDir) {
    const sha = crypto.createHash('sha256');
    for (const file of fs.readdirSync(srcDir)) {
        const data = fs.readFileSync(path.join(srcDir, file));
        sha.update(`<file name=${file}>`);
        sha.update(data);
        sha.update('</file>');
    }
    return sha.digest('hex');
}
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase().startsWith('x-amzn-meta-')
        ? key.toLowerCase()
        : `x-amzn-meta-${key.toLowerCase()}`;
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(value) {
        this.value = value;
    }
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    static noCache() { return new CacheControl('no-cache'); }
    static noTransform() { return new CacheControl('no-transform'); }
    static setPublic() { return new CacheControl('public'); }
    static setPrivate() { return new CacheControl('private'); }
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
//# sourceMappingURL=data:application/json;base64,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