"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Asset = void 0;
const fs = require("fs");
const path = require("path");
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const compat_1 = require("./compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new cdk.AssetStaging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
            follow: compat_1.toSymlinkFollow(props.follow),
            assetHash: (_a = props.assetHash) !== null && _a !== void 0 ? _a : props.sourceHash,
        });
        this.assetHash = staging.assetHash;
        this.sourceHash = this.assetHash;
        this.assetPath = staging.stagedPath;
        const packaging = determinePackaging(staging.sourcePath);
        // sets isZipArchive based on the type of packaging and file extension
        this.isZipArchive = packaging === cdk.FileAssetPackaging.ZIP_DIRECTORY
            ? true
            : ARCHIVE_EXTENSIONS.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        const stack = cdk.Stack.of(this);
        const location = stack.synthesizer.addFileAsset({
            packaging,
            sourceHash: this.sourceHash,
            fileName: staging.stagedPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        const kmsKey = location.kmsKeyArn ? kms.Key.fromKeyArn(this, 'Key', location.kmsKeyArn) : undefined;
        this.bucket = s3.Bucket.fromBucketAttributes(this, 'AssetBucket', {
            bucketName: this.s3BucketName,
            encryptionKey: kmsKey,
        });
        for (const reader of ((_b = props.readers) !== null && _b !== void 0 ? _b : [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the assets bucket.
     */
    grantRead(grantee) {
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
exports.Asset = Asset;
function determinePackaging(assetPath) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    if (fs.statSync(assetPath).isDirectory()) {
        return cdk.FileAssetPackaging.ZIP_DIRECTORY;
    }
    if (fs.statSync(assetPath).isFile()) {
        return cdk.FileAssetPackaging.FILE;
    }
    throw new Error(`Asset ${assetPath} is expected to be either a directory or a regular file`);
}
//# sourceMappingURL=data:application/json;base64,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