"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PerformanceInsightRetention = exports.SnapshotCredentials = exports.Credentials = void 0;
/**
 * Username and password combination
 */
class Credentials {
    /**
     * Creates Credentials for the given username, and optional password and key.
     * If no password is provided, one will be generated and stored in SecretsManager.
     */
    static fromUsername(username, options = {}) {
        return {
            ...options,
            username,
        };
    }
    /**
     * Creates Credentials from an existing SecretsManager ``Secret`` (or ``DatabaseSecret``)
     *
     * The Secret must be a JSON string with a ``username`` and ``password`` field:
     * ```
     * {
     *   ...
     *   "username": <required: username>,
     *   "password": <required: password>,
     * }
     * ```
     */
    static fromSecret(secret) {
        return {
            username: secret.secretValueFromJson('username').toString(),
            password: secret.secretValueFromJson('password'),
            encryptionKey: secret.encryptionKey,
            secret,
        };
    }
}
exports.Credentials = Credentials;
/**
 * Credentials to update the password for a ``DatabaseInstanceFromSnapshot``.
 */
class SnapshotCredentials {
    /**
     * Generate a new password for the snapshot, using the existing username and an optional encryption key.
     *
     * Note - The username must match the existing master username of the snapshot.
     */
    static fromGeneratedPassword(username, options = {}) {
        return {
            ...options,
            generatePassword: true,
            username,
        };
    }
    /**
     * Update the snapshot login with an existing password.
     */
    static fromPassword(password) {
        return { generatePassword: false, password };
    }
    /**
     * Update the snapshot login with an existing password from a Secret.
     *
     * The Secret must be a JSON string with a ``password`` field:
     * ```
     * {
     *   ...
     *   "password": <required: password>,
     * }
     * ```
     */
    static fromSecret(secret) {
        return {
            generatePassword: false,
            password: secret.secretValueFromJson('password'),
            secret,
        };
    }
}
exports.SnapshotCredentials = SnapshotCredentials;
/**
 * The retention period for Performance Insight.
 */
var PerformanceInsightRetention;
(function (PerformanceInsightRetention) {
    /**
     * Default retention period of 7 days.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["DEFAULT"] = 7] = "DEFAULT";
    /**
     * Long term retention period of 2 years.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["LONG_TERM"] = 731] = "LONG_TERM";
})(PerformanceInsightRetention = exports.PerformanceInsightRetention || (exports.PerformanceInsightRetention = {}));
//# sourceMappingURL=data:application/json;base64,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