"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogGroup = exports.RetentionDays = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const log_stream_1 = require("./log-stream");
const logs_generated_1 = require("./logs.generated");
const metric_filter_1 = require("./metric-filter");
const pattern_1 = require("./pattern");
const subscription_filter_1 = require("./subscription-filter");
/**
 * An CloudWatch Log Group
 */
class LogGroupBase extends core_1.Resource {
    /**
     * Create a new Log Stream for this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the LogStream
     */
    addStream(id, props = {}) {
        return new log_stream_1.LogStream(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Create a new Subscription Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the SubscriptionFilter
     */
    addSubscriptionFilter(id, props) {
        return new subscription_filter_1.SubscriptionFilter(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Create a new Metric Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the MetricFilter
     */
    addMetricFilter(id, props) {
        return new metric_filter_1.MetricFilter(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Extract a metric from structured log events in the LogGroup
     *
     * Creates a MetricFilter on this LogGroup that will extract the value
     * of the indicated JSON field in all records where it occurs.
     *
     * The metric will be available in CloudWatch Metrics under the
     * indicated namespace and name.
     *
     * @param jsonField JSON field to extract (example: '$.myfield')
     * @param metricNamespace Namespace to emit the metric under
     * @param metricName Name to emit the metric under
     * @returns A Metric object representing the extracted metric
     */
    extractMetric(jsonField, metricNamespace, metricName) {
        new metric_filter_1.MetricFilter(this, `${metricNamespace}_${metricName}`, {
            logGroup: this,
            metricNamespace,
            metricName,
            filterPattern: pattern_1.FilterPattern.exists(jsonField),
            metricValue: jsonField,
        });
        return new cloudwatch.Metric({ metricName, namespace: metricNamespace }).attachTo(this);
    }
    /**
     * Give permissions to write to create and write to streams in this log group
     */
    grantWrite(grantee) {
        return this.grant(grantee, 'logs:CreateLogStream', 'logs:PutLogEvents');
    }
    /**
     * Give the indicated permissions on this log group and all streams
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            // A LogGroup ARN out of CloudFormation already includes a ':*' at the end to include the log streams under the group.
            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#w2ab1c21c10c63c43c11
            resourceArns: [this.logGroupArn],
            scope: this,
        });
    }
}
/**
 * How long, in days, the log contents will be retained.
 */
var RetentionDays;
(function (RetentionDays) {
    /**
     * 1 day
     */
    RetentionDays[RetentionDays["ONE_DAY"] = 1] = "ONE_DAY";
    /**
     * 3 days
     */
    RetentionDays[RetentionDays["THREE_DAYS"] = 3] = "THREE_DAYS";
    /**
     * 5 days
     */
    RetentionDays[RetentionDays["FIVE_DAYS"] = 5] = "FIVE_DAYS";
    /**
     * 1 week
     */
    RetentionDays[RetentionDays["ONE_WEEK"] = 7] = "ONE_WEEK";
    /**
     * 2 weeks
     */
    RetentionDays[RetentionDays["TWO_WEEKS"] = 14] = "TWO_WEEKS";
    /**
     * 1 month
     */
    RetentionDays[RetentionDays["ONE_MONTH"] = 30] = "ONE_MONTH";
    /**
     * 2 months
     */
    RetentionDays[RetentionDays["TWO_MONTHS"] = 60] = "TWO_MONTHS";
    /**
     * 3 months
     */
    RetentionDays[RetentionDays["THREE_MONTHS"] = 90] = "THREE_MONTHS";
    /**
     * 4 months
     */
    RetentionDays[RetentionDays["FOUR_MONTHS"] = 120] = "FOUR_MONTHS";
    /**
     * 5 months
     */
    RetentionDays[RetentionDays["FIVE_MONTHS"] = 150] = "FIVE_MONTHS";
    /**
     * 6 months
     */
    RetentionDays[RetentionDays["SIX_MONTHS"] = 180] = "SIX_MONTHS";
    /**
     * 1 year
     */
    RetentionDays[RetentionDays["ONE_YEAR"] = 365] = "ONE_YEAR";
    /**
     * 13 months
     */
    RetentionDays[RetentionDays["THIRTEEN_MONTHS"] = 400] = "THIRTEEN_MONTHS";
    /**
     * 18 months
     */
    RetentionDays[RetentionDays["EIGHTEEN_MONTHS"] = 545] = "EIGHTEEN_MONTHS";
    /**
     * 2 years
     */
    RetentionDays[RetentionDays["TWO_YEARS"] = 731] = "TWO_YEARS";
    /**
     * 5 years
     */
    RetentionDays[RetentionDays["FIVE_YEARS"] = 1827] = "FIVE_YEARS";
    /**
     * 10 years
     */
    RetentionDays[RetentionDays["TEN_YEARS"] = 3653] = "TEN_YEARS";
    /**
     * Retain logs forever
     */
    RetentionDays[RetentionDays["INFINITE"] = 9999] = "INFINITE";
})(RetentionDays = exports.RetentionDays || (exports.RetentionDays = {}));
/**
 * Define a CloudWatch Log Group
 */
class LogGroup extends LogGroupBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.logGroupName,
        });
        let retentionInDays = props.retention;
        if (retentionInDays === undefined) {
            retentionInDays = RetentionDays.TWO_YEARS;
        }
        if (retentionInDays === Infinity || retentionInDays === RetentionDays.INFINITE) {
            retentionInDays = undefined;
        }
        if (retentionInDays !== undefined && !core_1.Token.isUnresolved(retentionInDays) && retentionInDays <= 0) {
            throw new Error(`retentionInDays must be positive, got ${retentionInDays}`);
        }
        const resource = new logs_generated_1.CfnLogGroup(this, 'Resource', {
            logGroupName: this.physicalName,
            retentionInDays,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.logGroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'logs',
            resource: 'log-group',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.logGroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import an existing LogGroup given its ARN
     */
    static fromLogGroupArn(scope, id, logGroupArn) {
        const baseLogGroupArn = logGroupArn.replace(/:\*$/, '');
        class Import extends LogGroupBase {
            constructor() {
                super(...arguments);
                this.logGroupArn = `${baseLogGroupArn}:*`;
                this.logGroupName = core_1.Stack.of(scope).parseArn(baseLogGroupArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing LogGroup given its name
     */
    static fromLogGroupName(scope, id, logGroupName) {
        const baseLogGroupName = logGroupName.replace(/:\*$/, '');
        class Import extends LogGroupBase {
            constructor() {
                super(...arguments);
                this.logGroupName = baseLogGroupName;
                this.logGroupArn = core_1.Stack.of(scope).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    sep: ':',
                    resourceName: baseLogGroupName + ':*',
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.LogGroup = LogGroup;
//# sourceMappingURL=data:application/json;base64,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