"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossAccountDestination = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.stringValue({
            produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,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