"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SingletonFunction = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * singleton Lambda function. A new version will be created every time the
     * function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.SingletonFunction`.
     */
    get currentVersion() {
        return this.lambdaFunction.currentVersion;
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /** @internal */
    _checkEdgeCompatibility() {
        return this.lambdaFunction._checkEdgeCompatibility();
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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