"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartingPosition = exports.EventSourceMapping = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *    import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *    lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        if (props.maxRecordAge && (props.maxRecordAge.toSeconds() < 60 || props.maxRecordAge.toDays({ integral: false }) > 7)) {
            throw new Error('maxRecordAge must be between 60 seconds and 7 days inclusive');
        }
        props.retryAttempts !== undefined && cdk.withResolved(props.retryAttempts, (attempts) => {
            if (attempts < 0 || attempts > 10000) {
                throw new Error(`retryAttempts must be between 0 and 10000 inclusive, got ${attempts}`);
            }
        });
        props.parallelizationFactor !== undefined && cdk.withResolved(props.parallelizationFactor, (factor) => {
            if (factor < 1 || factor > 10) {
                throw new Error(`parallelizationFactor must be between 1 and 10 inclusive, got ${factor}`);
            }
        });
        let destinationConfig;
        if (props.onFailure) {
            destinationConfig = {
                onFailure: props.onFailure.bind(this, props.target),
            };
        }
        const cfnEventSourceMapping = new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            bisectBatchOnFunctionError: props.bisectBatchOnError,
            destinationConfig,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: (_a = props.maxBatchingWindow) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            maximumRecordAgeInSeconds: (_b = props.maxRecordAge) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            maximumRetryAttempts: props.retryAttempts,
            parallelizationFactor: props.parallelizationFactor,
            topics: props.kafkaTopic !== undefined ? [props.kafkaTopic] : undefined,
        });
        this.eventSourceMappingId = cfnEventSourceMapping.ref;
    }
    /**
     * Import an event source into this stack from its event source id.
     */
    static fromEventSourceMappingId(scope, id, eventSourceMappingId) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.eventSourceMappingId = eventSourceMappingId;
            }
        }
        return new Import(scope, id);
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start
 * reading.
 */
var StartingPosition;
(function (StartingPosition) {
    /**
     * Start reading at the last untrimmed record in the shard in the system,
     * which is the oldest data record in the shard.
     */
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    /**
     * Start reading just after the most recent record in the shard, so that you
     * always read the most recent data in the shard
     */
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,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